/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.cmd;

import java.util.ArrayList;
import java.util.LinkedList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.lang.Logging;
import org.unitarou.lang.NameDisplayable;
import org.unitarou.lang.Strings;
import org.unitarou.lang.Logging.Level;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.view.monitor.CommandDriverMonitor;

/**
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public class CommandDriver {
	static private final Log log_s_ = LogFactory.getLog(CommandDriver.class);
	
	/**
	 * {@link org.unitarou.cmd.CommandDriver#execute(Command)}A
	 * {@link org.unitarou.cmd.CommandDriver#undo()}A
	 * {@link org.unitarou.cmd.CommandDriver#redo()}ɕԂ
	 * {@link org.unitarou.cmd.CommandDriver}̏ԂԂ܂B
	 */
	public class Status {
		private Command executedCommand_;
		private Status() {
			super();
		}
		
		/**
		 * ŌɎs(execute, redo, undo)ꂽR}hԂ܂B
		 * @return R}hs̏ꍇnullԂ܂B
		 */
		public Command getExecutedCommand() {
			return executedCommand_;
		}
		
		/**
		 * RedoR}ĥ̖Ԃ܂B {@link #execute(Command)}̈{@link NameDisplayable}
		 * CX^Xo^ꍇ̂݁A{@link NameDisplayable#displayName()}Ԃ܂B
		 * ȊȌꍇRedołR}hꍇ{@link Strings#EMPTY}Ԃ܂B
		 */
		public String getRedoName() {
			if (isRedoable() && (undoneCommands_.getFirst() instanceof NameDisplayable)) {
				NameDisplayable command = (NameDisplayable) undoneCommands_.getFirst();
				return command.displayName();
			}
			return Strings.EMPTY;
		}
		
		/**
		 * UndoR}ĥ̖Ԃ܂B {@link #execute(Command)}̈{@link NameDisplayable}
		 * CX^Xo^ꍇ̂݁A{@link NameDisplayable#displayName()}Ԃ܂B
		 * ȊȌꍇRedołR}hꍇ{@link Strings#EMPTY}Ԃ܂B
		 */
		public String getUndoName() {
			if (isUndoable() && (executedCommands_.getLast() instanceof NameDisplayable)) {
				NameDisplayable command = (NameDisplayable) executedCommands_.getLast();
				return command.displayName();
			}
			return Strings.EMPTY;
		}
		
		/** {@link #undo()}\łtrueԂ܂B */
		public boolean isUndoable() {
			return !executedCommands_.isEmpty();
		}

		/** {@link #redo()}\łtrueԂ܂B */
		public boolean isRedoable() {
			return !undoneCommands_.isEmpty();
		}
	}
	
	
	/** sꂽR}hĂ܂B^[Command]łB */
	private final LinkedList<Command> executedCommands_;

	/** AhDꂽR}hĂ܂B^[Command]łB */
	private final LinkedList<Command> undoneCommands_;

	/** ^[{@link CommandDriverMonitor}]Ao^ĂSXi[łB */
	private final ArrayList<CommandDriverListener> listeners_;

	private Status status_;

	/**
	 * 
	 */
	public CommandDriver() {
		super();
		executedCommands_ = new LinkedList<Command>();
		undoneCommands_ = new LinkedList<Command>();
		listeners_ = new ArrayList<CommandDriverListener>();
		status_ = new Status();
	}

	/** undoAredoXgSď܂B */
	public void clear() {
		clearUndo();
		clearRedo();
	}
	
	/**
	 * undooXĝݏ܂B
	 */
	public void clearUndo() {
		executedCommands_.clear();
	}

	/**
	 * redoXĝݏ܂B
	 */
	public void clearRedo() {
		undoneCommands_.clear();
	}
	
	/**
	 * CX^X̏ԃIuWFNgԂ܂B
	 * @return
	 */
	public Status getStatus() {
		return status_;
	}

	/**
	 * commandsAundo\XgɒǉƋɁA redoXgɂ܂B<br>
	 * 
	 * @throws org.unitarou.lang.NullArgumentException
	 *         null̏ꍇB
	 */
	public Status execute(Command command) {
		ArgumentChecker.throwIfNull(command);

		command.execute();
		executedCommands_.addLast(command);
		undoneCommands_.clear();
		for (CommandDriverListener listener : listeners_) {
			listener.executed(command);
		}
		status_.executedCommand_ = command;
		return status_;
	}

	/**
	 * O{@link #undo()}R}hĎs܂B
	 * 
	 * @return 
	 */
	@Logging(level = Level.TRACE, contents = "redoꂽR}h") //$NON-NLS-1$
	public Status redo() {
		Command command = undoneCommands_.removeFirst();
		command.execute();
		
		if (log_s_.isTraceEnabled()) {
			log_s_.trace("REDONE: " + command); //$NON-NLS-1$
		}
		
		executedCommands_.addLast(command);
		for (CommandDriverListener listener : listeners_) {
			listener.redoExecuted(command);
		}
		status_.executedCommand_ = command;
		return status_;
	}


	/**
	 * Ŋ{@link #execute(Command)}邢{@link #redo()}R}hundo܂B
	 * 
	 * @return undoR}h
	 */
	@Logging(level = Level.TRACE, contents = "undoꂽR}h") //$NON-NLS-1$
	public Status undo() {
		Command command = executedCommands_.removeLast();
		command.undo();
		
		if (log_s_.isTraceEnabled()) {
			log_s_.trace("UNDONE: " + command); //$NON-NLS-1$
		}
		
		undoneCommands_.addFirst(command);
		for (CommandDriverListener listener : listeners_) {
			listener.undoExecuted(command);
		}
		status_.executedCommand_ = command;
		return status_;
	}

	/**
	 * listenero^܂Bnull̏ꍇ͖܂B
	 */
	public void addListener(CommandDriverListener listener) {
		if (listener == null) {
			return;
		}
		listeners_.add(listener);
	}

	/**
	 * listenero^܂B݂Ȃꍇnull̏ꍇ͉܂B
	 */
	public void removeListener(CommandDriverListener listener) {
		listeners_.remove(listener);
	}
}
