/*  
 * Copyright 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.io;

import java.io.File;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Queue;
import java.util.Set;
import java.util.StringTokenizer;

import org.apache.commons.io.FilenameUtils;

import org.unitarou.lang.NullArgumentException;
import org.unitarou.ml.MessageResource;
import org.unitarou.util.ArgumentChecker;


/**
 * pbP[Wjava.ioȂтjava.nio֘Ã\bh[eBeBłB
 * 
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
public class Ios {
	/**
	 * t@C̖{̂Ɗgq؂Zp[^(e.f)łB
	 */
	static public final char EXTENSION_SEPARATOR = '.';

	/**
	 * parentefBNg[Ƃ݂ȂāA΃pXŎw肳ꂽfilenamȇ΃pXԂ܂B<br>
	 * ϊłȂꍇnullԂ܂B
	 * 
	 * <pre>
	 * parent = C:\dir1, fileName = C:\dir1\file.name  file.name 
	 * parent = C:\dir1, fileName = C:\dir1\dir2\file.name  dir2\file.name 
	 * parent = C:\dir1\dir2, fileName = C:\dir1\file.name  ..\file.name
	 * parent = C:\dir1\dir2, fileName = C:\dir3\file.name  ..\..\dir3\file.name
	 * parent = C:\dir1\dir2, fileName = C:\dir3\file.name  ..\..\dir3\file.name
	 * parent = C:\dir1, fileName = D:\dirA\file.name  null
	 * parent = C:\dir1, fileName = file.name  null
	 * parent = \dir1\dir2, fileName = C:\dir3\file.name  null
	 * </pre>
	 * @param parent ƂȂfBNg̐Ε\(pXZp[^\/ǂOKB
	 *                ȂÃZp[^͎ŕt)
	 * @param fileName ΏۂƂt@C̐Ε\(pXZp[^\/ǂOK)
	 * @return fileNameparentƂ΃pX{t@C
	 * @throws NullArgumentException null̏ꍇ
	 */
	static public String calcRelativePath(String parent, String fileName) {
		ArgumentChecker.throwIfNull(parent, fileName);
		parent = FilenameUtils.separatorsToSystem(parent);
		if (parent.charAt(parent.length() - 1) != File.separatorChar) {
			parent = parent + File.separator;
		}
		fileName = FilenameUtils.separatorsToSystem(fileName);
		
		String parentRoot = FilenameUtils.getPrefix(parent);
		String fileRoot =  FilenameUtils.getPrefix(fileName);
		if (!parentRoot.equals(fileRoot)) {
			return null;
		}

		// pXfBNgɕB擪̈v镔폜B
		// P dir1\dir2, F dir1  P dir2, F <<empty>>
		Queue<String> parentPathes = divide(FilenameUtils.getPath(parent));
		Queue<String> filePathes = divide(FilenameUtils.getPath(fileName));
		while(!parentPathes.isEmpty() && !filePathes.isEmpty() 
				&& parentPathes.peek().equals(filePathes.peek())) { 
			parentPathes.poll();
			filePathes.poll();
		}
		
		// parent̃fBNg̎c萔"../"
		// ̌filenamẽfBNg̎c萔Α΃pXłB
		StringBuilder builder = new StringBuilder();
		while (parentPathes.poll() != null) {
			builder.append("..").append(File.separatorChar); //$NON-NLS-1$
		}
		for (String dir : filePathes) {
			builder.append(dir).append(File.separatorChar);
		}
		if ((builder.length() != 0) && 
				(builder.charAt(builder.length() - 1) != File.separatorChar))
		{
			builder.append(File.separator);
		}
		builder.append(FilenameUtils.getName(fileName));
		return builder.toString();
	}
	
	/**
	 * Q̃pX̋ʂ̃[gpX߂܂B<br>
	 * ȉ͗łF
	 * <pre>
	 * pathA = C:\dir1, pathB = C:\dir1\file.name  C:\dir1 
	 * pathA = C:\dir1, pathB = C:\dir1\dir2  C:\dir1 
	 * pathA = C:\dir1, pathB = C:\dir2  C:\ 
	 * pathA = C:\dir1, pathB = D:\dir2  null
	 * </pre>
	 * @param pathA
	 * @param pathB
	 * @return
	 */
	static public String calcCommonRoot(File folderA, File folderB) {
		String pathA = getFullPath(folderA);
		String pathB = getFullPath(folderB);
		String rootA = FilenameUtils.getPrefix(pathA);
		String rootB =  FilenameUtils.getPrefix(pathB);
		if (!rootA.equals(rootB)) {
			return null;
		}
		pathA = pathA.substring(rootA.length());
		pathB = pathB.substring(rootB.length());
		// pXfBNgɕB擪̈v镔݂̂ǉ܂B
		// A dir1\dir2, B dir1  P dir2, F <<empty>>
		Queue<String> pathesA = divide(FilenameUtils.getPath(pathA));
		Queue<String> pathesB = divide(FilenameUtils.getPath(pathB));
		StringBuilder builder = new StringBuilder(rootA);
		while(!pathesA.isEmpty() && !pathesB.isEmpty() 
				&& pathesA.peek().equals(pathesB.peek())) { 
			builder.append(pathesA.poll()).append(File.separator);
			pathesB.poll();
		}
		return builder.toString();
	}

	/**
	 * pathBpathA̎qłƂtrueԂ܂B<br>
	 * ȉ͗łF
	 * <pre>
	 * folderA = C:\dir1, folderB = C:\dir1\file.name  true 
	 * folderA = C:\dir1, folderB = C:\dir1\dir2  true
	 * folderA = C:\dir1, folderB = C:\dir2  false
	 * folderA = C:\dir1, folderB = D:\dir2  false
	 * </pre>
	 * 
	 * @param folderA
	 * @param pathB
	 * @return
	 */
	static public boolean isChild(File folderA, File folderB) {
		String pathA = getFullPath(folderA);
		String pathB = getFullPath(folderB);

		String rootA = FilenameUtils.getPrefix(pathA);
		String rootB =  FilenameUtils.getPrefix(pathB);
		if (!rootA.equals(rootB)) {
			return false;
		}
		pathA = FilenameUtils.getPath(pathA.substring(rootA.length()));
		pathB = FilenameUtils.getPath(pathB.substring(rootB.length()));
		return pathB.startsWith(pathA);
	}

	/**
	 * pX(a/b/c)̊efBNg[zɕ܂B<br>
	 * "a/b/c"  String[]{"a","b","c"}
	 * @param path
	 * @return
	 */
	static private Queue<String> divide(String path) {
		StringTokenizer stringTokenizer = new StringTokenizer(path, File.separator);
		Queue<String> ret = new LinkedList<String>();
		while(stringTokenizer.hasMoreTokens()) {
			ret.add(stringTokenizer.nextToken());
		}
		return ret;
	}
	
	/**
	 * filefBNg[̏ꍇɖFile.separatorĕԂ܂B
	 * @param file
	 * @return
	 */
	static private String getFullPath(File file) {
		String path = file.getAbsolutePath();
		if (file.isDirectory() 
				&& (path.charAt(path.length() - 1) != File.separatorChar)) {
			return path + File.separator;
		}
		return path;
	}
	
	/**
	 * JgfBNgԂ܂B
	 * @return
	 */
	static public File getCurrentDirectory() {
		return new File(".").getAbsoluteFile().getParentFile().getAbsoluteFile(); //$NON-NLS-1$
	}
	
	
	/**
	 * ꎞt@C[UƂƂ̃G[bZ[WłB
	 * u{0}\n̓VXepꎞt@CłB\n[UނƂ͂ł܂Bv
	 */
	static public final MessageResource MSG_CANT_WRITE_TO_TMP_FILE
			= new MessageResource(Ios.class, "mgsCantWriteToTmpFile"); //$NON-NLS-1$
	
	/**
	 * {@link #createTemporaryFile(File, int)}ōXVA
	 * {@link #isTemporaryFile(File)}ŎQƂ}bvłB
	 * ꎞt@C(AvIɍ폜)̏WłB
	 */
	static private final Set<File> tempFiles_s_ = new HashSet<File>(1);
	
	/**
	 * baseFile̊gq̎OdigitŃtH[}bg1`ő݃`FbNsA
	 * ܂݂Ȃt@CԂ܂<br>
	 * e|[t@Cw肵fBNgɍ쐬邽߂ɗp܂B<br>
	 * ǂ̔ԍɂ󂫂ꍇ{@link java.io.IOException}𑗏o܂B
	 * @param baseFile
	 * @param digit
	 * @return
	 * @throws IOException 
	 */
	static public File createTemporaryFile(File baseFile, int digit) throws IOException {
		ArgumentChecker.throwIfNull(baseFile);
		if (digit <= 0) {
			throw new IllegalArgumentException();
		}
		String absolutePath = baseFile.getAbsolutePath();
		String fullPath = FilenameUtils.getFullPath(absolutePath);
		String baseName = FilenameUtils.getBaseName(absolutePath);
		String extension = FilenameUtils.getExtension(absolutePath);
		
		int maxNo = 1;
		StringBuilder builder = new StringBuilder(digit);
		for (int i = 0; i < digit; ++i) {
			builder.append('0');
			maxNo *= 10;
		}
		DecimalFormat decimalFormat = new DecimalFormat(builder.toString());
		
		for (int i = 1; i <= (maxNo - 1); ++i) {
			File tmpFile = new File(fullPath + File.separator + baseName + decimalFormat.format(i) + EXTENSION_SEPARATOR + extension);
			if (!tmpFile.exists()) {
				tmpFile.createNewFile();
				tmpFile.deleteOnExit();
				synchronized (tempFiles_s_) {
					tempFiles_s_.add(tmpFile);
				}
				return tmpFile;
			}
		}
		throw new IOException();
	}

	/**
	 * file{@link #createTemporaryFile(File, int)}
	 * 쐬ꂽꎞt@C(AvIɍ폜)̎trueԂ܂B
	 * @param file
	 * @return
	 */
	static public boolean isTemporaryFile(File file) {
		return tempFiles_s_.contains(file);
	}

	/**
     * 
     */
    protected Ios() {
        super();
    }
}
