/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.unitarou.lang.NullArgumentException;
import org.unitarou.lang.Strings;
import org.unitarou.sgf.type.SgfColor;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.util.ArgumentChecker;

/**
 * SGFtH[}bgɂNode\NXłB
 * ɕID̈قȂPropertyێ܂B
 * 
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public class Node {
	/**
	 * colorŎw肳ꂽ΂pointŎw肳ꂽʒuɒ肵NodeCX^XԂ܂B
	 * 
	 * @throws NullArgumentException ǂ炩łnull̏ꍇ
	 */
	static public Node makeMoveNode(SgfColor color, SgfPoint point) {
		ArgumentChecker.throwIfNull(color, point);
	    Property property = color.moveType().makeProperty(point);
		Node node = new Node();
		node.properties_.add(property);
		node.propertyMap_.put(property.sgfId(), property);
		return node;
	}
	
	
	/** ^[Property]łBget()pĂ̂ArrayListŒłB*/
	private final ArrayList<Property> properties_;
	
	/**
	 * {@link #properties_}̌pɗpӂꂽ}bvłB<br>
	 * K{@link #properties_}Ƃ̓eĂKv܂B
	 */
	private final Map<SgfId, Property> propertyMap_;


	/**
	 * Node|̈̑Ouc;vɂ|̈łB
	 */
	private String openUtr_;
	
	/**
	 *  Node|̈̌u';c'vɂ|̈łB
	 */
	private String closeUtr_;
	
	/**
	 * ̃CX^X̃nbVR[hłB<br> 
	 * CX^XύXǗł
	 * {@link #openUtr_}A{@link #closeUtr_}A{@link #propertyMap_}
	 * XVꂽ0Zbg܂B
	 */
	private int hashCode_;

	/**
	 * 
	 */
	public Node() {
		super();
		
		//vpeB̔z񐔂(唼Ȃ̂)P
		properties_ = new ArrayList<Property>(1);
		propertyMap_ = new HashMap<SgfId, Property>(1);
		openUtr_ = Strings.EMPTY;
		closeUtr_ = Strings.EMPTY;
		hashCode_ = 0;
	}
	
	/**
	 * ̃NX̃Rs[RXgN^[łB
	 * @param src
	 * @throws NullArgumentException srcnull̏ꍇ
	 */
	public Node(Node src) {
	    super();
	    ArgumentChecker.throwIfNull(src);

		properties_ = new ArrayList<Property>(src.properties_.size());
		propertyMap_ = new HashMap<SgfId, Property>(src.properties_.size());

		for (Property srcProperty : src.properties_) {
		    Property copied = new Property(srcProperty);
			properties_.add(copied);
			propertyMap_.put(copied.sgfId(), copied);
		}
		openUtr_ = src.openUtr_;
		closeUtr_ = src.closeUtr_;
		hashCode_ = src.hashCode_;
	}
	
	/**
	 * vpeBƂpropertyRXgN^łB
	 * property<code>null</code>̏ꍇ́A{@link #Node()}ƓɂȂ܂B
	 * @throws IllegalArgumentException {@link Property#isValid()}falsȅꍇB
	 */
	public Node(Property property) {
		this();
		if (property == null) {
			return;
		}
		if (!property.isValid()) {
			throw new IllegalArgumentException("A property must be valid."); //$NON-NLS-1$
		}
		Property copied = new Property(property);
		properties_.add(copied);
		propertyMap_.put(copied.sgfId(), copied);
	}
	
	
	/**
	 * ̃CX^Xpropertyǉ܂B<br>
	 * ɃCX^XɓID݂ꍇ͏㏑A
	 * łȂꍇɂ͒ǉ܂B
	 * property͕ʂĕێ܂B
	 * Ẵ\bhɓnCX^XĂA
	 * {NodeCX^Xproperty͉e󂯂܂B
	 * <b></b>CA^Ô݂͌뎯ʂ邽߂ɕK擪ɒǉ܂B
	 * 
	 * @throws NullArgumentException property<code>null</code>̏ꍇB
	 * @throws IllegalArgumentException {@link Property#isValid()}falsȅꍇB
	 */
	public void addProperty(Property property) {
		if (property == null) {
			throw new NullArgumentException();
		} else if (!property.isValid()) {
			throw new IllegalArgumentException("A property must be valid. property = " + property); //$NON-NLS-1$
		}
		
		hashCode_ = 0;
		Property copied = new Property(property);
        propertyMap_.put(copied.sgfId(), copied);
		for (int i = 0; i < properties_.size(); ++i) {
		    Property p = properties_.get(i);
		    if (p.getId().equals(property.getId())) {
		        properties_.set(i, copied);
		        return;
		    }
		}
		if (copied.sgfId().equals(SgfId.CHARSET)) {
			properties_.add(0, copied);
		} else {
			properties_.add(copied);
		}
	}
 
 	/**
 	 * idPropertỹm[h폜܂B
 	 * @param id
 	 * @return 폜ꂽNodeBYȂnullB
	 * @throws NullArgumentException id<code>null</code>̏ꍇB
 	 */
 	public Property removeProperty(String id) {
 		ArgumentChecker.throwIfNull(id);
		for (Iterator<Property> ip = properties_.iterator(); ip.hasNext(); ) {
		    Property property = ip.next();
		    if (property.getId().equals(id)) {
		        ip.remove();
		        propertyMap_.remove(property.sgfId());
				hashCode_ = 0;
		        return property;
		    }
		}
 		return null;
 	}
 	
 	/**
 	 * ŕێĂ{@link Property}SăNA[܂B 	 */
 	public void clearProperty() {
 		properties_.clear();
 		propertyMap_.clear();
		hashCode_ = 0;
 	}

 	
 	/**
	 * Node|̈̑Ouc;vɂ|̈ݒ肵܂B
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
	 */
	public void setOpenUtr(String value) {
		ArgumentChecker.throwIfNull(value);
		openUtr_ = value;
		hashCode_ = 0;
	}

	/**
	 * Node|̈̌u';c'vɂ|̈ݒ肵܂B
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
	 */
	public void setCloseUtr(String value) {
		ArgumentChecker.throwIfNull(value);
		closeUtr_ = value;
		hashCode_ = 0;
	}

 	/**
 	 * typeɕRÂProperty݂ꍇtrueԂ܂B
	 * @throws NullArgumentException type<code>null</code>̏ꍇB
 	 */
	public boolean contains(SgfId type) {
 		ArgumentChecker.throwIfNull(type);
 		return propertyMap_.containsKey(type);
	}

 	/**
 	 * idɕRÂPropertyԂ܂B<br>
 	 * ȂꍇnullԂ܂B
 	 * ܂ÃCX^XێĂProperty𕡎ʂĕԂ܂̂ŁA
 	 * ʂɑĂÃCX^X͉e󂯂܂B
 	 * {@link #addProperty(Property)}ŕύXsĂB
	 * @throws NullArgumentException id<code>null</code>̏ꍇB
 	 */
 	public Property getProperty(String id) {
 		ArgumentChecker.throwIfNull(id);
 		return getProperty(SgfId.find(id));
 	}
 	
 	/**
 	 * typeɕRÂPropertyԂ܂B<br>
 	 * ȂꍇnullԂ܂B
 	 * ܂ÃCX^XێĂProperty𕡎ʂĕԂ܂̂ŁA
 	 * ʂɑĂÃCX^X͉e󂯂܂B
 	 * {@link #addProperty(Property)}ŕύXsĂB
	 * @throws NullArgumentException id<code>null</code>̏ꍇB
 	 */
 	public Property getProperty(SgfId id) {
 		ArgumentChecker.throwIfNull(id);
 		return propertyMap_.get(id);
 	}

 	/**
	 * CX^XێĂSPropertyԂ܂B<br>
	 * vfString̏ɂȂ܂܂BȂꍇ͗vfO̔zԂ܂B
	 * ̃\bh̓CX^XێĂProperty𕡎ʂĕԂ̂ŁA
	 * ʂɑĂÃCX^X͉e󂯂܂B
	 * {@link #addProperty(Property)}ŕύXsĂB
	 */
 	public Property[] getProperties() {
 		Property[] ret = new Property[properties_.size()];
	    for(int i = 0; i  < properties_.size(); ++i) {
	        ret[i] = new Property(properties_.get(i));
	    }
 		return ret;
 	}
 	
 	/**
 	 * ݕێĂPropertyid̏WԂ܂B<br>
 	 * ^[String]łBȂȀW͕ҏWsłB
 	 */
 	public Set<SgfId> getPropertyIds() {
 	    Set<SgfId> idSet = new HashSet<SgfId>(properties_.size());
	    for (Property property : properties_) {
 	        idSet.add(property.sgfId());
	    }
 		return Collections.unmodifiableSet(idSet);	
 	}

	/**
	 * ̃m[hpropertyTypeŎw肵vpeB݂ꍇ
	 * trueԂ܂B
	 * @throws NullArgumentException null̏ꍇ
	 */
	public boolean contains(PropertyType propertyType) {
		ArgumentChecker.throwIfNull(propertyType);
	    for (Property property : properties_) {
	        if (property.sgfId().propertyType().equals(propertyType)) {
	        	return true;
	        }
	    }
	    return false;
	}
	
	/**
	 * Node|̈̑Ouc;vɂ|̈Ԃ܂B<br>
	 * <code>null</code>͕Ԃ܂B
	 */
	public String getOpenUtr() {
		return openUtr_;
	}

	/**
	 * Node|̈̌u';c'vɂ|̈Ԃ܂B<br>
	 * <code>null</code>͕Ԃ܂B
	 */
	public String getCloseUtr() {
		return closeUtr_;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		
		if (obj == null || !obj.getClass().equals(getClass())) {
			return false;
		}
		
		Node des = (Node)obj;
		return properties_.equals(des.properties_)
		        && openUtr_.equals(des.openUtr_)
		        && closeUtr_.equals(des.closeUtr_);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		if (hashCode_ == 0) {
			hashCode_ = properties_.hashCode()
						+ openUtr_.hashCode() * 7
      					+ closeUtr_.hashCode() * 11;
		}
		return hashCode_;
	}
}
