/*  
 * Copyright 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.parser;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CoderResult;
import java.nio.charset.CodingErrorAction;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.util.ArgumentChecker;

/**
 * {@link java.nio.ByteBuffer}
 * R[h؂ւ悤ɒ[_[łB
 * RootGameTreeɕR[hZbg͊蓖Ă܂B
 * 
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
final class NioReader implements SgfParserReader {
    /** ̃NX̃K[łB*/
	static private final Log log_s_ = LogFactory.getLog(NioReader.class);
   
    
    /** SGFt@C̃oCgC[WłB */
    private final ByteBuffer byteBuffer_;
        
    /**
     * ftHg̕R[hZbgłB
     * {@link #markRootGameTree()}̒͂̕R[h
     * fR[h܂B
     */
    private final Charset defaultCharset_;
    
    /**
     * ݓǂݏoƂĂRootGameTreep̕R[hƂ
     * r܂ŃfR[hiĎsjR[h܂B
     * {@link #setCharset(Charset)}ł̃CX^XɊ܂܂Ă
     * R[hw肷ƃG[͑S{@link CodingErrorAction#REPLACE}
     * ɂȂ܂B
     */
    private final Set<Charset> triedCharsets_;
    
    /**
     * ݒڂĂ{@link org.unitarou.sgf.RootGameTree}
     * {@link #byteBuffer_}ł̊Jnʒuێ܂B
     * {@link #markRootGameTree()}Őݒ肳܂B
     */
    private int rgtStartPos_;
    
    
    /**
     * ݒڂĂ{@link org.unitarou.sgf.RootGameTree}
     * {@link #byteBuffer_}ϊR[h邽߂̃fR[_łB
     * {@link #markRootGameTree()}{@link #defaultCharset_}ɑΉfR[_ɐݒ肳A
     * ܂B
     */
    private CharsetDecoder rgtDecoder_;
    
    /**
     * {@link #read()}\bhp̈ꎞobt@[łB
     * \bhďoɃCX^X쐬邱Ƃ邽߁A
     * CX^XňێĂ܂B
     */
    private final CharBuffer charBuffer_;
    
    /**
     * Pn܂sԍłB
     * s̒`SGFɏ]A\rA\r\nA\nA\n\r̉ꂩŉsƂ܂B
     * ŒvƂ܂B
     */
    private int lineNumber_ = 1;
    
    /** Ōɓǂݍ񂾕CR('\r')̏ꍇtrueɂȂĂtOłB */
    private boolean isLastCharacterCR_ = false;
    
    /** {@link #mark(int)}ꂽ_ł{@link #lineNumber_}łB*/
    private int markedLineNubmer_ = -1;
    
    /** {@link #mark(int)}ꂽ_ł{@link #isLastCharacterCR_}łB*/
    private boolean markedIsLastCharacterCR_;
    
    
    /**
     *  
     * @param byteBuffer asReadOnlyBufferɂėp܂Bposition0ɃZbg܂B
     * @param defaultCharset
     * 
     * @throws org.unitarou.lang.NullArgumentException nullꍇB
     */
    public NioReader(ByteBuffer byteBuffer, Charset defaultCharset) {
        super();
        ArgumentChecker.throwIfNull(byteBuffer, defaultCharset);
        byteBuffer_ = byteBuffer.asReadOnlyBuffer();
        byteBuffer_.rewind();
        defaultCharset_ = defaultCharset;
        triedCharsets_ = new HashSet<Charset>(2);
        charBuffer_  = CharBuffer.allocate(1);
        markRootGameTree(); // finalNXȂ̂ŃI[o[Ch̖͂Ȃ
    }
    
    /*
     * @see SgfParserReader#markRootGameTree()
     */
    public void markRootGameTree() {
        rgtStartPos_ = byteBuffer_.position();
        triedCharsets_.clear();
        rgtDecoder_ = defaultCharset_.newDecoder();
        rgtDecoder_.onMalformedInput(CodingErrorAction.REPLACE);
        rgtDecoder_.onUnmappableCharacter(CodingErrorAction.REPLACE);
    }

    /* (non-Javadoc)
     * @see org.unitarou.sgf.parser.SgfParserReader#setCharset(java.nio.charset.Charset)
     */
    public boolean setCharset(Charset newCharset) {
        ArgumentChecker.throwIfNull(newCharset);

        if (triedCharsets_.contains(newCharset)) {
        	return false;
        }

		logChasetChange(newCharset);
        rgtDecoder_ = newCharset.newDecoder();
        rgtDecoder_.onMalformedInput(CodingErrorAction.REPORT);
        rgtDecoder_.onUnmappableCharacter(CodingErrorAction.REPLACE);
        triedCharsets_.add(newCharset);
        byteBuffer_.rewind();
        byteBuffer_.position(rgtStartPos_);
        return true;
    }
    
    private void logChasetChange(Charset newCharset) {
        if (!log_s_.isDebugEnabled()) {
        	return;
        }
    	StringBuilder builder = new StringBuilder();
    	if (rgtDecoder_ != null) {
    		builder.append(rgtDecoder_.charset()).append("->");//$NON-NLS-1$
    	}
    	builder.append(newCharset); 
    	log_s_.debug(builder.toString());
    }
    
    /* (non-Javadoc)
     * @see org.unitarou.sgf.parser.SgfParserReader#getCharset()
     */
    public Charset getCharset() {
        return rgtDecoder_.charset();
    }


    /**
	 * @return Returns the defaultCharset.
	 */
	public Charset getDefaultCharset() {
		return defaultCharset_;
	}

	/* (non-Javadoc)
     * @see org.unitarou.sgf.parser.SgfParserReader#read()
     */
    public int read() throws BadCharsetException {
        if (byteBuffer_.position() == byteBuffer_.limit()) {
            return -1; 
        }
        int lastPos = byteBuffer_.position();
        charBuffer_.clear();
        CoderResult cr = rgtDecoder_.decode(byteBuffer_, charBuffer_, false);
        if (lastPos == byteBuffer_.position()) {
    		log_s_.info("Bad encoding: skip at line " + lineNumber_); //$NON-NLS-1$
        	byteBuffer_.position(lastPos + 1);
        }
    	if (cr.isMalformed()) {
    		log_s_.info("Bad encoding: Malformed at line " + lineNumber_); //$NON-NLS-1$
        	throw new BadCharsetException();
    	}
        char c = charBuffer_.get(0);
        countLineNumber(c);
        return c;
    }

    /* (non-Javadoc)
     * @see org.unitarou.sgf.parser.SgfParserReader#mark(int)
     */
    public void mark(int readAheadLimit) {
        byteBuffer_.mark();
        markedLineNubmer_ = lineNumber_;
        markedIsLastCharacterCR_ = isLastCharacterCR_;
    }

    /* (non-Javadoc)
     * @see org.unitarou.sgf.parser.SgfParserReader#reset()
     */
    public void reset() {
        byteBuffer_.reset();
        lineNumber_ = markedLineNubmer_;
        isLastCharacterCR_ = markedIsLastCharacterCR_;
    }

    /* (non-Javadoc)
     * @see org.unitarou.sgf.parser.SgfParserReader#getLineNumber()
     */
    public int getLineNumber() {
        return lineNumber_;
    }
    
    /**
     * sJEg܂B
     * \rA\r\nA\n\rA\nƂ(Œv)A
     * CNg܂B
     * 
     * @param character {@link #read()}œǂݏoꂽꕶ
     */
    private void countLineNumber(int character) {
        if (character == '\r') {
            if (isLastCharacterCR_) {
                ++lineNumber_;
            } else {
                isLastCharacterCR_ = true;
            }
            
        } else if (character == '\n') {
            ++lineNumber_;
            isLastCharacterCR_ = false;

        } else {
            if (isLastCharacterCR_) {
                ++lineNumber_;
            }
            isLastCharacterCR_ = false;
        }
    }

}
