/*  
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.parser;

import java.io.IOException;
import java.io.Reader;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.text.CharacterIterator;
import java.text.StringCharacterIterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.ml.MessageResource;
import org.unitarou.sgf.Collection;
import org.unitarou.sgf.GameTree;
import org.unitarou.sgf.Node;
import org.unitarou.sgf.Property;
import org.unitarou.sgf.RootGameTree;
import org.unitarou.sgf.Sequence;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.Sgfs;
import org.unitarou.sgf.Value;
import org.unitarou.sgf.ValueType;
import org.unitarou.sgf.util.SgfArgumentChecker;
import org.unitarou.util.ArgumentChecker;


/**
 * Sgfp[T̃RAWbNi[\bhNXłB
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
public class SgfParserCore {
    /** ̃NX̃VXeIȃK[łB*/
	static private final Log log_s_ = LogFactory.getLog(SgfParserCore.class);
	
	/** uJnǂݍݎɕsȕ̂Ŗ܂Bv*/
	static private final MessageResource NT_IGNORE_FILEHEAD 	
			= new MessageResource(SgfParserCore.class, "ntIgnoreFilehead"); //$NON-NLS-1$

	/** ů̕Ԃɕsȕ̂Ŗ܂Bv*/
	static private final MessageResource NT_IGNORE_INTER_TREE 	
			= new MessageResource(SgfParserCore.class, "ntIgnoreInterTree"); //$NON-NLS-1$
	
	/** uSGFNodeԂɕsȕ̂Ŗ܂Bv*/
	static private final MessageResource NT_IGNORE_INTER_NODE 
			= new MessageResource(SgfParserCore.class, "ntIgnoreInterNode"); //$NON-NLS-1$
	
	/** uSGFPropValuePropValue̊Ԃɕsȕ̂Ŗ܂Bv*/
	static private final MessageResource NT_IGNORE_INTER_VALUE	
			= new MessageResource(SgfParserCore.class, "ntIgnoreInterValue"); //$NON-NLS-1$
	
	/** uSGFIDǂݍݎɕsȕ̂Ŗ܂Bv*/
	static private final MessageResource NT_IGNORE_ID 
			= new MessageResource(SgfParserCore.class, "ntIgnoreId"); //$NON-NLS-1$
	
	/** uSGFIDłB̃^O𖳎ď𑱍s܂Bv*/
	static private final MessageResource NT_IGNORE_EMPTY_ID	 	
			= new MessageResource(SgfParserCore.class, "ntIgnoreEmptyId"); //$NON-NLS-1$
	
	/** u^O͕̒lĂȂ̂ŁAŏ̃^Ô݂܂Bv*/
	static private final MessageResource NT_IGNORE_MULTI_VALUES 	
			= new MessageResource(SgfParserCore.class, "ntIgnoreMultiValues"); //$NON-NLS-1$
	
	/** uRoot property[gm[hȊOŌ܂Bȉ̃vpeB[gm[hɑgݍ݂܂Bv */
	static private final MessageResource NT_INVALID_POSITION_ROOT_PROPERTY 
			= new MessageResource(SgfParserCore.class, "ntInvalidPositionRootProperty"); //$NON-NLS-1$

	/** usȃvpeB܂Bv*/
	static private final MessageResource NT_REMOVE_INVALID_PROPERTY 
			= new MessageResource(SgfParserCore.class, "ntRemoveInvalidIdProperty"); //$NON-NLS-1$

	/** uSGFIDǂݍݎɋ󔒂̂Ŗ̗̕p܂v */
	static private final MessageResource NT_TRIM_ID 
			= new MessageResource(SgfParserCore.class, "ntTrimId"); //$NON-NLS-1$
	

	/**
	 * a-zAA-Z̊ԂtrueԂ܂B 
	 */
	static private boolean isAlphabet(int c) {
		return (('A' <= c) && (c <= 'Z')) || (('a' <= c) && (c <= 'z'));
	}

	/**
	 * ł{@link Character#isWhitespace(char)}ƓłB
	 * R[hłĂtrueԂ܂B
	 */
	static private boolean isWhitespace(int c) {
	    return Character.isWhitespace((char)c);
	}

	private SgfParserLogger parserLogger_;
	private SgfParserReader reader_;
	private boolean isInputTerminated_;
	
	/**
	 * CX^XŜŋL{@link StringBuilder}łB
	 */
	private final StringBuilder sharedBuffer_;

	/**
	 * 
	 */
	public SgfParserCore() {
		super();
		sharedBuffer_ = new StringBuilder();
		parserLogger_ = new SgfParserLogger();
	}

	/**
	 * @deprecated ̃\bhreaderw肷ƁAVM̃ftHgȊO̕R[hɑΉł܂B
	 *               {@link #setByteBuffer(ByteBuffer, Charset)}gĂB
	 */
	@Deprecated
	public void setReader(Reader reader) {
	    reader_ = new WellbehavedLineNumberReader(reader);
	}
	
	/**
	 * 
	 * @param byteBuffer
	 * @param defaultCharset
     * @throws org.unitarou.lang.NullArgumentException nullꍇB
	 */
	public void setByteBuffer(ByteBuffer byteBuffer, Charset defaultCharset) {
        ArgumentChecker.throwIfNull(byteBuffer, defaultCharset);
	    reader_ = new NioReader(byteBuffer, defaultCharset);
	}
	
	public void setParserLogger(SgfParserLogger parserLogger) {
	    parserLogger_ = parserLogger;
	}
	
	public SgfParserLogger getParserLogger() {
	    return parserLogger_;
	}

	/** collection̍ŌɁARNVǉ܂B
	 * ܂p[XłꍇtrueAȏp[XłȂꍇfalseԂ܂B
	 * @throws IOException
	 */
	public boolean parseCollection(Collection collection) throws IOException {
	    reader_.markRootGameTree();
	    while(true) {
		    try {
				return parseCollectionImpl(collection);
			} catch (BadCharsetException e) {
				if (e.getNewCharset() == null) {
					//TODO CA^OƈvĂȂ̂ŃOׂB
					reader_.setCharset(reader_.getDefaultCharset());
				}
			}
	    }
	}
	
	public boolean parseCollectionImpl(Collection collection) 
			throws IOException, BadCharsetException {
		StringBuilder utr = new StringBuilder();
		StringBuilder utrNode = new StringBuilder();
	    if (!hasMoreRootGame(utr, utrNode)) {
	        // GameTreeȂꍇ͖B
	        if (collection.size() != 0) {
	            utr.append(utrNode);
	            collection.get(collection.size() - 1).setCloseUtr(utr.toString());
	        }
	        return false;
	    }

	    RootGameTree rootGameTree = new RootGameTree();
	    rootGameTree.setOpenUtr(utr.toString());
		if (parseGameTree(rootGameTree, rootGameTree)) {
			collection.addLast(rootGameTree);
			rootGameTree.getSequence().getFirst().setOpenUtr(utrNode.toString());
		}
		rootGameTree.setActualCharset(reader_.getCharset());
	    SgfArgumentChecker.throwIfInvalid(rootGameTree);
	    return true;
	}
	
	/**
	 * ܂ǂݍ߂RootGameꍇi'('';'̑gݍ킹j
	 * ꍇtrueԂ܂B
	 * '('';'̊ԂɂWhitespacê݂܂(UTRƂ܂)B
	 * WhitespaceȊO̒lꍇ́ARootGameƂ݂͂Ȃ܂B
	 * ÕQ[SɓǂݏIĂ邱ƂOłB
	 * @param utrRgt RootGameTree܂łɓǂݍ񂾃eLXgf[^AUTRɂȂB
	 * @throws BadCharsetException 
	 */
	private boolean hasMoreRootGame(
			StringBuilder utrRgt, StringBuilder utrNode)
			throws IOException, BadCharsetException {
		boolean ret = false;
		boolean searchNodeStart = false;
		StringBuilder ignoreString = new StringBuilder();
        utrNode.delete(0, utrNode.length());
		int character = 0;
		do {
			reader_.mark(1);
			character = reader_.read();
			if (character == -1) {
			    break;
			    
			} else if (character == Sgfs.GAME_TREE_START_MARK) {
			    if (!searchNodeStart) {// n 
			        searchNodeStart = true;			        

			    // '('B
			    // d؂ȂɂȂ̂ŁAutrForNode̒gutrForRootGameTreeɈڂ
			    } else { 
			        utrRgt.append(Sgfs.GAME_TREE_START_MARK);
			        utrRgt.append(utrNode);
			        utrNode.delete(0, utrNode.length());
			    }
				
			// ';'tF[Y';'Δ	
			} else if (searchNodeStart && (character == Sgfs.NODE_START_MARK)) {
				reader_.reset();
				ret =  true;
				break;
				
			// WhitespaceȊȌꍇ
			// GameTree̒T蒼
			} else if (!isWhitespace(character)) {
			    if (searchNodeStart) {
			        searchNodeStart = false;
			        utrRgt.append(Sgfs.GAME_TREE_START_MARK);
			        utrRgt.append(utrNode);
			        utrRgt.append((char)character);
			        utrNode.delete(0, utrNode.length());
			    } else {
					ignoreString.append((char)character);
				    utrRgt.append((char)character);
			    }

			} else if (searchNodeStart) { 
			    utrNode.append((char)character);

			} else {
			    utrRgt.append((char)character);
			}
		} while(true);
		
		if (ignoreString.length() != 0) {
			writeInfoLog(NT_IGNORE_FILEHEAD, ignoreString);
		}
		return ret;
	}


	/**
	 * gemeTreẽV[PX̍ŌɁAm[hǉĂ܂B
	 * ̌AgameTree̎qgameTree()̃\bhċNI
	 * ĂяoƂŒǉĂ܂B
	 * p[XrŏIꍇfalseԂ܂B
	 * @throws IOException
	 * @throws BadCharsetException
	 */
	public boolean parseGameTree(RootGameTree root, GameTree gameTree) 
			throws IOException, BadCharsetException 
	{
		
//		if (!parseSequence(root, gameTree, gameTree.getSequence())) {
//			return false;
//		}
		// MEMO LRgAEg̕⑫
		// ł̃t@C̃G[(Sequence݂AȂGameTree
		// n܂Ăꍇ́AŏGameTreeSequenceɏiāAc
		// Siblings͖ĉ߂)ɑΉ邽߂ɁA
		// sequencẽp[Xɂ́AI`FbNȂB
		parseSequence(root, gameTree, gameTree.getSequence());
		
		GameTree childTree = new GameTree();
		do {
			StringBuilder utr2 = new StringBuilder();
			if (!hasMoreGameTree(utr2)) {
			    assert utr2.length() == 0 
			    	|| ((0 < utr2.length()) && (0 < gameTree.getChildrenSize()))
			    	: "Can't set utr2 in the last node."; //$NON-NLS-1$
			    if (0 < gameTree.getChildrenSize()) {
				    gameTree.getChild(gameTree.getChildrenSize() - 1)
				    	.setCloseUtr(utr2.toString());
			    }
			    break;
			}
			childTree.setOpenUtr(utr2.toString());
			if (parseGameTree(root, childTree)) {
				gameTree.addGameTree(childTree);
				childTree = new GameTree();		    
			} else {
			    break;
			}
		}while (true);

		
		// n
		if (gameTree.getSequence().size() != 0) {
		    return true;
		}
		
		// SequencegameTreeȂꍇ́Ap[XIƂ݂ȂB
		if (gameTree.getChildrenSize() == 0) {
		    return false;
		}
		
		// ł̃G[ɑΉ
		GameTree child = gameTree.getChild(0);
		gameTree.getSequence().clear();
		gameTree.getSequence().addAll(child.getSequence());
		gameTree.clearGameTree();
		
		GameTree[] gameTrees = child.getChildren();
		for (int i = 0; i < gameTrees.length; ++i) {
			gameTree.addGameTree(gameTrees[i]);
        }
		
		return true;
	}
	
	/**
	 * ݂Sequence('AB[...]'̎)̏ԂƂreaderǂݐi߂܂BA
	 * GameTree̊Jn('(')trueԂA
	 * GameTreȅI(')')EOFfalseԂ܂B
	 * ȊO̕utrɊi[܂B
	 * @throws IOException
	 * @throws BadCharsetException 
	 */
	private boolean hasMoreGameTree(StringBuilder utr) throws IOException, BadCharsetException {
		boolean ret = false;
		StringBuilder ignoreString = new StringBuilder();
		int character = 0;
		do {
			character = reader_.read();
			if (character == Sgfs.GAME_TREE_START_MARK){
				ret = true;
				break;

			} else if ((character == -1) || (character == Sgfs.GAME_TREE_END_MARK)) {
			    break;
				
			} else if (isWhitespace(character)) {
			    utr.append((char)character);
				 
			} else {
			    utr.append((char)character);
				ignoreString.append((char)character);
			}
			
		} while (true);

		if (ignoreString.length() != 0) {
			writeWarningLog(NT_IGNORE_INTER_TREE, ignoreString);
		}
		isInputTerminated_ = (character == -1);
		
		return ret;		
	}
	
	/** 
	 * p[XrŏIꍇȉꍇALȃm[hꍇj
	 * falseԂ܂
	 * @throws IOException
	 * @throws BadCharsetException
	 */
	public boolean parseSequence(
	        RootGameTree root, 
	        GameTree parent, 
	        Sequence sequence) 
					throws IOException, BadCharsetException
	{
		StringBuilder utr = new StringBuilder();
		Node node = new Node();
		while(parseNode(node, utr)) {
			sequence.addLast(node);
			node = new Node();
		}
		if (sequence.isEmpty()) {
		    return false;
		}
	    sequence.getLast().setCloseUtr(utr.toString());
	    return true;
	}
	
	/**
	 * p[XrŏIꍇȉꍇAm[h̐擪Ȃꍇj
	 * falseԂ܂B
	 * @throws IOException
	 * @throws BadCharsetException
	 */
	public boolean parseNode(Node node, StringBuilder utr) 
			throws IOException, BadCharsetException
	{
		if (!hasMoreNode(utr)) {
			return false;
		}

		node.setOpenUtr(utr.toString());
		utr.delete(0, utr.length());
		Property property = new Property();
		while(parseProperty(property, utr)) {
		    //TODO ɓm[hɓIDpropertyꍇ́F
		    //1. vpeBPlŁASimple text̏ꍇ͊̃m[h̍Ōɒǉ
		    //2. vpeBl̏ꍇ́Ãm[hɒǉ
		    //3. ȊO͖
		    //P`R̂ǂłĂG[bZ[W
		    
			property.setOpenUtr(utr.toString());
		    node.addProperty(property);

		    property = new Property();
			utr.delete(0, utr.length());
		}
		node.setCloseUtr(utr.toString());
		return true;
	}


	private boolean hasMoreNode(StringBuilder utr) throws IOException, BadCharsetException {
		boolean found = false;
		StringBuilder ignoreString = new StringBuilder();
		int character = 0;
		do {
			reader_.mark(1);
			character = reader_.read();

			if (character == Sgfs.NODE_START_MARK) {
				found = true;
				break;

			} else if (character == -1) {
				break;

			} else if ((character == Sgfs.GAME_TREE_START_MARK) || (character == Sgfs.GAME_TREE_END_MARK)) {
				reader_.reset();
				break;
				
			} else if (isWhitespace(character)) {
			    utr.append((char)character);
				 
			} else {
			    utr.append((char)character);
				ignoreString.append((char)character);
			}
		} while (true);

		if (ignoreString.length() != 0) {
			writeWarningLog(NT_IGNORE_INTER_NODE, ignoreString);
		}
		isInputTerminated_ = (character == -1);

		return found;
	}

	/**
	 * m[h̃p[XɎsꍇfalseԂ܂B
	 * @throws BadCharsetException
	 */
	public boolean parseProperty(
	        Property property, StringBuilder utr) 
					throws IOException, BadCharsetException 
	{
		StringBuilder utrVal = new StringBuilder();
		String entireId = parseIdentifier(utr, utrVal);
		if ((entireId == null) || (entireId.length() == 0)){
			return false;
		}
		
		property.setEntireId(entireId);
		
		parseValues(property, utrVal);
		if (property.size() == 0) {
			return false;			
		}
		
		// Pl͂Ȃ̂ɕ̒lB[jOOođs
		if (!ValueType.permitsMultiValues(property.sgfId().cardinalityType())
		        && (property.size() != 1)) {
		    Value[] values = property.getValues();
		    StringBuilder builder = new StringBuilder(property.getEntireId());
			for (int i = 1; i < values.length; ++i) {
			    builder.append(values[i].toString());
			}
			writeWarningLog(NT_IGNORE_MULTI_VALUES, builder.toString());

		    property.setValue(values[0]);
		}
		
		if (property.sgfId().equals(SgfId.CHARSET)) {
			try {
			    String code = property.getString();
			    Charset charset = Charset.forName(code);
		        if (reader_.setCharset(charset)) {
				    throw new BadCharsetException(charset);
			    } 
			} catch (IllegalCharsetNameException e) {
				//TODO O
			} catch(UnsupportedCharsetException e) {
				//TODO O
			}
		}

		
		// vpeB̒lsłꍇ
		// TODO p[T̃Oɏׂ
		if (log_s_.isDebugEnabled() && !property.isValid()) {
			StringBuilder sb = new StringBuilder();
			sb.append("ID: '" + property.getId() + "'(" + property.getEntireId() + ")"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String[] v = null;
			if (ValueType.permitsMultiValues(property.sgfId().cardinalityType())) {
				v = property.getStrings();
			} else {
				v = new String[] {property.getString()};
			}
			for (int i = 0; i < v.length; ++i) {
				sb.append("[" + v[i]+ "]");  //$NON-NLS-1$//$NON-NLS-2$
			}
			log_s_.debug(sb);
			
			property.clear();
		}
		return true;
	}

	/**
	 * At@xbgȊO͓ǂݔ΂B
	 * '['̎O܂łǂݍށB
	 * ';'A'('A')'A']'AnullԂB
	 * 
	 * @param reader
	 * @return ǂݏoꂽID
	 * @throws IOException
	 * @throws BadCharsetException 
	 */
	public String parseIdentifier(
			StringBuilder utrProp, StringBuilder utrVal) throws IOException, BadCharsetException {
		String id = null;
		StringBuilder ignoreString = new StringBuilder();
		StringBuilder identifier = new StringBuilder();
		int character = 0;
		do {
			reader_.mark(1); // Pǂݍނ̂łPnB
			character = reader_.read();
			if (isAlphabet(character)) {
				identifier.append((char)character);

			} else if (character == Sgfs.VALUE_START_MARK) {
				id = identifier.toString().trim();
				// ";[AP[`]B[`]̂悤ȏꍇɑΉ
				// LIDł͂Ȃꍇ́A܂łID╶utr
				// ڂČ𑱍sB
				if (id.length() == 0) { 
					writeWarningLog(NT_IGNORE_EMPTY_ID, new Object());
					utrProp.append(identifier);
					utrProp.append((char)character);
					identifier.delete(0, identifier.length());
					continue;
				}
				
				// Value̎Outro
				final String idSource = identifier.toString();
				CharacterIterator itr2 = new StringCharacterIterator(idSource);
			    for(char c = itr2.last(); c != CharacterIterator.DONE; c = itr2.previous()) {
			        if (Character.isWhitespace(c)) {
			            utrVal.insert(0, c);
			        } else {
			            break;
			        }
			    }

				// ^ɗLID(whitespace܂܂Ȃ)߂
			    int lastWhiteSpacePos = -1;
				CharacterIterator itr = new StringCharacterIterator(id);
			    for(char c = itr.last(); c != CharacterIterator.DONE; c = itr.previous()) {
			        if (Character.isWhitespace(c)) {
			            lastWhiteSpacePos = itr.getIndex();
			            break;
			        }
			    }
			    if (0 <= lastWhiteSpacePos) {
					writeWarningLog(NT_TRIM_ID, id);
					id = id.substring(lastWhiteSpacePos).trim();
				}
			    
				// ID̎Outro
				utrProp.append(idSource.substring(0, idSource.lastIndexOf(id)));
			    break;

			} else if ( (character == -1) 
			        || (character == Sgfs.GAME_TREE_START_MARK) 
			        || (character == Sgfs.GAME_TREE_END_MARK) 
					|| (character == Sgfs.NODE_START_MARK) 
					/*|| (character == Sgfs.VALUE_END_MARK)*/) {
				reader_.reset();
				isInputTerminated_ = (character == -1);
				utrProp.append(identifier);
				break;			

			} else if(isWhitespace(character)){
				identifier.append((char)character);

			} else {
			    utrProp.append(identifier);
			    utrProp.append((char)character);
			    identifier.delete(0, identifier.length());
				ignoreString.append((char)character);
			}
		} while(true);

		if (ignoreString.length() != 0) {
			writeWarningLog(NT_IGNORE_ID, ignoreString);
		}
		return id;
	}

	/**
	 * ]xɃR[hϊ
	 * [c]̓GXP[vKvB
	 * [܂łɃAt@xbgIB
	 * 
	 * @param reader
	 * @return
	 * @throws BadCharsetException 
	 */
	public void parseValues(Property property, StringBuilder utr) 
			throws IOException, BadCharsetException 
	{
		Value value = parseValue(property.sgfId().valueType());
		while (value != null) {
		    property.addValue(
		    		new Value(value.getString(), utr.toString()));
		    
		    utr.delete(0, utr.length());
			value = hasMoreValue(utr) ? parseValue(property.sgfId().valueType()) : null;
		}
		property.setCloseUtr(utr.toString());
	}

	
	/**
	 * ']'܂őSĂ̕l߂
	 * '\'玟̕']'łĂ̂܂܋l߂
	 * '\'ꍇ͖Ŏ̕i[܂B
	 * ܂A'\'̂̂i[܂B
	 * 
	 * @return
	 * @throws IOException
	 * @throws BadCharsetException 
	 */	
	public Value parseValue(ValueType valueType) throws IOException, BadCharsetException {
	    sharedBuffer_.delete(0, sharedBuffer_.length());
		int  character;
		do {
			reader_.mark(1); // Pǂݍނ̂łPnB
			character = reader_.read();
			if (character == -1) { // ]IĂ̂ŃG[
				isInputTerminated_ = true;
				return null; 	

			} else if (character == Sgfs.VALUE_END_MARK) {
			    String string = sharedBuffer_.toString();
			    CharSequence charSequence;
			    switch (valueType) {
				case TEXT:
					charSequence = Sgfs.unescapeTextForRead(string);
					break;
					
				case SIMPLE_TEXT:
					charSequence = Sgfs.unescapeSimpleTextForRead(string);
					break;
					
				case OTHER:
					charSequence = Sgfs.unespaceForRead(sharedBuffer_);
					break;
					
				default: //ϊȂ
					charSequence = string;
				}
			    return new Value(charSequence);

			} else if (character == Sgfs.ESCAPE_SOFT_LINE_BREAK) {
			    sharedBuffer_.append((char)character);
			    character = (char)reader_.read();
			    // EscapeŃt@CIĂ̂ŃG[
			    if (character == -1) {
					isInputTerminated_ = true;
					return null;
			    }
			}
			sharedBuffer_.append((char)character);
		} while(true);
	}

	/**
	 * '['TBtrueԂB
	 * ܂łWhitespaceȊO݂̕ꍇ́A
	 * (ŁFAt@xbg'('A')'A';'݂ꍇ)
	 * ߂falseԂB
	 * @throws BadCharsetException 
	 */
	private boolean hasMoreValue(StringBuilder utr) throws IOException, BadCharsetException {
		boolean ret = false;
		StringBuilder ignoreString = new StringBuilder();
		int character;
		do {
			reader_.mark(1);
			character = reader_.read();
			if (character == Sgfs.VALUE_START_MARK) {
				ret = true;
				break;
				
			} else if (character == -1) {
				break;
			
			} else if (isWhitespace(character)) {
			    utr.append((char)character);

			} else {
			    reader_.reset();
				break;
			}
		} while(true);

		if (ignoreString.length() != 0) {
			writeWarningLog(NT_IGNORE_INTER_VALUE, ignoreString);
		}
		isInputTerminated_ = (character == -1);

		return ret;
	}


	/**  */
	public void writeWarningLog(MessageResource messageResource, Object parameter) {
	    parserLogger_.warn(reader_.getLineNumber(), messageResource.get(),	parameter.toString());
	}

	/** p[Xʂ̃K[infox̃Oo܂B*/
	public void writeInfoLog(MessageResource messageResource, Object parameter) {
	    parserLogger_.info(reader_.getLineNumber(), messageResource.get(), parameter.toString());
	}
    /**
     * @return Returns the isInputTerminated.
     */
    public boolean isInputTerminated() {
        return isInputTerminated_;
    }
    /**
     * @param isInputTerminated The isInputTerminated to set.
     */
    public void setInputTerminated(boolean isInputTerminated) {
        isInputTerminated_ = isInputTerminated;
    }
}
