/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.type;

import java.util.EnumSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.lang.NameDisplayable;
import org.unitarou.ml.Messages;
import org.unitarou.sgf.SgfId;
import org.unitarou.util.ArgumentChecker;

/**
 * IWi^O[GT]̒lɈӖt^NXłB
 * 
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public enum GameType implements TypedString<GameType>, NameDisplayable {
    
 	GAME("1"), //$NON-NLS-1$
	PROBLEM("3"), //$NON-NLS-1$
	DRILL("4"); //$NON-NLS-1$

    static private final Log log_s_ = LogFactory.getLog(GameType.class);
    /**
     * {@link #displayName()}p̃bZ[WCX^XłB
     */
    static private final Messages messages_s_ = Messages.createByPackage(GameType.class);

	static private final Pattern condition_s_ = Pattern.compile("1|3|4"); //$NON-NLS-1$
	static private final int POS = 0; //K\̃O[vID
	
	/**
	 * 
	 * @param value
	 * @return
	 * @throws TypeParseException p[XɎsꍇ
	 * @throws org.unitarou.lang.NullArgumentException valuenull̏ꍇ
	 */
	static public GameType parse(String value) throws TypeParseException {
	    ArgumentChecker.throwIfNull(value);
		Matcher matched = condition_s_.matcher(value);
		if (!matched.matches()) {
			throw new TypeParseException("Bad argument value = " + value); //$NON-NLS-1$
		}
		String gameType = matched.group(POS);
		for (GameType type : GameType.values()) {
			if (type.typeName_.equals(gameType)) {
				return type;
			}
		}
		assert false;
		throw new TypeParseException();
	}
	
	/**
	 * {@link #parse(String)}ƈႢ{@link TypeParseException}𑗏oȂɁA
	 * ϊɎsꍇɂdefԂp[TłB 
	 * @param value
	 * @return
	 * @throws org.unitarou.lang.NullArgumentException valuenull̏ꍇ
	 */
	static public GameType parseQuietly(String value, GameType def) {
	    ArgumentChecker.throwIfNull(value);
	    try {
	        return parse(value);
	        
	    } catch (TypeParseException e) {
	        log_s_.debug("Unknown value for GameType: " + value, e); //$NON-NLS-1$
	        return def;
	    }
	}
	
	/**
	 * "134"̌`̕񂩂WԂ܂B<br>
	 * p[XɎsꍇdefԂ܂B
	 * @param value
	 * @param def
	 * @return
	 */
	static public EnumSet<GameType> parseSetQuietly(String value, EnumSet<GameType> def) {
		ArgumentChecker.throwIfNull(value, def);
		EnumSet<GameType> ret = EnumSet.noneOf(GameType.class);
		for (int i = 0; i < value.length(); ++i) {
			try {
				char c = value.charAt(i);
				ret.add(parse(String.valueOf(c)));
			} catch (TypeParseException e) {
		        log_s_.debug("Unknown value for GameType: " + value, e); //$NON-NLS-1$
			}
		}
		return (ret.isEmpty()) ? def : ret;
	}
 
	
	private final String typeName_;
	private final String displayNameKey_;
	
	/**
	 * @param typeName
	 */
	GameType(String typeName) {
		ArgumentChecker.throwIfNull(typeName);
		typeName_ = typeName;
		displayNameKey_ = Messages.createKey(getClass(), typeName_);
	}

	/**
	 * {@link SgfId#GAME_TYPE}̂ݎ󂯕t܂B
	 * 
	 * @see org.unitarou.sgf.type.TypedString#acceptable(org.unitarou.sgf.SgfId)
	 */
	public boolean acceptable(SgfId sgfId) {
		ArgumentChecker.throwIfNull(sgfId);
		return SgfId.GAME_TYPE.equals(sgfId);
	}

	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.Type#getValue()
	 */
	public String getString() {
		return typeName_;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Enum#toString()
	 */
	@Override
	public String toString() {
		return displayName();
	}

	/* (non-Javadoc)
	 * @see org.unitarou.lang.NameDisplayable#displayName()
	 */
	public String displayName() {
        return messages_s_.get(displayNameKey_);
	}
}
