/*  
 * Copyright 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.type;

import java.util.EnumSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.unitarou.lang.NameDisplayable;
import org.unitarou.ml.Messages;
import org.unitarou.sgf.SgfId;
import org.unitarou.util.ArgumentChecker;

/**
 * RE^O^tNXłB<br>
 * SGF̃[ł͌܂܂l݂̂ƋK肳Ă܂A
 * ɂ͂قƂǂǎĂ܂B
 * 
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
public class Result implements TypedString<Result> {
    /**
     * ʂ̌^K肷񋓎qłB
     * ^̎ނɂXRA肵܂B
     */
    public enum Archetype implements NameDisplayable {
    	/** W+ */
        WHITE_WIN		("W+"), //$NON-NLS-1$

    	/** W+Resign */
        WHITE_RESIGN 	("W+Resign"), //$NON-NLS-1$
        
        WHITE_R	 		("W+R"), //$NON-NLS-1$
        WHITE_Time 		("W+Time"), //$NON-NLS-1$
        WHITE_T 		("W+T"), //$NON-NLS-1$
        WHITE_FORFEIT 	("W+Forfeit"), //$NON-NLS-1$
        WHITE_F 		("W+F"), //$NON-NLS-1$
        
        /** B+ */
        BLACK_WIN    	("B+"), //$NON-NLS-1$

        BLACK_RESIGN 	("B+Resign"), //$NON-NLS-1$
        BLACK_R 		("B+R"), //$NON-NLS-1$
        BLACK_Time 		("B+Time"), //$NON-NLS-1$
        BLACK_T 		("B+T"), //$NON-NLS-1$
        BLACK_FORFEIT 	("B+Forfeit"), //$NON-NLS-1$
        BLACK_F 		("B+F"), //$NON-NLS-1$
        VOID 			("Void"), //$NON-NLS-1$
        UNKNOWN 		("?"), //$NON-NLS-1$
        ZERO 			("0"), //$NON-NLS-1$
        DRAW 			("Draw"), //$NON-NLS-1$
        OTHER			("Other"); //$NON-NLS-1$
    	
    	/**
    	 * XRAEnum̏WłB
    	 */
    	static public final EnumSet<Archetype> SCORED 
    			= EnumSet.of(Archetype.WHITE_WIN, Archetype.BLACK_WIN);

    	/**
         * {@link #displayName()}p̃bZ[WCX^XłB
         */
        static private final Messages messages_s_ = Messages.createByPackage(Archetype.class);
        static private final EnumSet<Archetype> findTargets_s_;
        static {
        	EnumSet<Archetype> buf = EnumSet.allOf(Archetype.class);
        	buf.remove(OTHER);
        	findTargets_s_ = buf;
        }
        
        /**
         * typeNameŎw肳ꂽlɂđ啶𖳎
         * v{@link Archetype}Ԃ܂B<br>
         * {@link Archetype#OTHER}ɂĂ͗OŁA
         * typeName{@link Archetype#OTHER}ȊÔǂƂvȂꍇɕԂ܂B
         * 
         * @param typeName <code>null</code>܂B
         * @return ɈvlAȂꍇ{@link Archetype#OTHER}ԂA
         *          <code>null</code>͕ԂȂB
         */
        static public Archetype find(String typeName) {
        	for (Archetype archetype : findTargets_s_) {
        		if (archetype.typeName_.equalsIgnoreCase(typeName)) {
        			return archetype;
        		}
        	}
        	return OTHER;
        }

        private final String typeName_;
    	private final String displayNameKey_;
	
    	/**
    	 * @param typeName
    	 */
    	Archetype(String typeName) {
    		ArgumentChecker.throwIfNull(typeName);
    		typeName_ = typeName;
    		displayNameKey_ = Messages.createKey(getClass(), typeName_);
    	}
    	
    	
		/* (non-Javadoc)
		 * @see java.lang.Enum#toString()
		 */
		@Override
		public String toString() {
			return displayName() + "(" + typeName_ + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		}


		/* (non-Javadoc)
		 * @see org.unitarou.lang.NameDisplayable#displayName()
		 */
		public String displayName() {
	        return messages_s_.get(displayNameKey_);
		}
		
		/**
		 * RXgN^Ŏw肵Ă^(ex, "W+Resign")Ԃ܂B
		 * @return
		 */
		public String typeName() {
			return typeName_;
		}
    }
    
    

    /** `ڏp̐K\łB */
    static private final Pattern scoreCondition_s_ = Pattern.compile("((W|B)\\+)(\\d+(\\.\\d+)?)"); //$NON-NLS-1$
    
    /** K\̃O[vIDłA{@link Matcher#group(int)}ł"W+""B+"܂B*/
	static private final int ENUM_POS = 1; //

	/** K\̃O[vIDłA{@link Matcher#group(int)}ł"1.5""3"Ȃǂ܂B*/
	static private final int SCORE_POS = 3; //K\̃O[vID
	
    /**
     * @param datum
     * @return
     * @throws TypeParseException
     */
    static public Result parse(String datum) {
        ArgumentChecker.throwIfNull(datum);
		Matcher matched = scoreCondition_s_.matcher(datum);
		if (matched.matches()) {
		    Archetype archetype = Archetype.find(matched.group(ENUM_POS));
		    double score = Double.parseDouble(matched.group(SCORE_POS));
		    return new Result(archetype, score);
		}
		
	    Archetype archetype = Archetype.find(datum);
	    if (archetype.equals(Archetype.OTHER)) {
	    	return new Result(datum);
	    }
	    return new Result(archetype);
    }
    
    private final Archetype archetype_;
    private final double score_;
    private final String value_;
    
    /**
     * @param resultEnum
     * @throws org.unitarou.lang.NullArgumentException  <code>null</code>̏ꍇB
     */
    public Result(Archetype resultEnum) {
    	super();
    	ArgumentChecker.throwIfNull(resultEnum);
    	archetype_ = resultEnum;
    	value_ = resultEnum.typeName_;
    	score_ = 0.0d;
    }

    /**
     * 
     * @param value
     * @throws org.unitarou.lang.NullArgumentException  <code>null</code>̏ꍇB
     */
    public Result(String value) {
    	super();
    	ArgumentChecker.throwIfNull(value);
    	archetype_ = Archetype.OTHER;
    	value_ = value;
    	score_ = 0.0d;
    }
    
    /**
     * @throws org.unitarou.lang.NullArgumentException  <code>null</code>̏ꍇB 
     */
    public Result(Archetype resultEnum, double score) {
        super();
        ArgumentChecker.throwIfNull(resultEnum);
        archetype_ = resultEnum;
        score_ = (Archetype.SCORED.contains(archetype_)) ? score : 0;
        value_ = archetype_.typeName_;
    }
    

	/**
	 * {@link SgfId#RESULT}̂ݎ󂯕t܂
	 * @see org.unitarou.sgf.type.TypedString#acceptable(org.unitarou.sgf.SgfId)
	 */
	public boolean acceptable(SgfId sgfId) {
		ArgumentChecker.throwIfNull(sgfId);
		return sgfId.equals(SgfId.RESULT);
	}
	
    /* (non-Javadoc)
     * @see org.unitarou.sgf.type.TypedString#getString()
     */
    public String getString() {
        if (Archetype.SCORED.contains(archetype_)) {
            return archetype_.typeName_ + String.valueOf(score_);
        }
        return archetype_.typeName_;
    }

    /* (non-Javadoc)
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     */
    public int compareTo(Result rhs) {
        int c = archetype_.typeName_.compareTo(rhs.archetype_.typeName_);
        return (c != 0) ? c : ((score_ - rhs.score_ > 0) ?  1 : -1);
    }


    /**
     * ʂ̌^K肷񋓎qԂ܂B
     * ^̎ނɂXRA肵܂B
     * @return
     */
    public Archetype getArchetype() {
    	return archetype_;
    }
    
    /**
     * @return
     */
    public double getScore() {
    	return score_;
    }
    
    public String getValue() {
    	return value_;
    }
}
