/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.type;

import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.lang.NameDisplayable;
import org.unitarou.ml.Messages;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.ValueType;
import org.unitarou.util.ArgumentChecker;

/**
 * ΂̐F\NXłB
 * ̃NX͊OWbNCX^X𐶐邱Ƃ͏o܂B
 * {@link SgfColor#BLACK}A{@link SgfColor#WHITE}݂̂p\łB
 * ܂̃NXImmutablełB
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public enum SgfColor implements TypedString<SgfColor>, NameDisplayable { 
    
	BLACK("B"), //$NON-NLS-1$
	WHITE("W"); //$NON-NLS-1$

	static private final Log log_s_ = LogFactory.getLog(SgfColor.class);
	static private final Pattern condition_s_ = Pattern.compile("B|W"); //$NON-NLS-1$
	static private final int POS = 0; //K\̃O[vID
	
	/**
	 * ӖIɔƒ`ł{@link SgfId}̏WłB
	 * ̓Iɂ́F{@link SgfId#WHITE}A{@link SgfId#ADD_WHITE}, 
	 *            {@link SgfId#TERRITORY_WHITE}A{@link SgfId#WHITE_RANK}A
	 *            {@link SgfId#WHITE_TEAM}A{@link SgfId#WHITE_TIME_LEFT}A
	 *            {@link SgfId#GOOD_FOR_WHITE}
	 * łB
	 */
	static private final Set<SgfId> TYPES_WHITE;
	
	/**
	 * ӖIɍƒ`ł{@link SgfId}̏WłB
	 * ̓Iɂ́F{@link SgfId#BLACK}A{@link SgfId#ADD_BLACK}, 
	 *            {@link SgfId#TERRITORY_BLACK}A{@link SgfId#BLACK_RANK}A
	 *            {@link SgfId#BLACK_TEAM}A{@link SgfId#BLACK_TIME_LEFT}
	 *            {@link SgfId#GOOD_FOR_BLACK}
	 * łB
	 */
	static private final Set<SgfId> TYPES_BLACK;

	static  {
	    TYPES_WHITE = new TreeSet<SgfId>();
	    TYPES_WHITE.add(SgfId.WHITE);
	    TYPES_WHITE.add(SgfId.ADD_WHITE);
	    TYPES_WHITE.add(SgfId.TERRITORY_WHITE);
	    TYPES_WHITE.add(SgfId.WHITE_RANK);
	    TYPES_WHITE.add(SgfId.WHITE_TEAM);
	    TYPES_WHITE.add(SgfId.WHITE_TIME_LEFT);
	    TYPES_WHITE.add(SgfId.GOOD_FOR_WHITE);
	    
	    TYPES_BLACK = new TreeSet<SgfId>();
	    TYPES_BLACK.add(SgfId.BLACK);
	    TYPES_BLACK.add(SgfId.ADD_BLACK);
	    TYPES_BLACK.add(SgfId.TERRITORY_BLACK);
	    TYPES_BLACK.add(SgfId.BLACK_RANK);
	    TYPES_BLACK.add(SgfId.BLACK_TEAM);
	    TYPES_BLACK.add(SgfId.BLACK_TIME_LEFT);
	    TYPES_BLACK.add(SgfId.GOOD_FOR_BLACK);
	}
	
	/**
	 * valueB̂Ƃ{@link #BLACK}AW̎{@link #WHITE}Ԃ܂B
	 *
	 * @throws TypeParseException valueBłWłȂꍇ
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
	 */
	static public SgfColor parse(String value) throws TypeParseException {
	    ArgumentChecker.throwIfNull(value);
		Matcher matched = condition_s_.matcher(value);
		if (!matched.matches()) {
			throw new TypeParseException("Bad argument value = " + value); //$NON-NLS-1$
		}
		String color = matched.group(POS);
		return color.equals(BLACK.string_) ? BLACK : WHITE;
	}
	/**
	 * {@link TypeParseException}𑗏oȂɁA
	 * p[XɎsꍇnullԂ܂B
	 * 
	 * @param value
	 * @return
	 */
	static public SgfColor parseQuietly(String value) {
	    try {
	        return parse(value);
	    } catch (TypeParseException e) {
	        log_s_.debug("Parse failure: ", e); //$NON-NLS-1$
	        return null;
	    }
	}
	
	/**
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ 
	 */
	static public boolean isValid(String value) {
	    ArgumentChecker.throwIfNull(value);
	    
		Matcher matched = condition_s_.matcher(value);
		return matched.matches();
	}
	
	
	/**
	 * sgfTypȇΉFԂ܂B
	 * ̓Iɂ́F{@link SgfId#WHITE}A{@link SgfId#ADD_WHITE}, 
	 *            {@link SgfId#TERRITORY_WHITE}A{@link SgfId#WHITE_RANK}A
	 *            {@link SgfId#WHITE_TEAM}A{@link SgfId#WHITE_TIME_LEFT}
	 *            {@link SgfId#GOOD_FOR_WHITE}
	 * {@link SgfColor#WHITE}A
	 * ̓Iɂ́F{@link SgfId#BLACK}A{@link SgfId#ADD_BLACK}, 
	 *            {@link SgfId#TERRITORY_BLACK}A{@link SgfId#BLACK_RANK}A
	 *            {@link SgfId#BLACK_TEAM}A{@link SgfId#BLACK_TIME_LEFT}
	 *            {@link SgfId#GOOD_FOR_BLACK}
	 * {@link SgfColor#BLACK}Ԃ܂B
	 * ȊȌꍇnullԂ܂B
	 * 
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ 
	 */
	static public SgfColor getColor(SgfId sgfType) {
	    ArgumentChecker.throwIfNull(sgfType);
	    if (TYPES_WHITE.contains(sgfType)) {
	        return WHITE;
	    } else if (TYPES_BLACK.contains(sgfType)) {
	        return BLACK;
	    }
	    return null;
	}
	
	
	/**
	 * sgfType̐Ί֘AőΉFԂ܂B
	 * ̓Iɂ́F{@link SgfId#WHITE}A{@link SgfId#ADD_WHITE}, 
	 * {@link SgfColor#WHITE}A
	 * ̓Iɂ́F{@link SgfId#BLACK}A{@link SgfId#ADD_BLACK}, 
	 * {@link SgfColor#BLACK}Ԃ܂B
	 * ȊȌꍇnullԂ܂B
	 * 
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ 
	 */
	static public SgfColor getStoneColor(SgfId sgfType) {
	    ArgumentChecker.throwIfNull(sgfType);
	    
	    if (SgfId.WHITE.equals(sgfType) || SgfId.ADD_WHITE.equals(sgfType)) {
	    	return WHITE;
	    } else if (SgfId.BLACK.equals(sgfType) || SgfId.ADD_BLACK.equals(sgfType)) {
	    	return BLACK;
	    }
	    return null;
	}

	
	private final String string_;
	private final String displayName_;
	
	SgfColor(String string) {
		ArgumentChecker.throwIfNull(string);
		string_ = string;
		displayName_  = 
				Messages.createByPackage(SgfColor.class).get(
							Messages.createKey(SgfColor.class, string_));
	}
	
	
	/**
	 * ݂̐Fɍ킹Ē{@link SgfId#WHITE}{@link SgfId#BLACK}Ԃ܂B
	 */
	public SgfId moveType() {
	    if (WHITE.equals(this)) { 
	        return SgfId.WHITE;
	    } else if (BLACK.equals(this)) {
	        return SgfId.BLACK;
	    } 
	    assert false : "Unknown type: " + this; //$NON-NLS-1$
	    return null;
	}
	
	/**
	 * ݂̐Fɍ킹Ĕz{@link SgfId#ADD_WHITE}{@link SgfId#ADD_BLACK}Ԃ܂B
	 */
	public SgfId setupType() {
	    if (WHITE.equals(this)) { 
	        return SgfId.ADD_WHITE;
	    } else if (BLACK.equals(this)) {
	        return SgfId.ADD_BLACK;
	    } 
	    assert false : "Unknown type: " + this; //$NON-NLS-1$
	    return null;
	}

	/**
	 * ݂̐Fɍ킹Ĕz{@link SgfId#WHITE_TIME_LEFT}{@link SgfId#BLACK_TIME_LEFT}Ԃ܂B
	 */
	public SgfId timeType() {
	    if (WHITE.equals(this)) { 
	        return SgfId.WHITE_TIME_LEFT;
	    } else if (BLACK.equals(this)) {
	        return SgfId.BLACK_TIME_LEFT;
	    } 
	    assert false : "Unknown type: " + this; //$NON-NLS-1$
	    return null;
	}

	/**
	 * ΂̐FԂ܂B
	 */
	public SgfColor opposite() {
	    return this.equals(BLACK) ? WHITE : BLACK;
	}
	
	/**
	 * {@link SgfId#valueType()}{@link ValueType#COLOR}̂ݎ󂯕t܂B
	 * 
	 * @see org.unitarou.sgf.type.TypedString#acceptable(org.unitarou.sgf.SgfId)
	 */
	public boolean acceptable(SgfId sgfId) {
		ArgumentChecker.throwIfNull(sgfId);
		return ValueType.COLOR.equals(sgfId.valueType());
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.TypedString#getString()
	 */
	public String getString() {
		return string_;
	}
	

	/* (non-Javadoc)
	 * @see org.unitarou.lang.NameDisplayable#displayName()
	 */
	public String displayName() {
		return displayName_;
	}
}