/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.type;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.Sgfs;
import org.unitarou.sgf.ValueType;
import org.unitarou.util.ArgumentChecker;

/**
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public class SgfLine implements TypedString {
    static private final Log log_s_ = LogFactory.getLog(SgfLine.class);

    static private final Pattern condition_s_ = Pattern.compile("([a-zA-Z]{2})\\s*:\\s*([a-zA-Z]{2})"); //$NON-NLS-1$
	static private final int START_POS = 1; //K\̃O[vID
	static private final int END_POS = 2; //K\̃O[vID

	static public SgfLine parse(SgfSize size, String string) throws TypeParseException {
		Matcher matched = condition_s_.matcher(string);
		if (!matched.matches()) {
			throw new TypeParseException("Bad argument value = " + string); //$NON-NLS-1$
		}
		
		SgfLine ret = new SgfLine(SgfPoint.parseMove(size, matched.group(START_POS)), 
							SgfPoint.parseMove(size, matched.group(END_POS)));
		return ret;
	}

	/**
	 * {@link TypeParseException}𑗏oȂɁA
	 * p[XɎsꍇnullԂ܂B
	 * 
	 * @param string
	 * @return
	 */
	static public SgfLine parseQuietly(SgfSize size, String string) {
	    try {
	        return parse(size, string);
	    } catch (TypeParseException e) {
	        log_s_.debug("Parse failure: ", e); //$NON-NLS-1$
	        return null;
	    }
	}

	private final SgfPoint start_;
	private final SgfPoint end_;
	/**
	 * 
	 */
	public SgfLine(SgfPoint start, SgfPoint end) {
		super();
		start_ = start;
		end_ = end;
	}


    public SgfPoint getStart() {
        return start_;
    }

    public SgfPoint getEnd() {
        return end_;
    }
	
	/**
	 * {@link SgfId#valueType()}{@link ValueType#CPOINT}̂ݎ󂯕t܂B
	 * 
	 * @see org.unitarou.sgf.type.TypedString#acceptable(org.unitarou.sgf.SgfId)
	 */
	public boolean acceptable(SgfId sgfId) {
		ArgumentChecker.throwIfNull(sgfId);
		return ValueType.CPOINT.equals(sgfId.valueType());
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.TypedString#getString()
	 */
	public String getString() {
		StringBuilder sb = new StringBuilder();
		sb.append(start_.getString());
		sb.append(Sgfs.COMPOSE_SEPARATOR);
		sb.append(end_.getString());
		return sb.toString();
	}

	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.Type#isValid(java.lang.String)
	 */
	public boolean isValid(String value) {
		return (value != null) ? condition_s_.matcher(value).matches() : false;
	}

	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Object o) {
		SgfLine obj = (SgfLine)o;
		return (start_.hashCode() - obj.start_.hashCode()) * SgfSize.MAX_LENGTH * (SgfSize.MAX_LENGTH + 1) 
				+ (end_.hashCode() - obj.end_.hashCode());
	}


	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if ( (obj == null) || (!(obj instanceof SgfLine))) {
			return false;

		} else if (this == obj) {
			return true;

		} else {
			SgfLine o = (SgfLine)obj;
			return (start_.equals(o.start_) && end_.equals(o.end_));			
		}
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		return start_.hashCode() * SgfSize.MAX_LENGTH * (SgfSize.MAX_LENGTH + 1) + end_.hashCode();
	}
}
