/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.type;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.lang.NullArgumentException;
import org.unitarou.lang.Strings;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.Sgfs;
import org.unitarou.sgf.ValueType;
import org.unitarou.sgf.util.SgfPointType;
import org.unitarou.util.ArgumentChecker;

/**
 * _\NXłB<br>
 * ̃NXImmutableFlyweightp^[ŎĂ܂B
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
final public class SgfPoint implements TypedString {
    
    static private final Log log_s_ = LogFactory.getLog(SgfPoint.class);
    
    /**
     * ̃NX̋̏WłB
     */
    static public final SgfPoint[] EMPTY_ARRAY = new SgfPoint[0];
    
    /** őߐڐB㉺EȂ̂łS */
	static public final int MAX_NEIGHBORS = 4;
	
	/** PW"aa"Ƌ""p[Xp^[łB*/
	static private final Pattern condition_s_ = Pattern.compile("|(([a-zA-Z])([a-zA-Z]))"); //$NON-NLS-1$
	static private final int ALL_POS = 0; //K\̃O[vID
	
	/**
	 * {@link #condition_s_}{@link Matcher}ۂXW̒lێO[vIndex
	 */
	static private final int X_POS = 2;
	
	/**
	 * {@link #condition_s_}{@link Matcher}ۂYW̒lێO[vIndex
	 */
	static private final int Y_POS = 3;
	
	
	/** kWp[Xp^[łB*/
	static private final Pattern compressedCondition_s_ 
		= Pattern.compile("\\s*([a-zA-Z])([a-zA-Z])\\s*:\\s*([a-zA-Z])([a-zA-Z])\\s*"); //$NON-NLS-1$

	/** kW̊JnXlO[vłB*/
	static private final int COMP_X1_POS = 1;

	/** kW̊JnYlO[vłB*/
	static private final int COMP_Y1_POS = 2;

	/** kW̏IXlO[vłB*/
	static private final int COMP_X2_POS = 3;

	/** kW̏IYlO[vłB*/
	static private final int COMP_Y2_POS = 4;

	
	/**
	 * @throws org.unitarou.lang.NullArgumentException ̂ǂ炩łnull̏ꍇ 
	 */
	static public boolean isValid(SgfSize size, String datum) {
	    ArgumentChecker.throwIfNull(size, datum);
		
		Matcher matched = condition_s_.matcher(datum);
		if (!matched.matches()) {
			return false;
		}
		
		if (matched.group(ALL_POS).length() == 0) {
			// pXȂǒg̏ꍇ
			return true;
		}
		
		int x = Sgfs.fromSgf(matched.group(X_POS).charAt(0));
		int y = Sgfs.fromSgf(matched.group(Y_POS).charAt(0));
		
		SgfPointType condition = size.check(x, y);
		return (condition.equals(SgfPointType.IN) || condition.equals((SgfPointType.PASS)));
	}

	/**
	 * valueWƂăp[XʂԂ܂B
	 * ł̈valueMove^Cvz肵Ă邽ߍW[aa]̌`݂̂ŁA
	 * [aa:bb]̂悤Ȉk`ɂ͑ΉĂ܂B
	 * k`w肷{@link TypeParseException}𑗏o܂B
	 * 
	 * @throws org.unitarou.lang.NullArgumentException nullꍇB
	 * @throws TypeParseException value̒lWƂĔFłȂꍇ
	 */
	static public SgfPoint parseMove(SgfSize size, String value)
			throws TypeParseException {
	    ArgumentChecker.throwIfNull(size, value);

	    try {
			Matcher matched = condition_s_.matcher(value);
			if (!matched.matches()) {
				throw new TypeParseException("Bad argument value = " + value); //$NON-NLS-1$
			}
			if (matched.group(ALL_POS).length() == 0) {
				return size.getPass();
			}
			int x = Sgfs.fromSgf(matched.group(X_POS).charAt(0));
			int y = Sgfs.fromSgf(matched.group(Y_POS).charAt(0));
			return size.get(x, y);
		
		} catch (IllegalArgumentException e) {
		    throw new TypeParseException(e);
		}
	}
	
	/**
	 * {@link #parseMove(SgfSize, String)}ƓlMove^CvValuep[X܂B
	 * ̃\bhvaluesłĂ{@link TypeParseException}𑗏o܂B
	 * nullԂ܂B
	 * @param size
	 * @param value
	 * @return
	 * @throws org.unitarou.lang.NullArgumentException nullꍇB
	 */
	static public SgfPoint parseMoveQuietly(SgfSize size, String value) {
	    try {
	        return parseMove(size, value);
	    } catch (TypeParseException e) {
	        log_s_.debug("Can't parse value: " + value, e); //$NON-NLS-1$
	        return null;
	    }
	}
	
	/**
	 * valuesׂĂWƂ݂Ȃăp[X܂B
	 * W[aa]̌`[aa:bb]̈k`ɑΉĂ܂B
	 * kWׂ͂ēWJ܂B
	 * z̈ꕔɍWƂ݂ȂȂlꍇ́A̗vf𖳎܂B
	 * 
	 * @throws NullArgumentException nullꍇB
	 */
	static public SgfPoint[] parse(SgfSize size, String[] values) {
	    ArgumentChecker.throwIfNull(size, values);
		
	    List<SgfPoint> ret = new ArrayList<SgfPoint>(values.length); 
	    for (int i = 0; i < values.length; ++i) {
	        try {
	            Matcher matched = compressedCondition_s_.matcher(values[i]);
	            if (!matched.matches()) {
	                ret.add(parseMove(size, values[i]));
	            } else {
	        		int x1 = Sgfs.fromSgf(matched.group(COMP_X1_POS).charAt(0));
	        		int y1 = Sgfs.fromSgf(matched.group(COMP_Y1_POS).charAt(0));
	        		int x2 = Sgfs.fromSgf(matched.group(COMP_X2_POS).charAt(0));
	        		int y2 = Sgfs.fromSgf(matched.group(COMP_Y2_POS).charAt(0));
	        		if (x2 < x1) {
	        		    int tmp = x1;
	        		    x1 = x2;
	        		    x2 = tmp;
	        		}
	        		if (y2 < y1) {
	        		    int tmp = y1;
	        		    y1 = y2;
	        		    y2 = tmp;
	        		}
	        		for (int x = x1; x <= x2; ++x) {
	        		    for (int y = y1; y <= y2; ++y) {
	        		        ret.add(size.get(x, y));
	        		    }
	        		}
	            }
		        
	        } catch(TypeParseException ignore) {
	            // MEMO Oł܂B

	        } catch(IllegalArgumentException ignore) {
	            // MEMO Oł܂B
	        }
	    }
	    return ret.toArray(new SgfPoint[ret.size()]);
	}

	/**
	 * ՃTCYsizeŁAx, ÿʒu_쐬܂B<br>
	 * CX^X̓LbVĂ܂B
	 * 
	 * @throws IllegalArgumentException ʒuՂ̊Oɂꍇ
	 * @throws NullArgumentException sizenull̏ꍇB
	 */
	static public SgfPoint create(SgfSize size, int x, int y) {
	    ArgumentChecker.throwIfNull(size);
		return size.get(x, y);
	}
	
	/**
	 * sizeɍ킹pX̒쐬܂B
	 * size19~19ȉ̏ꍇ́AFF[3]ƌ݊̂pX쐬܂B
	 * 
	 * @throws NullArgumentException sizenull̏ꍇ
	 */
	static public SgfPoint createPass(SgfSize size) {
	    ArgumentChecker.throwIfNull(size);
	    return size.getPass();
	}
	
	
	private final SgfSize size_;
	private final int x_;
	private final int y_;
	
	/**
	 * {@link SgfPointType#IN}A
	 * {@link SgfPointType#OUT}A
	 * {@link SgfPointType#PASS}
	 * ꂩ܂B
	 */	
	private final SgfPointType condition_;
	
	/** ^[Point]łB*/
	private SortedSet<SgfPoint> neighbors_;
	private int hashCode_;
	private String toString_;
	/**
	 * SGF`̃XgOłB
	 */
	private String string_;

	/**
	 * {@link SgfSize}ĂяoRXgN^łB
	 * @param size
	 * @param i
	 * @param j
	 */
	SgfPoint(SgfSize size, int x, int y) {
		super();
		assert size != null;

		condition_ = size.check(x, y);
		size_ = size;
		if (condition_.equals(SgfPointType.IN)) {
			x_ = x;
			y_ = y; 

		} else {
			x_ = 0;
			y_ = 0;
		}
		neighbors_ = null;
		hashCode_ = x_ + y_ * SgfSize.MAX_LENGTH;
		toString_ = null;
		string_ = null;
	}

	public SgfSize size() {
		return size_;
	}
	
	public int x() {
		return x_;
	}
	
	public int y() {
		return y_;
	}
	
	/**
	 * {@link SgfPointType#IN}A
	 * {@link SgfPointType#OUT}A
	 * {@link SgfPointType#PASS}
	 * ꂩԂ܂B
	 */	
	public SgfPointType condition() {
		return condition_;
	}
	
	/** ̃|Cg̋ߐڏWԂ܂iҏWsjB^[{@link SgfPoint}]łB*/
	public SortedSet<SgfPoint> neighbors() {
		if (neighbors_ != null) {
			return neighbors_;
		}

		SortedSet<SgfPoint> set = new TreeSet<SgfPoint>();
		if (1 < x_) {
			set.add(size_.get(x_ - 1, y_));
		}
		if (x_ < size_.width()) {
			set.add(size_.get(x_ + 1, y_));
		}
		if (1 < y_) {
			set.add(size_.get(x_, y_ - 1));
		}
		if (y_ < size_.height()) {
			set.add(size_.get(x_, y_ + 1));
		}
		neighbors_ = Collections.unmodifiableSortedSet(set);
		return neighbors_;
	}
	
	/**
	 * {@link SgfId#valueType()}{@link ValueType#POINT}܂{@link ValueType#MOVE}̂ݎ󂯕t܂B
	 * 
	 * @see org.unitarou.sgf.type.TypedString#acceptable(org.unitarou.sgf.SgfId)
	 */
	public boolean acceptable(SgfId sgfId) {
		ArgumentChecker.throwIfNull(sgfId);
		return ValueType.POINT.equals(sgfId.valueType())
					|| ValueType.MOVE.equals(sgfId.valueType());
	}

	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.TypedString#getString()
	 */
	public String getString() {
		if (string_ != null) {
			return string_;
		}
		if (x_ == 0) { //pX̏ꍇ
			string_ = Strings.EMPTY;
		} else {
			string_ = new String(new char[]{Sgfs.toSgf(x_), Sgfs.toSgf(y_)});
		}
		return string_;
	}

	
	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.Type#compareTo(java.lang.Object)
	 */
	public int compareTo(Object o) {
		SgfPoint obj = (SgfPoint)o;
		int comp = size_.compareTo(obj.size_);
		return (comp * SgfSize.MAX_LENGTH * SgfSize.MAX_LENGTH) 
				+ ((x_ + y_ * SgfSize.MAX_LENGTH) - (obj.x_ + obj.y_ * SgfSize.MAX_LENGTH));
	}


	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
	    if (this == obj) {
			return true;
	    }
		if ( (obj == null) || (!obj.getClass().equals(this.getClass()))) {
			return false;

		} 
		SgfPoint p = (SgfPoint)obj;
		return (size_.equals(p.size_) && (x_ == p.x_) && (y_ == p.y_));			
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		return hashCode_;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		if (toString_ == null) {
			StringBuilder sb = new StringBuilder();
			sb.append('[');
			sb.append(x_);			
			sb.append(',');
			sb.append(y_);			
			sb.append(']');
			sb.append(size_);
			toString_ = sb.toString();
		}
		return toString_;
	}
}