/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.type;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.ValueType;
import org.unitarou.util.ArgumentChecker;

/**
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public class SgfReal implements TypedString {
    static private final Log log_s_ = LogFactory.getLog(SgfReal.class);
    
	static private final Pattern condition_s_ = Pattern.compile("(\\+|-)?\\d+(|\\.\\d+)"); //$NON-NLS-1$
	
	/**
	 * @param value
	 * @return
	 * @throws TypeParseException
	 */
	static public SgfReal parse(String value) throws TypeParseException {
		Matcher matched = condition_s_.matcher(value);
		if (!matched.matches()) {
			throw new TypeParseException("Bad argument value = " + value); //$NON-NLS-1$
		}
		try {
			return new SgfReal(Double.parseDouble(matched.group()));
		} catch (NumberFormatException e) {
			throw new TypeParseException("Bad argument value = " + value); //$NON-NLS-1$
		}
	}

	/**
	 * {@link TypeParseException}𑗏oȂɕsȒlnullԂ܂B
	 * @param value
	 * @return
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
	 */
	static public SgfReal parseQuietly(String value) {
	    try {
	        return parse(value);
	    } catch (TypeParseException e) {
	        log_s_.debug("Bad value for parse: " + value, e); //$NON-NLS-1$
            return null;
	    }
	}

	private final double real_;
	/**
	 * 
	 */
	public SgfReal(double real) {
		super();
		real_ = real;
	}

	public double getReal() {
		return real_;
	}

	/**
	 * {@link SgfId#valueType()}{@link ValueType#REAL}̂ݎ󂯕t܂B
	 * 
	 * @see org.unitarou.sgf.type.TypedString#acceptable(org.unitarou.sgf.SgfId)
	 */
	public boolean acceptable(SgfId sgfId) {
		ArgumentChecker.throwIfNull(sgfId);
		return ValueType.REAL.equals(sgfId.valueType());
	}

	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.TypedString#getString()
	 */
	public String getString() {
		return Double.toString(real_);
	}

	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.Type#isValid(java.lang.String)
	 */
	public boolean isValid(String value) {
		return condition_s_.matcher(value).matches(); 
	}

	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Object o) {
		SgfReal obj = (SgfReal)o;
		return Double.compare(real_, obj.real_);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if ( (obj == null) || (!(obj instanceof SgfReal))) {
			return false;
		}

		SgfReal o = (SgfReal)obj; 
		return Double.doubleToLongBits(real_) == Double.doubleToLongBits(o.real_);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		long bits = Double.doubleToLongBits(real_);
		return (int)(bits ^ (bits >>> 32));
	}

}
