/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.type;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.unitarou.lang.NullArgumentException;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.Sgfs;
import org.unitarou.sgf.util.SgfPointType;
import org.unitarou.util.ArgumentChecker;

/**
 * Ղ̃TCYiF13*13Hj\NXłB
 * ̃NXImmutablełB
 * 
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
final public class SgfSize implements TypedString {
	/** SGF̎dl\\ȃ{[h̍őTCYiTQjłB*/
	static public final int MAX_LENGTH = 52;
	
	/** FF[3]ł̃pXW(20)łB*/
	static public final int OLD_PASS_POS = 20;

	/** {@link #parse(String)}{@link #isValid(String)}ŎgK\p̃p[^łB*/	
	static private final Pattern condition_s_ = Pattern.compile("(\\d{1,2})((\\s*:\\s*)(\\d{1,2}))?"); //$NON-NLS-1$
	static private final int WIDTH_POS = 1; //K\̃O[vID
	static private final int HEIGHT_POS = 4; //K\̃O[vID
	
	static private final int DEFAULT_WIDTH = 19;
	static private final int DEFAULT_HEIGHT = 19;

	/**
	 * CX^XLbVp̃}bvłB
	 * ^[Integer, Size]łB
	 */
	static private final SortedMap<Integer, SgfSize> cache_s_ = new TreeMap<Integer, SgfSize>();

	/** ftHg̔ՃTCYiPXHjłB*/
	static public final SgfSize DEFAULT = create(DEFAULT_WIDTH, DEFAULT_HEIGHT);
	
	/** ő̔ՃTCYiTQHjłB*/
	static public final SgfSize MAX = create(MAX_LENGTH, MAX_LENGTH);
	
	/**
	 * valueSGFPropertyƌȂăp[X܂B
	 * p[XɎs{@link IllegalArgumentException}𑗏o܂B
	 * p[Xł邩mFꍇ́A{@link #isValid(String)}ĂяoĂB
	 * @throws NullArgumentException valuenull̏ꍇB
	 */
	static public SgfSize parse(String value) throws TypeParseException {
	    if (value == null) {
	        throw new NullArgumentException("vaule must not be null."); //$NON-NLS-1$
	    }
		Matcher matched = condition_s_.matcher(value);
		if (!matched.matches()) {
			throw new TypeParseException("Bad argument value = " + value); //$NON-NLS-1$
		}
		int w = Integer.parseInt(matched.group(WIDTH_POS));
		String height = matched.group(HEIGHT_POS);
		int h = (height != null) ? Integer.parseInt(height) : w;
		return create(w, h);
	}
	
	
	/** CX^X̓LbVĂ܂B
	 * @throws IllegalArgumentException elOȉA{@link #MAX_LENGTH}𒴂ꍇB*/
	static public SgfSize create(int width, int height) {
		if ((width < 1) || (MAX_LENGTH < width) || (height < 1) || (MAX_LENGTH < height)) {
			throw new IllegalArgumentException();
		}

		Integer key = new Integer(width + height * MAX_LENGTH);
		synchronized (cache_s_) {
			SgfSize ret = cache_s_.get(key);
			if (ret == null) {
				ret = new SgfSize(width, height);
				cache_s_.put(key, ret);
			}
			return ret;
		}
	}


	private final int width_;
	private final int height_;	
	/**
	 * ̃TCYɊ܂܂SĂ̍Wێ܂B
	 */
	private final SgfPoint[] allPoints_;
	
	/**
	 * {@link #all()}\bhɂ쐬܂B
	 */
	private Set<SgfPoint> all_;
	
	/**
	 * 
	 */
	private final SgfPoint passPoint_;
	
	private Integer primaryKey_;
	private String toString_;

	/**
	 * 
	 */
	private SgfSize(int width, int height) {
		assert((1 <= width) && (width <= MAX_LENGTH) && (1 <= height) && (height <= MAX_LENGTH));
		width_ = width;
		height_ = height;
		allPoints_ = new SgfPoint[width * height];
		all_ = null;
		
	    final int POS = ((height_ < SgfSize.OLD_PASS_POS) && (width_ < SgfSize.OLD_PASS_POS)) 
					? SgfSize.OLD_PASS_POS : 0;  

		passPoint_ = new SgfPoint(this, POS, POS);
		primaryKey_ = null;
		toString_ = null;
	}
	
	/**
	 * Ղ̘̉HԂ܂B
	 * @return
	 */
	public int width() {
		return width_;
	}

	/**
	 * Ղ̏c̘HԂ܂B
	 * @return
	 */
	public int height() {
		return height_;
	}
	
	/**
	 * ̔Ղ̃TCYx,y̍WƂȂ{@link SgfPoint}Ԃ܂B
	 * l͓ŃLbVĂ܂B
	 * ܂lɂĂ̓pXԂƂ܂B
	 * @param x
	 * @param y
	 * @return
	 */
	SgfPoint get(int x, int y) {
		SgfPointType pointType = check(x, y);
		if (pointType == SgfPointType.OUT) {
			throw new IllegalArgumentException(
					"You cannot create a point in outside of board." //$NON-NLS-1$
					+ "[Size:" + this + " x:" + x + " y:" + y + "]"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (pointType == SgfPointType.PASS) {
			return passPoint_;
		}
		int index = (x - 1) + (y - 1) * height_;
		SgfPoint ret = allPoints_[index];
		if (ret == null) {
			ret = new SgfPoint(this, x, y);
			allPoints_[index] = ret;
		}
		return ret;
	}
	
	/**
	 * pX̒Ԃ܂B
	 * @return
	 */
	SgfPoint getPass() {
		return passPoint_;
	}
	
	/**
	 * ̌Ղ̔ՏɋK肳ꂽSvfԂ܂B<br>
	 * 19HȂ[1,1]`[19,19]̑SĂ̗vf(361_)łB<br>
	 * Ȃ߂l͕ҏWsłB
	 * @return
	 */
	public Set<SgfPoint> all() {
		if (all_ == null) {
			createAll();
			all_ = new HashSet<SgfPoint>(Arrays.asList(allPoints_));
		}
		return all_;
	}
	
	private void createAll() {
		for (int i = 0; i < width_; ++i) {
			for (int j = 0; j < height_; ++j) {
				int index = i + j * height_;
				if (allPoints_[index] != null) {
					continue;
				}
				allPoints_[index] = new SgfPoint(this, i + 1, j + 1);
			}
		}
	}
	
	/**
	 * {@link SgfPointType#IN}A
	 * {@link SgfPointType#OUT}A
	 * {@link SgfPointType#PASS}
	 * ꂩԂ܂B
	 */
	public SgfPointType check(int x, int y) {
		if ( (0 < x) && (x <= width_) && (0 < y) && (y <= height_) ) {
			return SgfPointType.IN;
			
		} else if (((x == 0) && (y == 0)) 
					|| ((x == OLD_PASS_POS) && (y == OLD_PASS_POS) 
						&& (width_ < OLD_PASS_POS) && (height_ < OLD_PASS_POS))) {

			return SgfPointType.PASS;

		} else {
			return SgfPointType.OUT;
		}
	}

	/**
	 * {@link SgfId#SIZE}̂ݎ󂯕t܂
	 * @see org.unitarou.sgf.type.TypedString#acceptable(org.unitarou.sgf.SgfId)
	 */
	public boolean acceptable(SgfId sgfId) {
		ArgumentChecker.throwIfNull(sgfId);
		return sgfId.equals(SgfId.SIZE);
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.TypedString#getString()
	 */
	public String getString() {
		if (width_ == height_) {
			return String.valueOf(width_);
		}
		StringBuilder sb = new StringBuilder();
		sb.append(width_);
		sb.append(Sgfs.COMPOSE_SEPARATOR);
		sb.append(height_);
		return sb.toString();
	}


	/* (non-Javadoc)
	 * @see org.unitarou.sgf.type.Type#compareTo(java.lang.Object)
	 */
	public int compareTo(Object o) {
		SgfSize obj = (SgfSize)o;
		return ((width_ + height_ * MAX_LENGTH ) - (obj.width_ + obj.height_ * MAX_LENGTH));
	}


	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;

		} else	if ( (obj == null) || (!(obj instanceof SgfSize))) {
			return false;
		}

		SgfSize size = (SgfSize)obj;
		return ((height_ == size.height_) && (width_ == size.width_));			
	}

	/*
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		return primaryKey();
	}
	
	/** 
	 * uobj1.equals(obj2)  obj1.primaryKey()==obj2.primaryKey()v
	 * 𖞂悤ȃj[NIDԂ܂B
	 */
	public int primaryKey() {
		if (primaryKey_ == null) {
			primaryKey_ = new Integer(height_ * MAX_LENGTH + width_);			
		}
		return primaryKey_.intValue();
	}

	/**
	 * "(,c)"̌`ŕԂ܂B
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		if (toString_ == null) {
			StringBuilder sb = new StringBuilder();
			sb.append('(');
			sb.append(width_);
			sb.append(',');
			sb.append(height_);
			sb.append(')');
			toString_ = sb.toString();
		}
		return toString_;
	}

	/** poinẗʒuɂtrueԂ܂B*/
	public boolean isStarPoint(SgfPoint point) {
		//V̔
		if( ( (width_ % 2) + (height_ % 2) == 2) 
			  && (point.x() == width_ / 2 + 1) && (point.y() == height_ / 2 + 1) ) {
			return true;
		} 

		//ɕ֗Ȃ悤ɁAŐ܂Ԃ
		int X = point.x();
		int Y = point.y();
		if(X > width_ / 2) {
			X = width_ - X + 1;
		} 
		if(Y > height_ / 2) {
			Y = height_ - Y + 1;
		} 

		// \OHȏł́A(4,4)ɐu
		if((width_ >= 13) && (height_ >= 13)) {
		  	if( (X==Y) && (X==4) ) {
		  		return true;
		  	} 
		}

		// \Hȏł́Aӂ̒_ɐu
		if( (width_ >= 19) && (point.x() == width_/ 2 + 1) && (Y == 4) ) {
			return true;
		} 
		
		if( (height_ >= 19) && (point.y() == height_ / 2 + 1) && (X == 4) ) {
			return true;
		} 
		
		// ȊOȂfalse
		return false;
	}
}
