/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.unitarou.lang.NullArgumentException;
import org.unitarou.sgf.Sgfs;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.sgf.type.SgfSize;
import org.unitarou.util.ArgumentChecker;

/**
 *  {@link org.unitarou.sgf.type.SgfPoint}̋`ƂĒ`܂B
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
public class SgfRectangle {
	
	
    /**
     * sgfPointsׂĊ܂ލŏ̋`쐬ĕԂ܂B
     * @throws IllegalArgumentException null̏ꍇ
     */
	static public SgfRectangle create(Collection<SgfPoint> sgfPoints) {
        // x1,y1,x2,y2ƂĈĂ܂B
        SgfRectangle redrawArea = new SgfRectangle();
        for (SgfPoint sgfPoint : sgfPoints) {
        	if (sgfPoint.condition().equals(SgfPointType.IN)) {
                redrawArea.intersect(sgfPoint);
        	}
        }
        return redrawArea;
	}

	/**
     * sgfPointsׂĊ܂ލŏ̋`쐬ĕԂ܂B
     * @throws IllegalArgumentException null̏ꍇ
     */
	static public SgfRectangle create(SgfPoint[] sgfPoints) {
        // x1,y1,x2,y2ƂĈĂ܂B
        SgfRectangle redrawArea = new SgfRectangle();
        for (SgfPoint sgfPoint : sgfPoints) {
        	if (sgfPoint.condition().equals(SgfPointType.IN)) {
                redrawArea.intersect(sgfPoint);
        	}
        }
        return redrawArea;
	}
	
    /**
     * sgfPointsɂāAKɋ`̈؂oĕԂ܂B<br>
     * {@link SgfPointType#PASS}͖܂B
     * @throws IllegalArgumentException null̏ꍇ
     */
	static public SgfRectangle[] createRegularRectangles(SgfPoint[] sgfPoints) {
        List<SgfRectangle> list = new ArrayList<SgfRectangle>();
        Set<SgfPoint> workSet = new TreeSet<SgfPoint>(Arrays.asList(sgfPoints));
        while (!workSet.isEmpty()) {
            SgfPoint element = workSet.iterator().next();
            if (!element.condition().equals(SgfPointType.IN)) {
            	workSet.remove(element);
            	continue;
            }
            SgfRectangle rectangle = findBestRectangle(workSet, element);
            list.add(rectangle);
            for (int x = rectangle.getStart().x(); x <= rectangle.getEnd().x(); ++x) {
                for (int y = rectangle.getStart().y(); y <= rectangle.getEnd().y(); ++y) {
                    workSet.remove(SgfPoint.create(element.size(), x, y));
                }
            }
        }
        return list.toArray(new SgfRectangle[list.size()]);
	}
	
    /**
     * start珇ɃT[`end܂őSĂɒl݂ő̋`Ԃ܂B
     */
    static private SgfRectangle findBestRectangle(Set<SgfPoint> pointSet, SgfPoint start) {
        // ߂ɍsƗōől𒲂ׂ
        int endX = start.x();
        for (int x = start.x() + 1; x <= start.size().width(); ++x) {
            SgfPoint point = SgfPoint.create(start.size(), x, start.y());
            if (!pointSet.contains(point)) {
                break;
            }
            endX = x;
        }
        
        int endY = start.y();
        for (int y = start.y() + 1; y <= start.size().height(); ++y) {
            SgfPoint point = SgfPoint.create(start.size(), start.x(), y);
            if (!pointSet.contains(point)) {
                break;
            }
            endY = y;
        }
        
        // s̔zAg͖ʐ(Ps̋`̖ʐρAQs̋`̖ʐρc)
        int[] area = new int[endY - start.y() + 1];
        area[0] = endX - start.x() + 1;
        
        // Psł̍őAQsł̍őARsł̋`ɂłől𒲂ׂ
        for (int y = start.y() + 1; y <= endY; ++y) {
            int lastX = start.x();
            for (int x = start.x() + 1; x <= endX; ++x) {
                SgfPoint point = SgfPoint.create(start.size(), x, y);
                if (!pointSet.contains(point)) {
                    endX = lastX;
                    break;
                }
                lastX = x;
            }
            // `̖ʐς߂
            area[y - start.y()] = (y - start.y() + 1) * (endX - start.x() + 1);
        }
        
        // ʐύő̍sT
        int maxRow = 0;
        int maxArea = area[0];
        for (int i = 1; i < area.length; ++i) {
            if (maxArea < area[i]) {
                maxArea = area[i];
                maxRow = i;
            }
        }
        ++maxRow; // PIWɏC
        
        // ʐς񐔂o
        int maxCol = maxArea / maxRow;
        
        // sƗ񐔂`쐬B
        SgfRectangle ret = new SgfRectangle();
        ret.set(start, SgfPoint.create(start.size(), start.x() + maxCol - 1, start.y() + maxRow - 1));
        return ret;
    }

	
    private SgfPoint start_;
    private SgfPoint end_;
    
    /**
     * Ԃłend < start̊֌WɂȂĂ܂B
     *
     */
    public SgfRectangle() {
        super();
        SgfSize size = SgfSize.create(SgfSize.MAX_LENGTH, SgfSize.MAX_LENGTH);
        start_ = SgfPoint.create(size, size.width(), size.height());
        end_ = SgfPoint.create(size, 1, 1);
    }
    
    /**
     * @param start
     * @param end
     * @throws NullArgumentException <code>null</code>̏ꍇ
     */
    public void set(SgfPoint start, SgfPoint end) {
    	ArgumentChecker.throwIfNull(start, end);
    	if (!start.size().equals(end.size())) {
    		throw new IllegalArgumentException(
    				"Inconsistent size: start:" + start.size()  //$NON-NLS-1$
    				+ ", end: " + end.size()); //$NON-NLS-1$
    	}
    	
        start_ = start;
        end_ = end;
    }
    
    
    /**
     * point̋`Ɍ,V`\܂B
     * ܂Apoint݂̋`̓ɂꍇ͉A
     * `̊Oɂꍇpointƌ݂̋`܂ލŏ̋`\܂B
     * 
     * @throws NullArgumentException null̏ꍇ
     */
    public void intersect(SgfPoint point) {
    	ArgumentChecker.throwIfNull(point);
        
        int x1 = Math.min(start_.x(), point.x());
        int y1 = Math.min(start_.y(), point.y());
        int x2 = Math.min(Math.max(end_.x(), point.x()), point.size().width());
        int y2 = Math.min(Math.max(end_.y(), point.y()), point.size().height());
        
        start_ = SgfPoint.create(point.size(), x1, y1);
        end_   = SgfPoint.create(point.size(), x2, y2);
    }
    
    /**
     * rectangleƂ̐ϏWɂV`\܂B
     * ܂A݂̋`rectangle̋ʂ镔̋`ɂȂ܂B
     * @param rectangle
     */
    public void union(SgfRectangle rectangle) {
    	ArgumentChecker.throwIfNull(rectangle);

        int x1 = Math.min(Math.max(start_.x(), rectangle.start_.x()),start_.size().width());
        int y1 = Math.min(Math.max(start_.y(), rectangle.start_.y()),start_.size().height());
        int x2 = Math.min(end_.x(), rectangle.end_.x());
        int y2 = Math.min(end_.y(), rectangle.end_.y());
        
        start_ = SgfPoint.create(start_.size(), x1, y1);
        end_   = SgfPoint.create(start_.size(), x2, y2);
    }

    public SgfPoint getStart() {
        return start_;
    }
    
    public SgfPoint getEnd() {
        return end_;
    }
    
    public int width() {
        return end_.x() - start_.x() + 1;
    }

    public int height() {
        return end_.y() - start_.y() + 1;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
	public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        
        if ((obj == null) || !obj.getClass().equals(this.getClass())) {
            return false;
        }
        SgfRectangle pr = (SgfRectangle)obj;
        return start_.equals(pr.start_) && end_.equals(pr.end_);
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
	public int hashCode() {
        return start_.hashCode() * 7 + end_.hashCode();
    }

    /**
     * SGF`̍W'aa:bb'Ԃ܂B
     * @return
     */
    public String getValue() {
        if ( (end_.x() < start_.x())
                && (end_.y() < start_.y())) {
            throw new IllegalArgumentException();
        }
        if (start_.equals(end_)) {
            return start_.getString();
        }
        return start_.getString() + Sgfs.COMPOSE_SEPARATOR + end_.getString();
    }
    
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
	public String toString() {
        if (start_.equals(end_)) {
            return start_.getString();
        }
        return start_.getString() + Sgfs.COMPOSE_SEPARATOR + end_.getString();
    }
}
