/*  
 * Copyright 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.sgf.util.filter;

import java.util.EnumSet;

import org.unitarou.lang.Strings;
import org.unitarou.ml.MessageResource;
import org.unitarou.sgf.Collection;
import org.unitarou.sgf.GameTree;
import org.unitarou.sgf.Node;
import org.unitarou.sgf.Property;
import org.unitarou.sgf.RootGameTree;
import org.unitarou.sgf.Value;
import org.unitarou.sgf.ValueType;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.util.Filter;
import org.unitarou.util.FilterException;
import org.unitarou.util.NestedFilter;

/**
 * @author unitarou &lt;boss@unitarou.org&gt; 
 */
public class InsertLineSeparatorFilter extends NestedFilter<Collection> {
	/**
	 * tB^[̃IvVK肷enumłB
	 */
	public enum Option {
		/** Textsoft line breakgƂӖ܂B */
		USE_SLB_FOR_TEXT,
		
		/** Simple Textsoft line breakgƂӖ܂B */ 
		USE_SLB_FOR_SIMPLETEXT;
		
		/* (non-Javadoc)
		 * @see java.lang.Object#toString()
		 */
		@Override
		public String toString() {
			return new MessageResource(this.getClass(), name()).get();
		}
	}
	
	private int maxLine_;

	private int currentLine_;
	
	private final EnumSet<Option> options_;
	/**
	 * 
	 */
	public InsertLineSeparatorFilter() {
		this(null);
	}

	/**
	 * 
	 */
	public InsertLineSeparatorFilter(int maxLine) {
		this(null, maxLine);
	}

	/**
	 * @param parent
	 */
	public InsertLineSeparatorFilter(Filter<Collection> parent) {
		this(parent, 80);
	}

	/**
	 * WłSoft-linebreak͎g܂B
	 * @param parent
	 */
	public InsertLineSeparatorFilter(Filter<Collection> parent, int maxLine) {
		super(parent);
		ArgumentChecker.throwIfNegative(maxLine);
		options_ = EnumSet.noneOf(Option.class);
		maxLine_ = maxLine;
		currentLine_ = 0;
	}
	
	/**
	 * @param options
	 */
	public void setOption(EnumSet<Option> options) {
		ArgumentChecker.throwIfNull(options);
		options_.clear();
		options_.addAll(options);
	}
	
	/**
	 * @return
	 */
	public EnumSet<Option> getOptions() {
		return options_.clone();
	}


	/* (non-Javadoc)
	 * @see org.unitarou.util.NestedFilter#filter(T)
	 */
	@Override
	public Collection filter(Collection collection) throws FilterException {
		collection = super.filter(collection);
		Collection ret = new Collection();
		for (RootGameTree rgt : collection) {
			ret.addLast(filterRootGameTreeFrame(rgt));
		}
		return ret;
	}
	
	/**
	 * @param rgt
	 * @return
	 */
	private RootGameTree filterRootGameTreeFrame(RootGameTree rgt) {
		RootGameTree ret = new RootGameTree(rgt);
		ret.setOpenUtr(insertLineSeparatorForUtr(ret.getOpenUtr()));
		++currentLine_;
		ret.getSequence().clear();
		for (Node node : rgt.getSequence()) {
			ret.getSequence().addLast(filterNodeFrame(node));
		}
		
		ret.clearGameTree();
		for (GameTree child : rgt.getChildren()) {
			ret.addGameTree(filterGameTreeFrame(child));
		}
		++currentLine_;
		ret.setCloseUtr(insertLineSeparatorForUtr(ret.getCloseUtr()));
		return ret;
	}
	
	/**
	 * @param rgt
	 */
	private GameTree filterGameTreeFrame(GameTree gameTree) {
		GameTree ret = new GameTree(gameTree);
		ret.setOpenUtr(insertLineSeparatorForUtr(ret.getOpenUtr()));
		++currentLine_;
		ret.getSequence().clear();
		for (Node node : gameTree.getSequence()) {
			ret.getSequence().addLast(filterNodeFrame(node));
		}
		
		ret.clearGameTree();
		for (GameTree child : gameTree.getChildren()) {
			ret.addGameTree(filterGameTreeFrame(child));
		}
		++currentLine_;
		ret.setCloseUtr(insertLineSeparatorForUtr(ret.getCloseUtr()));
		return ret;
	}

	/**
	 * @param node
	 * @return
	 */
	private Node filterNodeFrame(Node node) {
		Node ret = new Node(node);
		ret.setOpenUtr(insertLineSeparatorForUtr(ret.getOpenUtr()));
		++currentLine_;
		ret.clearProperty();
		for (Property property : node.getProperties()) {
			property = filterPropertyFrame(property);
			if (property.isValid()) {
				ret.addProperty(property);
			}
		}
		ret.setCloseUtr(insertLineSeparatorForUtr(ret.getCloseUtr()));
		return ret;
	}

	/**
	 * @param property
	 * @return
	 */
	private Property filterPropertyFrame(Property property) {
		Property ret = new Property(property);
		ret.setOpenUtr(insertLineSeparatorForUtr(ret.getOpenUtr()));
		updateString(ret.getEntireId());
		ret.clearValue();
		Value[] values = property.getValues();
		boolean lbString = usesSoftLineBreak(ret.sgfId().valueType());
		for (int i = 0; i < values.length; ++i){
			ret.addValue(filterValue(values[i], (i!=0), lbString));
		}
		ret.setCloseUtr(insertLineSeparatorForUtr(ret.getCloseUtr()));
		return ret;
	}
	
	private boolean usesSoftLineBreak(ValueType valueType) {
		return  valueType.equals(ValueType.TEXT) && options_.contains(Option.USE_SLB_FOR_TEXT)
				|| valueType.equals(ValueType.SIMPLE_TEXT) && options_.contains(Option.USE_SLB_FOR_SIMPLETEXT);
	}
	
	/**
	 */
	/**
	 * @param value
	 * @param lbInUtr UTRŉsǉꍇtrueݒ肵܂B
	 * @param lbString l{̂̒soft-linebreakǉꍇtrueݒ肵܂B
	 * @return
	 */
	private Value filterValue(Value value, boolean lbInUtr, boolean lbString) {
		String string = value.getString();
		String utr = value.getOpenUtr();
		if (lbInUtr) {
			utr = insertLineSeparatorForUtr(utr);
		} else {
			updateString(utr);
		}
		++currentLine_;
		if (lbString) {
			string = insertLineSeparator(string, "\\" + Strings.LINE_SEPARATOR); //$NON-NLS-1$
		} else {
			updateString(string);
		}
		++currentLine_;
		return new Value(string, utr);
	}
	


	private void updateString(String string) {
		for (int i = 0; i < string.length(); ++i) {
			char c = string.charAt(i);
			switch (c) {
			case '\r':
			case '\n':
				currentLine_ = 0;
				break;
			default:
				currentLine_ += (c < 0xFF) ? 1 : 2;
			}
		}
	}
	
	/**
	 * string̕JEg{@link #maxLine_}
	 * ĂA̒{@link Strings#LINE_SEPARATOR}
	 * }܂B\r\n{@link #currentLine_}
	 * 0ɐݒ肵܂B
	 * @param string
	 * @return
	 */
	private String insertLineSeparatorForUtr(String string) {
		return insertLineSeparator(string, Strings.LINE_SEPARATOR);
	}

	/**
	 * string̕JEg{@link #maxLine_}
	 * ĂA̒lineSeparator
	 * }܂B\r\n{@link #currentLine_}
	 * 0ɐݒ肵܂B
	 * @param string
	 * @return
	 */
	private String insertLineSeparator(String string, String lineSeparator) {
		// \zsR[hǉTCYō쐬
		StringBuilder builder = new StringBuilder(
				string.length() + ((currentLine_ + string.length()) / maxLine_) * lineSeparator.length());
		if (maxLine_ <= currentLine_) {
			builder.append(lineSeparator);
			currentLine_ = 0;			
		}
		for (int i = 0; i < string.length(); ++i) {
			char c = string.charAt(i);
			builder.append(c);
			switch (c) {
			case '\r':
			case '\n':
				currentLine_ = 0; // ɑ
			default:
				if (currentLine_ < maxLine_) {
					currentLine_ += (c < 0xFF) ? 1 : 2;
				} else {
					builder.append(lineSeparator);
					currentLine_ = 0;
				}
			}
		}
		return builder.toString();
	}
}
