/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     unitarou - update for Tiger (Java 5)
 *******************************************************************************/
package org.unitarou.swt;

import java.lang.reflect.Array;

import org.eclipse.jface.util.Assert;

import org.unitarou.util.ArgumentChecker;

/**
 * This class is used to maintain a list of listeners, and
 * is used in the implementations of several classes within JFace
 * which allow you to register listeners of various kinds.
 * It is a fairly lightweight object, occupying minimal space when
 * no listeners are registered.
 * <p>
 * Note that the <code>add</code> method checks for and eliminates 
 * duplicates based on identity (not equality).  Likewise, the
 * <code>remove</code> method compares based on identity.
 * </p>
 * <p>
 * Use the <code>getListeners</code> method when notifying listeners.
 * Note that no garbage is created if no listeners are registered.
 * The recommended code sequence for notifying all registered listeners
 * of say, <code>FooListener.eventHappened</code>, is:
 * <pre>
 * Object[] listeners = myListenerList.getListeners();
 * for (int i = 0; i < listeners.length; ++i) {
 *    ((FooListener) listeners[i]).eventHappened(event);
 * }
 * </pre>
 * </p>
 */
public class ListenerList<E> {
	
	private final Class<E> clazz_;
    /**
     * The initial capacity_ of the list. Always >= 1.
     */
    private int capacity_;

    /**
     * The current number of listeners.
     * Maintains invariant: 0 <= size_ <= listeners.length.
     */
    private int size_;

    /**
     * The list of listeners.  Initially <code>null</code> but initialized
     * to an array of size_ capacity_ the first time a listener is added.
     * Maintains invariant: listeners != null IFF size_ != 0
     */
    private E[] listeners = null;

    /**
     * Creates a listener list with an initial capacity_ of 1.
     */
    public ListenerList(Class<E> clazz) {
        this(clazz, 1);
    }

    /**
     * Creates a listener list with the given initial capacity_.
     *
     * @param capacity_ the number of listeners which this list can initially accept 
     *    without growing its internal representation; must be at least 1
     */
    public ListenerList(Class<E> clazz, int capacity) {
    	ArgumentChecker.throwIfNull(clazz);
        Assert.isTrue(capacity >= 1);
        capacity_ = capacity;
        clazz_ = clazz;
    }

    /**
     * Adds the given listener to this list. Has no effect if an identical listener
     * is already registered.
     *
     * @param listener the listener
     */
    @SuppressWarnings("unchecked") //$NON-NLS-1$
	public void add(E listener) {
        Assert.isNotNull(listener);
        if (size_ == 0) {
            listeners = (E[])Array.newInstance(listener.getClass(), capacity_);
        } else {
            // check for duplicates using identity
            for (int i = 0; i < size_; ++i) {
                if (listeners[i] == listener) {
                    return;
                }
            }
            // grow array if necessary
            if (size_ == listeners.length) {
                System.arraycopy(listeners, 0,
                        listeners = (E[])Array.newInstance(listener.getClass(), size_ * 2 + 1), 0, size_);
            }
        }

        listeners[size_] = listener;
        size_++;
    }

    /**
     * Removes all listeners from this list.
     */
    public void clear() {
        size_ = 0;
        listeners = null;
    }

    /**
     * Returns an array containing all the registered listeners,
     * in the order in which they were added.
     * <p>
     * The resulting array is unaffected by subsequent adds or removes.
     * If there are no listeners registered, the result is an empty array
     * singleton instance (no garbage is created).
     * Use this method when notifying listeners, so that any modifications
     * to the listener list during the notification will have no effect on the
     * notification itself.
     * </p>
     *
     * @return the list of registered listeners
     */
    @SuppressWarnings("unchecked") //$NON-NLS-1$
	public E[] getListeners() {
        if (size_ == 0) {
            return (E[])Array.newInstance(clazz_, 0);
        }
        
        E[] result = (E[])Array.newInstance(listeners.getClass().getComponentType(), size_);
        System.arraycopy(listeners, 0, result, 0, size_);
        return result;
    }

    /**
     * Returns whether this listener list is empty.
     *
     * @return <code>true</code> if there are no registered listeners, and
     *   <code>false</code> otherwise
     */
    public boolean isEmpty() {
        return size_ == 0;
    }

    /**
     * Removes the given listener from this list. Has no effect if an identical
     * listener was not already registered.
     *
     * @param listener the listener
     */
    public void remove(Object listener) {
        Assert.isNotNull(listener);
        for (int i = 0; i < size_; ++i) {
            if (listeners[i] == listener) {
                if (size_ == 1) {
                    listeners = null;
                    size_ = 0;
                } else {
                    System
                            .arraycopy(listeners, i + 1, listeners, i, --size_
                                    - i);
                    listeners[size_] = null;
                }
                return;
            }
        }
    }

    /**
     * Returns the number of registered listeners.
     *
     * @return the number of registered listeners
     */
    public int size() {
        return size_;
    }
}
