/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation, unitarou <boss@unitarou.org> and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     unitarou - refactoring(reduce field scope, divide long line method, etc...)
 *                customize VCL like(set minimum size, some part fix at resize, 
 *                decorate splitter, etc...)
 *******************************************************************************/

package org.unitarou.swt;

import org.eclipse.swt.SWT;
import org.eclipse.swt.SWTException;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Sash;

/**
 * The SashForm lays out its children in a Row or Column
 * arrangement (as specified by the orientation) and places
 * a Sash between the children. One child may be maximized
 * to occupy the entire size of the SashForm. The relative
 * sizes of the children may be specfied using weights.
 * <p>
 * <dl>
 * <dt><b>Styles:</b>
 * <dd>HORIZONTAL, VERTICAL, SMOOTH
 * </dl>
 */
public class USashForm extends Composite {
	
	/**
	 * Ԏd؂̕(contentŝ)sNZPʂŕێ܂B
	 */
	private int sashWidth_ = 3;

	/**
	 * ̃CX^XSWTł̃X^Cێ܂B
	 */
	private int sashStyle_;

	/**
	 * ̃CX^X̊Ԏd؂(Sash)zŕێ܂B
	 */
	private Sash[] sashes_ = new Sash[0];

	// Remember background_ and foreground_
	// colors to determine whether to set
	// sashes_ to the default color (null) or
	// a specific color
	private Color background_ = null;

	private Color foreground_ = null;

	/**
	 * ݕ\Ă(Sash)vfzŕێ܂B
	 */
	private Control[] controls_ = new Control[0];

	/**
	 * ݍő剻(̗vfSďĂ)ĂCX^XԂ܂B
	 * ő剻ԈȊOł <code>null</code>ɂȂ܂B
	 */
	private Control maxControl_ = null;

	private final Listener sashListener_;

	/**
	 * e{@link Sash}CX^X`悷郊Xi[łB
	 */
	private final PaintListener sashPaintaListener_;


	/**
	 * Constructs a new instance of this class given its
	 * parent and a style value describing its behavior and
	 * appearance.
	 * <p>
	 * The style value is either one of the style constants
	 * defined in class <code>SWT</code> which is
	 * applicable to instances of this class, or must be
	 * built by <em>bitwise OR</em>'ing together (that
	 * is, using the <code>int</code> "|" operator) two or
	 * more of those <code>SWT</code> style constants. The
	 * class description lists the style constants that are
	 * applicable to the class. Style bits are also
	 * inherited from superclasses.
	 * </p>
	 * 
	 * @param parent
	 *        a widget which will be the parent of the new
	 *        instance (cannot be null)
	 * @param style
	 *        the style of widget to construct
	 * @exception IllegalArgumentException
	 *            <ul>
	 *            <li>ERROR_NULL_ARGUMENT - if the parent
	 *            is null</li>
	 *            </ul>
	 * @exception SWTException
	 *            <ul>
	 *            <li>ERROR_THREAD_INVALID_ACCESS - if not
	 *            called from the thread that created the
	 *            parent</li>
	 *            </ul>
	 * @see SWT#HORIZONTAL
	 * @see SWT#VERTICAL
	 * @see #getStyle()
	 */
	public USashForm(Composite parent, int style) {
		super(parent, checkStyle(style));
		super.setLayout(new USashFormLayout());
		sashStyle_ = ((style & SWT.VERTICAL) != 0) ? SWT.HORIZONTAL : SWT.VERTICAL;
		if ((style & SWT.BORDER) != 0)
			sashStyle_ |= SWT.BORDER;
		if ((style & SWT.SMOOTH) != 0)
			sashStyle_ |= SWT.SMOOTH;
		sashListener_ = new Listener() {
			public void handleEvent(Event e) {
				onDragSash(e);
			}
		};
		sashPaintaListener_ = new PaintListener() {
			public void paintControl(PaintEvent e) {
				Sash sash = (Sash)e.widget;
				Color colBar1 = sash.getDisplay().getSystemColor(SWT.COLOR_WIDGET_HIGHLIGHT_SHADOW);
				Color colBar2 = sash.getDisplay().getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW);
				Point p = sash.getSize();
				if (0 != (sash.getStyle() & SWT.HORIZONTAL)) {
					int y1 = p.y / 2;
					int y2 = y1 + 1;
					e.gc.setForeground(colBar1);
					e.gc.drawLine(0, y1, p.x, y1);
					e.gc.setForeground(colBar2);
					e.gc.drawLine(0, y2, p.x, y2);
					
				} else {
					int x1 = p.x / 2;
					int x2 = x1 + 1;
					e.gc.setForeground(colBar1);
					e.gc.drawLine(x1 , 0, x1, p.y);
					e.gc.setForeground(colBar2);
					e.gc.drawLine(x2 , 0, x2, p.y);
				}
			}
		};
	}

	static private int checkStyle(int style) {
		int mask = SWT.BORDER | SWT.LEFT_TO_RIGHT | SWT.RIGHT_TO_LEFT;
		return style & mask;
	}

	/**
	 * Returns SWT.HORIZONTAL if the controls_ in the
	 * SashForm are laid out side by side or SWT.VERTICAL if
	 * the controls_ in the SashForm are laid out top to
	 * bottom.
	 * 
	 * @return SWT.HORIZONTAL or SWT.VERTICAL
	 */
	public int getOrientation() {
		// checkWidget();
		return (sashStyle_ & SWT.VERTICAL) != 0 ? SWT.HORIZONTAL : SWT.VERTICAL;
	}

	@Override
	public int getStyle() {
		int style = super.getStyle();
		style |= getOrientation() == SWT.VERTICAL ? SWT.VERTICAL : SWT.HORIZONTAL;
		if ((sashStyle_ & SWT.SMOOTH) != 0)
			style |= SWT.SMOOTH;
		return style;
	}

	/**
	 * Answer the control that currently is maximized in the
	 * SashForm. This value may be null.
	 * 
	 * @return the control that currently is maximized or
	 *         null
	 */
	public Control getMaximizedControl() {
		// checkWidget();
		return this.maxControl_;
	}

	/**
	 * Answer the relative weight of each child in the
	 * SashForm. The weight represents the percent of the
	 * total width (if SashForm has Horizontal orientation)
	 * or total height (if SashForm has Vertical
	 * orientation) each control occupies. The weights are
	 * returned in order of the creation of the widgets
	 * (weight[0] corresponds to the weight of the first
	 * child created).
	 * 
	 * @return the relative weight of each child
	 * @exception SWTException
	 *            <ul>
	 *            <li>ERROR_WIDGET_DISPOSED - if the
	 *            receiver has been disposed</li>
	 *            <li>ERROR_THREAD_INVALID_ACCESS - if not
	 *            called from the thread that created the
	 *            receiver</li>
	 *            </ul>
	 */

	public int[] getWeights() {
		checkWidget();
		Control[] cArray = getControls(false);
		int[] ratios = new int[cArray.length];
		for (int i = 0; i < cArray.length; i++) {
			Object data = cArray[i].getLayoutData();
			if (data != null && data instanceof USashFormData) {
				ratios[i] = (int)(((USashFormData)data).weight * 1000 >> 16);
			} else {
				ratios[i] = 200;
			}
		}
		return ratios;
	}

	/**
	 * ݂̎qvf̒SashCX^XȊO
	 * Visible==true(onlyVisible==truȅꍇ)̗vf̔zԂ܂B
	 * 
	 * @param onlyVisible
	 * @return
	 */
	Control[] getControls(boolean onlyVisible) {
		Control[] children = getChildren();
		Control[] result = new Control[0];
		for (int i = 0; i < children.length; i++) {
			if (children[i] instanceof Sash)
				continue;
			if (onlyVisible && !children[i].getVisible())
				continue;

			Control[] newResult = new Control[result.length + 1];
			System.arraycopy(result, 0, newResult, 0, result.length);
			newResult[result.length] = children[i];
			result = newResult;
		}
		return result;
	}

	void onDragSash(Event event) {
		Sash sash = (Sash)event.widget;
		int sashIndex = -1;
		for (int i = 0; i < sashes_.length; i++) {
			if (sashes_[i] == sash) {
				sashIndex = i;
				break;
			}
		}
		if (sashIndex == -1)
			return;

		Control c1 = controls_[sashIndex];
		Control c2 = controls_[sashIndex + 1];
		checkAndCreateLayoutData(c1);
		checkAndCreateLayoutData(c2);
		Rectangle b1 = c1.getBounds();
		Rectangle b2 = c2.getBounds();
		USashFormData data1 = (USashFormData)c1.getLayoutData();
		USashFormData data2 = (USashFormData)c2.getLayoutData();

		Rectangle sashBounds = sash.getBounds();
		Rectangle area = getClientArea();
		boolean correction = false;
		if (getOrientation() == SWT.HORIZONTAL) {
			correction = b1.width < data1.getDragMinimum() || b2.width < data2.getDragMinimum();
			int totalWidth = b2.x + b2.width - b1.x;
			int shift = event.x - sashBounds.x;
			b1.width += shift;
			b2.x += shift;
			b2.width -= shift;
			if (b1.width < data1.getDragMinimum()) {
				b1.width = data1.getDragMinimum();
				b2.x = b1.x + b1.width + sashBounds.width;
				b2.width = totalWidth - b2.x;
				event.x = b1.x + b1.width;
				event.doit = false;
			}
			if (b2.width < data2.getDragMinimum()) {
				b1.width = totalWidth - data2.getDragMinimum() - sashBounds.width;
				b2.x = b1.x + b1.width + sashBounds.width;
				b2.width = data2.getDragMinimum();
				event.x = b1.x + b1.width;
				event.doit = false;
			}
			updateHorizontalLayoutData(area, c1, b1);
			updateHorizontalLayoutData(area, c2, b2);

		} else {
			correction = b1.height < data1.getDragMinimum() || b2.height < data2.getDragMinimum();
			int totalHeight = b2.y + b2.height - b1.y;
			int shift = event.y - sashBounds.y;
			b1.height += shift;
			b2.y += shift;
			b2.height -= shift;
			if (b1.height < data1.getDragMinimum()) {
				b1.height = data1.getDragMinimum();
				b2.y = b1.y + b1.height + sashBounds.height;
				b2.height = totalHeight - b2.y;
				event.y = b1.y + b1.height;
				event.doit = false;
			}
			if (b2.height < data2.getDragMinimum()) {
				b1.height = totalHeight - data2.getDragMinimum() - sashBounds.height;
				b2.y = b1.y + b1.height + sashBounds.height;
				b2.height = data2.getDragMinimum();
				event.y = b1.y + b1.height;
				event.doit = false;
			}
			updateVerticalLayoutData(area, c1, b1);
			updateVerticalLayoutData(area, c2, b2);	
		}
		if (correction || (event.doit && event.detail != SWT.DRAG)) {
			c1.setBounds(b1);
			sash.setBounds(event.x, event.y, event.width, event.height);
			c2.setBounds(b2);
		}
	}
	
	/**
	 * control{@link Control#getLayoutData()} <code>null</code>
	 * {@link USashFormData}ɃLXgłȂɁAVɍ쐬ď㏑܂B
	 * @param control
	 */
	private void checkAndCreateLayoutData(Control control) {
		Object data1 = control.getLayoutData();
		if (data1 == null || !(data1 instanceof USashFormData)) {
			control.setLayoutData(new USashFormData());
		}		
	}

	
	private void updateHorizontalLayoutData(
			Rectangle area, Control control, Rectangle bound) 
	{
		USashFormData data1 = (USashFormData)control.getLayoutData();
		if (0 < data1.getActualSize()) {
			data1.setActualSize(bound.width);
		} else {
			data1.weight = (((long)bound.width << 16) + area.width - 1) / area.width;
		}
	}
	
	private void updateVerticalLayoutData(
			Rectangle area, Control control, Rectangle bound) 
	{
		USashFormData data1 = (USashFormData)control.getLayoutData();
		if (0 < data1.getActualSize()) {
			data1.setActualSize(bound.height);
		} else {
			data1.weight = (((long)bound.height << 16) + area.height - 1) / area.height;
		}
	}
	

	/**
	 * If orientation is SWT.HORIZONTAL, lay the controls_ in
	 * the SashForm out side by side. If orientation is
	 * SWT.VERTICAL, lay the controls_ in the SashForm out
	 * top to bottom.
	 * 
	 * @param orientation
	 *        SWT.HORIZONTAL or SWT.VERTICAL
	 * @exception SWTException
	 *            <ul>
	 *            <li>ERROR_WIDGET_DISPOSED - if the
	 *            receiver has been disposed</li>
	 *            <li>ERROR_THREAD_INVALID_ACCESS - if not
	 *            called from the thread that created the
	 *            receiver</li>
	 *            <li>ERROR_INVALID_ARGUMENT - if the value
	 *            of orientation is not SWT.HORIZONTAL or
	 *            SWT.VERTICAL
	 *            </ul>
	 */
	public void setOrientation(int orientation) {
		checkWidget();
		if (getOrientation() == orientation)
			return;
		if (orientation != SWT.HORIZONTAL && orientation != SWT.VERTICAL) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}
		sashStyle_ &= ~(SWT.HORIZONTAL | SWT.VERTICAL);
		sashStyle_ |= orientation == SWT.VERTICAL ? SWT.HORIZONTAL : SWT.VERTICAL;
		for (int i = 0; i < sashes_.length; i++) {
			sashes_[i].dispose();
			sashes_[i] = createSash();
		}
		layout(false);
	}
	
	/**
	 * w肳ꂽ{@link Sash}쐬ĕԂ܂B
	 * @return
	 */
	private Sash createSash() {
		Sash ret = new Sash(this, sashStyle_);
		ret.setBackground(background_);
		ret.setForeground(foreground_);
		ret.addListener(SWT.Selection, sashListener_);
		ret.addPaintListener(sashPaintaListener_);
		return ret;
	}

	@Override
	public void setBackground(Color color) {
		super.setBackground(color);
		background_ = color;
		for (int i = 0; i < sashes_.length; i++) {
			sashes_[i].setBackground(background_);
		}
	}

	@Override
	public void setForeground(Color color) {
		super.setForeground(color);
		foreground_ = color;
		for (int i = 0; i < sashes_.length; i++) {
			sashes_[i].setForeground(foreground_);
		}
	}

	/**
	 * Sets the layout which is associated with the receiver
	 * to be the argument which may be null.
	 * <p>
	 * Note : No Layout can be set on this Control because
	 * it already manages the size and position of its
	 * children.
	 * </p>
	 * 
	 * @param layout
	 *        the receiver's new layout or null
	 * @exception SWTException
	 *            <ul>
	 *            <li>ERROR_WIDGET_DISPOSED - if the
	 *            receiver has been disposed</li>
	 *            <li>ERROR_THREAD_INVALID_ACCESS - if not
	 *            called from the thread that created the
	 *            receiver</li>
	 *            </ul>
	 */
	@Override
	public void setLayout(Layout layout) {
		checkWidget();
		return;
	}

	/**
	 * Specify the control that should take up the entire
	 * client area of the SashForm. If one control has been
	 * maximized, and this method is called with a different
	 * control, the previous control will be minimized and
	 * the new control will be maximized.. if the value of
	 * control is null, the SashForm will minimize all
	 * controls_ and return to the default layout where all
	 * controls_ are laid out separated by sashes_.
	 * 
	 * @param control
	 *        the control to be maximized or null
	 * @exception SWTException
	 *            <ul>
	 *            <li>ERROR_WIDGET_DISPOSED - if the
	 *            receiver has been disposed</li>
	 *            <li>ERROR_THREAD_INVALID_ACCESS - if not
	 *            called from the thread that created the
	 *            receiver</li>
	 *            </ul>
	 */
	public void setMaximizedControl(Control control) {
		checkWidget();
		if (control == null) {
			if (maxControl_ != null) {
				this.maxControl_ = null;
				layout(false);
				for (int i = 0; i < sashes_.length; i++) {
					sashes_[i].setVisible(true);
				}
			}
			return;
		}

		for (int i = 0; i < sashes_.length; i++) {
			sashes_[i].setVisible(false);
		}
		maxControl_ = control;
		layout(false);
	}

	/**
	 * ̒lw肷ƁÃRg[̒ĺuΓIvɂȂA
	 * TCYsĂTCYύX܂B
	 * Specify the relative weight of each child in the
	 * SashForm. This will determine what percent of the
	 * total width (if SashForm has Horizontal orientation)
	 * or total height (if SashForm has Vertical
	 * orientation) each control will occupy. The weights
	 * must be positive values and there must be an entry
	 * for each non-sash child of the SashForm.
	 * 
	 * @param weights
	 *        the relative weight of each child
	 * @exception SWTException
	 *            <ul>
	 *            <li>ERROR_WIDGET_DISPOSED - if the
	 *            receiver has been disposed</li>
	 *            <li>ERROR_THREAD_INVALID_ACCESS - if not
	 *            called from the thread that created the
	 *            receiver</li>
	 *            <li>ERROR_INVALID_ARGUMENT - if the
	 *            weights value is null or of incorrect
	 *            length (must match the number of children)</li>
	 *            </ul>
	 */
	public void setWeights(int[] weights) {
		checkWidget();
		Control[] cArray = getControls(false);
		if (weights == null || weights.length != cArray.length) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}

		int[] actualSize = new int[weights.length];
		int total = 0;
		for (int i = 0; i < weights.length; i++) {
			if (weights[i] < 0) {
				actualSize[i] = -weights[i];
				weights[i] = 0;
			}
			total += weights[i];
		}
		if (total == 0) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}
		for (int i = 0; i < cArray.length; i++) {
			Object data = cArray[i].getLayoutData();
			if (data == null || !(data instanceof USashFormData)) {
				data = new USashFormData();
				cArray[i].setLayoutData(data);
			}
			USashFormData formData = (USashFormData)data;
			
			formData.weight = (((long)weights[i] << 16) + total - 1) / total;
			formData.setActualSize(actualSize[i]);
		}

		layout(false);
	}
	
	/**
	 * hbNłŏTCYRg[ƂɎw肵܂B<br>
	 * 
	 * @param sizes nullł̃CX^X̎qControlƓKvŁAvf̒l0ȏオKvłB
	 */
	public void setDragMinimums(int[] sizes) {
		checkWidget();
		Control[] cArray = getControls(false);
		if (sizes == null || sizes.length != cArray.length) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}
		
		for (int i = 0; i < cArray.length; i++) {
			if (sizes[i] < 0) {
				SWT.error(SWT.ERROR_INVALID_ARGUMENT);
			}
			Object data = cArray[i].getLayoutData();
			if (data == null || !(data instanceof USashFormData)) {
				data = new USashFormData();
				cArray[i].setLayoutData(data);
			}
			USashFormData formData = (USashFormData)data;
			formData.setDragMinimum(sizes[i]);
		}
	}

	/**
	 * SashIuWFNg̃Rec̈̕Ԃ܂B
	 * @return 
	 */
	public int getSashWidth() {
		return sashWidth_;
	}

	/**
	 * SashIuWFNg̃Rec̈̕ݒ肵܂B
	 * @param sashWidth
	 */
	public void setSashWidth(int sashWidth) {
		sashWidth_ = sashWidth;
	}
	
	/**
	 * ̒lԂ܂B<br>
	 * {@link #sashes_}̗vfȌꍇF		{@link #sashWidth_}<br>
	 * {@link #sashes_}Pȏ̗vfꍇF{@link #sashWidth_} + sashes_[0]borderWidth * 2<br>
	 * 
	 * @return
	 */
	int getWholeSashWidth() {
		return (sashes_.length > 0) ? sashWidth_ + sashes_[0].getBorderWidth() * 2 : sashWidth_;
	}

	/**
	 * @return Returns the sashStyle_.
	 */
	public int getSashStyle() {
		return sashStyle_;
	}

	/**
	 * @param sashStyle_
	 *        The sashStyle_ to set.
	 */
	public void setSashStyle(int style) {
		this.sashStyle_ = style;
	}
	
	/**
	 * 
	 * @return layout\ȏꍇtrue
	 */
	boolean updateSashAndControl() {
		Rectangle area = getClientArea();
		if (area.width <= 1 || area.height <= 1) {
			return false;
		}
		
		Control[] newControls = getControls(true);
		if (controls_.length == 0 && newControls.length == 0) {
			return false;
		}
		controls_ = newControls;

		if (maxControl_ != null && !maxControl_.isDisposed()) {
			for (int i = 0; i < controls_.length; i++) {
				if (controls_[i] != maxControl_) {
					controls_[i].setBounds(-200, -200, 0, 0);
				} else {
					controls_[i].setBounds(area);
				}
			}
			return false;
		}

		// keep just the right number of sashes_
		if (sashes_.length < controls_.length - 1) {
			Sash[] newSashes = new Sash[controls_.length - 1];
			System.arraycopy(sashes_, 0, newSashes, 0, sashes_.length);
			for (int i = sashes_.length; i < newSashes.length; i++) {
				newSashes[i] = createSash();
			}
			sashes_ = newSashes;
		}
		if (sashes_.length > controls_.length - 1) {
			if (controls_.length == 0) {
				for (int i = 0; i < sashes_.length; i++) {
					sashes_[i].dispose();
				}
				sashes_ = new Sash[0];
			} else {
				Sash[] newSashes = new Sash[controls_.length - 1];
				System.arraycopy(sashes_, 0, newSashes, 0, newSashes.length);
				for (int i = controls_.length - 1; i < sashes_.length; i++) {
					sashes_[i].dispose();
				}
				sashes_ = newSashes;
			}
		}
		if (controls_.length == 0) {
			return false;
		}
		return true;
	}

	/**
	 * ݕ\Ă(Sash)vfzŕԂ܂B
	 * zCX^X͕̂ʂĕԂ̂ŁA
	 * z̗vf폜EւĂ
	 * 瑤̃CX^Xɉe^邱Ƃ͂ł܂B
	 * 
	 * @return Returns the controls_.
	 */
	Control[] getControls() {
		Control[] ret = new Control[controls_.length];
		System.arraycopy(controls_, 0, ret, 0, ret.length);
		return ret;
	}

	/**
	 * ێĂSash̔zԂ܂B
	 * zCX^X͕̂ʂĕԂ̂ŁA
	 * z̗vf폜EւĂ
	 * 瑤̃CX^Xɉe^邱Ƃ͂ł܂B
	 * 
	 * @return Returns the controls_.
	 */
	Sash[] getSashes() {
		Sash[] ret = new Sash[sashes_.length];
		System.arraycopy(sashes_, 0, ret, 0, ret.length);
		return ret;
	}
}
