/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.io.FileSnapshot;
import org.unitarou.ml.MessageResource;
import org.unitarou.sgf.Collection;
import org.unitarou.sgf.RootGameTree;
import org.unitarou.sgf.io.CollectionRepository;
import org.unitarou.sgf.parser.SgfFormatter;
import org.unitarou.sgf.type.GameType;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.Application.MessageLevel;

/**
 * LmV^ҏWĂt@CǗNXłB
 * ǂWindowManagerǂ̃t@Ĉǂ̕ҏWE\Ă̂
 * IɈ܂B
 * 
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
final public class FileManager {	
	
	static private final Log log_s_ = LogFactory.getLog(FileManager.class);

	/**
	 * t@CANZXɃG[܂B\n{0}
	 */
	static private final MessageResource MSG_FILE_IO_ERROR 
			= new MessageResource(FileManager.class, "msgFileIoError");//$NON-NLS-1$
	
	/**
	 * t@CJƂł܂łB\n{0}
	 */
	static private final MessageResource MSG_FILE_OPEN_ERROR 
			= new MessageResource(FileManager.class, "msgFileOpenError");//$NON-NLS-1$

	/**
	 * t@Cւ̏oɃG[܂B\n{0}
	 */
	static private final MessageResource MSG_ERROR_ON_WRITE 
			= new MessageResource(FileManager.class, "msgErrorOnWrite");//$NON-NLS-1$

	/** t@C̓ǂݍ݂Ɏs܂B */
	static private final MessageResource NT_FILE_READ_FAILURE
			= new MessageResource(FileManager.class, "ntFileReadFailure"); //$NON-NLS-1$

	/**
	 * {0}̒f[^܂łB
	 * ̃t@C͖܂B
	 */
	static private final MessageResource MSG_GAME_NOT_FOUND 
			= new MessageResource(FileManager.class, "msgGameNotFound"); //$NON-NLS-1$

	
	/** LmV^͏㏑ۑs܂B */
	static private final MessageResource NT_CANT_OVERWRITE
			= new MessageResource(FileManager.class, "ntCantOverwrite"); //$NON-NLS-1$

	/**
	 * t@C{0}ɏ݋֎~ݒ肪܂B
	 */
	static private final MessageResource MSG_READ_ONLY_FILE 
			= new MessageResource(FileManager.class, "msgReadOnlyFile"); //$NON-NLS-1$

	/**
	 * t@C{0}͑̃AvP[VɍXVĂ܂B
	 * [:{1}, TCY:{2}]\LmV^̃t@C
	 * [:{3}, TCY:{4}]\݂̃t@C
	 */
	static private final MessageResource MSG_ALREADY_UPDATED
			= new MessageResource(FileManager.class, "msgAlreadyUpdated"); //$NON-NLS-1$

	/** 
	 * {@link RootGameTree}̂Ă̂ݔłB
	 */
	static private final File trashCan_s_= new File("./trushCan.sgf"); //$NON-NLS-1$
	
	/** ̃NXB̃CX^XłB*/
	static private final FileManager instance_s_ = new FileManager();
	

	/** ̃NXB̃CX^XԂ܂B*/
	static public FileManager instance() {
		return instance_s_;
	}
	
	/** ^[File, Collection] */
	private final Map<File, Collection> fileMap_;


	/** */
	private FileManager() {
		super();
		fileMap_ = new HashMap<File, Collection>();
	}

	/**
	 * w肳ꂽt@Cǂݍ݂܂B
	 * ɓǂݍ܂Ăꍇ́A
	 * IOǂݍ܂ɂ̂܂܃CX^XԂ܂B
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
	 * @return t@CɈꌏRGTꍇ̓bZ[W_CAO\NULLԂ܂B
	 */
	public Collection open(File file) {
	    ArgumentChecker.throwIfNull(file);
	    Collection collection = openImpl(file);
	    if (collection.size() != 0) { 
	    	return collection;
	    }
		String msg = MSG_GAME_NOT_FOUND.get(file);
		log_s_.info(msg);
		Yukinoshita.application().openMessageDialog(
				MessageLevel.INFORMATION,
				NT_FILE_READ_FAILURE.get(), msg);
	    return null;
	}
	
	/**
	 * w肳ꂽt@Cǂݍ݂܂B
	 * ɓǂݍ܂Ăꍇ́A
	 * IOǂݍ܂ɂ̂܂܃CX^XԂ܂B
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
	 */
	private Collection openImpl(File file) {
	    ArgumentChecker.throwIfNull(file);

		// t@C̏d`FbN
		Collection collection = fileMap_.get(file);
		if (collection != null) {
			return collection;
		}

		collection = CollectionRepository.instance().load(file);
//		fileMap_.put(file, collection);
		return collection;
	}

    /**
     * ̃CX^X쐬Aǂݍ񂾃RNVJ܂B
     * RNV̕ύX̓t@CɔfȂ̂ŒӁI
     * Ƀt@Cۑ{@link #save(Collection)}ĂяoKvB
     * 
     * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
     */
    public void close(Collection collection) {
	    ArgumentChecker.throwIfNull(collection);

	    if (collection.getFile() != null) {
            fileMap_.remove(collection.getFile());
        }
    }

    /**
     * collectionۑ܂B
     * {@link Collection#getFile()}
     * @throws YukinoshitaException
     * 
     * @throws org.unitarou.lang.NullArgumentException collectionnull̏ꍇ
     * @throws IllegalArgumentException {@link Collection#getFile()}nullԂꍇB
     */
    public void save(Collection collection) throws YukinoshitaException {
	    ArgumentChecker.throwIfNull(collection);
        
        if (collection.getFile() == null) {
            throw new IllegalArgumentException("collection.getFile() must not be null."); //$NON-NLS-1$
        }
        
        
        try {
            if (!GameType.DRILL.equals(collection.get(0).getGameType())) {
            	saveImpl(collection);
            } else {
        		// Drill̏ꍇAWĂŒ͎ۂ̋l邪A
                // ۑĂ܂̂ŁAUʂĐ擪ȊORootGameTree폜B
            	Collection colForSave = new Collection(collection);
    			while (1 < colForSave.size()) {
    				colForSave.remove(colForSave.get(1));
    			}
            	saveImpl(colForSave);
            }
        } finally {
            collection.setFile(collection.getFile());//MEMO XibvVbg̍XV
        }
    }
    
    private void saveImpl(Collection collection) throws YukinoshitaException {
        SgfFormatter sgfFormater = new SgfFormatter();
        FileOutputStream fileOutputStream = null;
        try {
            ByteBuffer byteBuffer = sgfFormater.format(collection);
            fileOutputStream = new FileOutputStream(collection.getFile());
            fileOutputStream.getChannel().write(byteBuffer);
            fileOutputStream.flush();
            
        } catch (FileNotFoundException e) {
            throw new YukinoshitaException(
            		MSG_ERROR_ON_WRITE.get(e.getLocalizedMessage()), e);

        } catch (IOException e) {
            throw new YukinoshitaException(
            		MSG_FILE_IO_ERROR.get(e.getLocalizedMessage()), e);
            
        } finally {
        	IOUtils.closeQuietly(fileOutputStream);
        }
    }
    
    /**
     * collectionw肷t@C㏑\
     * (݉\collection쐬̃^CX^vωȂ)
     * ̏ꍇtrueԂ܂B
     * ȊȌꍇ̓_CAOofalseԂ܂B
     * t@Cnull̏ꍇfalseԂ܂B
     * @param collection
     * @return
     */
    public boolean checkOverwritable(Collection collection) {
    	File file = collection.getFile();
    	if (file == null) {
    		return false;
    	}
    	FileSnapshot last = collection.getFileSnapshot();
    	FileSnapshot now = new FileSnapshot(file);
    	if (now.equals(last) && file.canWrite()) {
    		return true;
    	}
    	String msg;
    	if (!file.canWrite()) {
    		msg = MSG_READ_ONLY_FILE.get(file.getAbsolutePath());
    	} else {
    		msg = MSG_ALREADY_UPDATED.get(
    				file.getAbsolutePath(),
    				new Date(last.lastModified()), new Long(last.length()),
    				new Date(now.lastModified()), new Long(now.length()));
    	}
    	Yukinoshita.application().openMessageDialog(
    			MessageLevel.INFORMATION,
    			NT_CANT_OVERWRITE.get(),
    			msg);
    	return false;
    }
    
    
    /**
     * rgtݔɈړ܂
     * @param rgt
     */
    public void drop(RootGameTree rgt) {
    	Collection collection = null;
    	try {
    		collection = openTrashCan();
        	collection.addLast(rgt);
			save(collection);
		
    	} catch (YukinoshitaException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();

		} finally {
			if (collection != null) {
		    	close(collection);
			}
		}
    }
    
    /**
     * ݔindexԖڂ{@link RootGameTree}o܂B
     * oꂽIuWFNg͂ݔ͍폜܂B
     * 
     * @param index
     * @return
     */
    public RootGameTree pickUp(int index) {
    	RootGameTree ret = null;
    	Collection trashCan = null;
    	try{
        	trashCan = openTrashCan();
        	if ((index < 0) || (trashCan.size() <= index)) {
        		throw new IllegalArgumentException("Bad index (out of range):" + index); //$NON-NLS-1$
        	}
        	ret = trashCan.get(index);
        	trashCan.remove(ret);
			save(trashCan);
        	return ret;

    	} catch (YukinoshitaException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return ret;
			
		} finally {
			if (trashCan != null) {
		    	close(trashCan);
			}
    	}
    }
    
    /**
     * ݔ̒{@link Collection}𕡎ʂĕԂ܂B
     * @return
     */
    public Collection getTrashCan() {
    	Collection trashCan = null;
    	try{
        	trashCan = openTrashCan();
        	return new Collection(trashCan);
    	} finally {
			if (trashCan != null) {
		    	close(trashCan);
			}
    	}
    }
    
    
    /**
     * ݔɂ܂B 
     */
    public void clearTrashCan() {
    	trashCan_s_.delete();
    }
    
    
    /**
     * ݔt@CǂݍŕԂ܂B
     * ݂Ȃꍇ͋{@link Collection}쐬āA
     * t@C{@link #trashCan_s_}ɐݒ肵ĕԂ܂B
     * @return
     */
    private Collection openTrashCan() {
    	if (!trashCan_s_.exists()) {
    		Collection collection = new Collection();
    		collection.setFile(trashCan_s_);
    		return collection;
    	}
    	return openImpl(trashCan_s_);
    }
}
