/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.model;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.unitarou.cmd.Command;
import org.unitarou.cmd.CommandDriver;
import org.unitarou.cmd.CommandDriverListener;
import org.unitarou.cmd.CommandDriver.Status;
import org.unitarou.sgf.PropertyType;
import org.unitarou.sgf.RootGameTree;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.type.GameType;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.sgf.util.BasicFinder;
import org.unitarou.sgf.util.provider.crdlp.CoordinatesLabelProvider;
import org.unitarou.sgf.util.provider.crdlp.NoCoordinatesProvider;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.Yukinoshita;
import org.unitarou.yukinoshita.context.Context;
import org.unitarou.yukinoshita.context.CurrentContext;
import org.unitarou.yukinoshita.model.board.IgoBoard;
import org.unitarou.yukinoshita.model.cmd.Command4NodeList;
import org.unitarou.yukinoshita.model.cmd.SelectVariation;
import org.unitarou.yukinoshita.model.cmd.StartSolvingProblem;
import org.unitarou.yukinoshita.view.provider.vlp.NoVariationProvider;
import org.unitarou.yukinoshita.view.provider.vlp.VariationLabelProvider;

/**
 * RootGameTreeɑΉControllerłB<br>
 * undo/redo@\܂B
 * 
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 * @see org.unitarou.sgf.Collection
 */
public class GameMediator {
	/** ̃CX^XǗc[ł */
	private final RootGameTree root_;
	
	/** ̃CX^XǗҏW\ȃm[hXgł */
	private final EditableNodeList nodeList_;
	
	/** ̒ǉȂǂ̃R}hۂɍshCo[ł */
	private final CommandDriver commandDriver_;
	
	/** 
	 * {@link #root_}ҏWꂽ񐔂ێ܂B
	 * {@link #isChanged()}ŗp܂B
	 */
	private int changedCounter_;

	
	
	// lp̏ԕێ
	/**
	 * lĂtF[ŶƂtrueێtOłB
	 */
	private boolean isSolvingProblem_;
	
	/**
	 * ^[String:key, Object:value]B
	 * lĂƂɁAꎞIɃReLXgޔ邽߂Ɏg܂B
	 */
	private final Map<String, Object> sidetrackContextMap_;
	
	
	/**
	 * 쐬Ԃł͖ύXԂŏ܂B
	 * @param 삳郂fw肵܂B
	 * @throws org.unitarou.lang.NullArgumentException rootnull̏ꍇB
	 */
	public GameMediator(RootGameTree root) {
		this(root, false);
	}

	/**
	 * ̃NX̃vCRXgN^łB
	 * @param 삳郂fw肵܂B
	 * @param defaultChanged trueɐݒ肷ƃRXgN^̏Ԃ
	 *                        {@link #isChanged()}<code>ture</code>Ԃ܂B
	 * @throws org.unitarou.lang.NullArgumentException rootnull̏ꍇB
	 */
	public GameMediator(RootGameTree root, boolean defaultChanged) {
		super();
		ArgumentChecker.throwIfNull(root);

		root_ = root;
		nodeList_ = new EditableNodeList(root_);
		commandDriver_ = new CommandDriver();
		commandDriver_.addListener(new ChangeListener());
		sidetrackContextMap_ = new HashMap<String, Object>(); 
		initializeByGameType();
		changedCounter_ = defaultChanged ? 1 : 0;
	}
	
	/**
	 * GT^O̒gɉݒs܂B
	 * GT[1]̏ꍇ͍ŏIԂ\܂B
	 * GT[3]̏ꍇ͍ŏ̒̈O\Aω}Ȃǂ̋L\ɂ܂B
	 */
	private void initializeByGameType() {
		isSolvingProblem_ = false;
		
		nodeList_.setPositionIndex(nodeList_.size() - 1);
		
		String datum = BasicFinder.findDatum(root_.getSequence(), SgfId.GAME_TYPE);
		if (datum.equals(GameType.PROBLEM.getString())) {// l̏ꍇ
			startSolvingProblem();
		}
	}
	
	
	/**
	 * ̃CX^XǗĂ{@link RootGameTree}Ԃ܂B
	 * @return ̃CX^XǗĂ{@link RootGameTree}B<code>null</code>͕ԂȂB
	 */
	public RootGameTree getRootGameTree() {
		return root_;
	}
	
	/**
	 * CX^Xg^[QbgƂR}hs܂B
	 * ̃R}hREDO/UNDOXgɒǉ܂B
	 * @param command
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇB
	 */
	public Status executeCommand(Command4NodeList command) {
		ArgumentChecker.throwIfNull(command);
		EditableNodeList editableNodeList = command.getEditableNodeList();

		/**
		 * ̃CX^XێĂ{@link #nodeList_}
		 * commandێĂCX^XقȂꍇ́A
		 * ̃CX^Xł͏łȂ̂ŗOo 
		 */
		if ((editableNodeList != null) && (nodeList_ != editableNodeList)) {
			throw new IllegalArgumentException("Bad match of EditableNodeList"); //$NON-NLS-1$
		}
		
		command.setEditableNodeList(nodeList_);
		return commandDriver_.execute(command);
	}

	/**
     * ÕR}h܂B
     * @return
     */
    public Status undo() {
    	return commandDriver_.undo();
    }
    
    /**
     * OɎĎs܂B
     * @return
     */
    public Status redo() {
    	return commandDriver_.redo();
    }

	/**
	 * @return
	 */
	public Status getCommandDriverStatus() {
		return commandDriver_.getStatus();
	}

    /**
	 * 胊XgԂ܂B
	 *  <code>null</code>͕Ԃ܂B
	 */
	public NodeList getNodeList() {
		return nodeList_;		
	}

	/**
	 * ݒڂĂ{@link NodeView}Ԃ܂B
     */
    public NodeView getCurrentNodeView() {
    	return nodeList_.getNodeView(nodeList_.getPositionIndex());
    }
    
    /**
     * ݒڂĂ{@link NodeView}̃CfbNXԂ܂B
     */
    public int getCurrentNodeIndex() {
    	return nodeList_.getPositionIndex();
    }
    
    /**
     * propertyTypeێNodeView̃CfbNXԂ܂B
     * Ȃꍇ-1Ԃ܂B
     * @param propertyType
     * @return
     */
    public int findNodeView(PropertyType propertyType) {
		int index = getCurrentNodeIndex();
		NodeView target = nodeList_.getNodeView(0);
		while (0 <= index) {
			target = nodeList_.getNodeView(index);
			if (target.containsPropertyType(propertyType)) {
				break;
			}
			--index;
		}
		return index;
    }

    /**
     * indexNodeListݒ肵܂B
     * 
     * @param index
     * @throws IndexOutOfBoundsException index{@link #nodeList_}̃Xg͈̔͂𒴂ĂƂ
     */
    public void setNodeIndex(int index) {
    	nodeList_.setPositionIndex(index);
    }

    /**
     * Xgdeltaړ܂B
     * Xg͈̔͂𒴂ꍇ͎Iɔ͈͓ɔ[߂܂B
     * @param delta
     * @return ړ{@link #getCurrentNodeView()}ԂCX^XB
     */
    public NodeView setNodeIndexDelta(int delta) {
        int newIndex = nodeList_.getPositionIndex();
        newIndex += delta;
        if (newIndex < 0) {
            newIndex = 0;
        } else if (nodeList_.size() <= newIndex ) {
            newIndex = nodeList_.size() - 1;
        }
        nodeList_.setPositionIndex(newIndex);
        return getCurrentNodeView();
    }


	/** ݒ̏ꍇGameType.GAMEԂ܂ */
	public GameType getGameType() {
		return root_.getGameType();
	}

    /**
     * Q[ҏWĂꍇtrueԂ܂B
     */
    public boolean isChanged() {
        // undo̐𐧌ẴR[hőΉ\B
        return changedCounter_ != 0;
    }	
    
    /**
     * t@CŕۑꂽȂǂɌĂяo܂B
     * UNDO/REDO͉\łȀԂN_ƂĕҏWꂽ
     * ǂ𔻒肵܂B 
     */
    public void clearChanged() {
        changedCounter_ = 0;
    }
    
    /**
     * l̉𓚊JnɌĂяo܂B
     * l̖Jnn_{@link #nodeList_}ړR}h
     * {@link StartSolvingProblem}s܂B
     * {@link #finishSolvingProblem()}Ăяo
     * JԂ̃\bhĂяoꂽꍇ́A
     * ȑÕR}hāAxs܂B
     */
    public void startSolvingProblem() {
        if (isSolvingProblem_) {
            finishSolvingProblem();
        }
        storeContextForSolvingProblem();
		isSolvingProblem_ = true;
		StartSolvingProblem command = new StartSolvingProblem(this);
        commandDriver_.execute(command);
    }
    /**
     * lʗpɃReLXg̑ޔƐݒ肵܂B
     * EŏI̋L͔\
     * E΂̃x\
     * Eω}̋L\
     * EՂ̍W\
     * Ɛݒ肳܂B
     */
    private void storeContextForSolvingProblem() {
		Context context = Yukinoshita.context();
		sidetrackContextMap_.clear();
		sidetrackContextMap_.put(Context.SHOW_LAST_MOVE_MARK.id(),
		        				context.getAttribute(Context.SHOW_LAST_MOVE_MARK.id(), root_));
		sidetrackContextMap_.put(VariationLabelProvider.class.getName(),
								context.getProvider(VariationLabelProvider.class, root_));
		sidetrackContextMap_.put(CoordinatesLabelProvider.class.getName(),
								context.getProvider(CoordinatesLabelProvider.class, root_));
		
		context.setAttribute(Context.SHOW_LAST_MOVE_MARK.id(), Boolean.FALSE, root_);
		context.setProvider(VariationLabelProvider.class, NoVariationProvider.class, root_);
		context.setProvider(CoordinatesLabelProvider.class, NoCoordinatesProvider.class, root_);
		Set<String> keyset = sidetrackContextMap_.keySet();
		context.fireAttributeChanged(keyset.toArray(new String[keyset.size()]));
    }
    
    /**
     * l̉𓚏I(AԈႢ킸)ɌĂяo܂B
     * {@link StartSolvingProblem}TĂ܂Ń[obN܂B
     * 
     * @throws IllegalStateException {@link StartSolvingProblem}Ȃꍇ
     */
    public void finishSolvingProblem() {
        if (!isSolvingProblem_) {
            return;
        }
        Command firstCommand = null;
        while (commandDriver_.getStatus().isUndoable()) {
        	Status status = commandDriver_.undo();
        	Command command = status.getExecutedCommand();
            if (firstCommand == null) { 
                firstCommand = command;
            }
            if (command instanceof StartSolvingProblem) {
                // isSolvingProblem_͊SundoIȂ
                // undochangeCounter_ɉeôŁA
                // \bh̍Ōfalseɐݒ肵ĂB
                isSolvingProblem_ = false;

                // lJnR}hĂB
                commandDriver_.clearRedo();

                // l𓚎̓ȏĂB
                restoreContextBySolvingProblem();
                return;
            }
        }
        

		// ȉ͓ԂԈĂ邽߂̗OB
        // undoJnn_܂ŃR}h߂ĂAO𑗏oĂB
        if (firstCommand != null) {
            while (firstCommand != commandDriver_.redo()) {
                // ŕKvȏ͍sĂB
            }
        }
        isSolvingProblem_ = false;

        // l𓚎̓ȏĂB
        restoreContextBySolvingProblem();
        throw new IllegalStateException("Can't find StartSolvingProblem command."); //$NON-NLS-1$
    }
    
    /**
     * l̉𓚂IƂɁAȑÕReLXg𕜋A܂B 
     *
     */
    private void restoreContextBySolvingProblem() {
		Context context = Yukinoshita.context();
		for (Iterator<Map.Entry<String, Object>> ip = sidetrackContextMap_.entrySet().iterator();
				ip.hasNext();) {
            Map.Entry<String, Object> entry = ip.next();
            context.setAttribute(entry.getKey(), entry.getValue(), root_);
        }
		Set<String> keyset = sidetrackContextMap_.keySet();
		context.fireAttributeChanged(keyset.toArray(new String[keyset.size()]));
    }
    
    /**
     * ݒڂĂm[hƂ̌̑Sm[hɂāA
     * ΂dȂĂ邩𒲍AdȂĂ΂̃|Cg̔zԂ܂B
     * m[hXg̓rSetupm[hA̐ݒ肪ύXꂽꍇ
     * Ăяo邱Ƃz肵Ă܂B
     * @return ΂dȂĂ|Cg̔zAΗvf0̔zԂB
     */
    public SgfPoint[] findInvalidPoints() {
		CommandDriver driver4Validate = new CommandDriver();
		Set<SgfPoint> invalidPoint = validatePoint(nodeList_, nodeList_.getPositionIndex(), driver4Validate);
		
		// I番͑SUndoB
		while(driver4Validate.getStatus().isUndoable()) {
		    driver4Validate.undo();
		}
		return invalidPoint.toArray(new SgfPoint[invalidPoint.size()]);
    }

    /**
     * ݒڂĂm[hȍ~̒̑Ó
     * (uȂƂɐ΂uĂȂ)`FbN郁\bhłB
     * 
     * ̓Iɂ{@link IgoBoard#validate()}\bhĂяoāA
     * StoneGroupɏdȂ肪΂̏W𒲂ׂĂ܂B
     * ڂm[hȍ~̑Sm[hɑ΂ă`FbNKv邽߁A
     * ʓrCommandDriverpӂāA̒SequencePʂōċAIɃ`FbNĂ܂B
     * 
     * @return dȂ肪΂{@link SgfPoint}̏W
     */
    private Set<SgfPoint> validatePoint(
            EditableNodeList nodeList, int nodeIndex, CommandDriver driver4Validate) 
    {
        Set<SgfPoint> invalidPoint = new HashSet<SgfPoint>();
        while(nodeIndex < nodeList.size()) {
        	nodeList.setPositionIndex(nodeIndex);
            NodeEntity entity = nodeList.getPosition();
            invalidPoint.addAll(entity.getIgoBoard().validate());

            if (entity.getNodeTree().getVariationSize() == 0) {
                ++nodeIndex;
                continue;
            }
            int nextNodeIndex = nodeIndex + 1;
            invalidPoint.addAll(validatePoint(nodeList, nextNodeIndex, driver4Validate));
            for (int i = 1; i < entity.getNodeTree().getVariationSize(); ++i) {
            	nodeList.setPositionIndex(nodeIndex);
                SelectVariation command = new SelectVariation(i);
                command.setEditableNodeList(nodeList);
                driver4Validate.execute(command);
                invalidPoint.addAll(validatePoint(nodeList, nextNodeIndex, driver4Validate));
            }
            break;
        }
        return invalidPoint;
    }
    
    /**
     * {@link RootGameTree}ɕRÂReLXgԂ܂B
     * @return
     */
    public CurrentContext getContext() {
        return Yukinoshita.context().getCurrent(root_);
    }

    /**
     * {@link GameMediator#root_}ɕύXǂ`FbN邽߂̃Xi[łB
     * ł{@link SelectVariation}{@link StartSolvingProblem}
     * {@link GameMediator#root_}ύXȂR}hȂ߁AꂾOĂ܂B
     * ܂{@link GameMediator#isSolvingProblem_}true͕̎ύX𖳎܂B
     */
    private class ChangeListener implements CommandDriverListener {

        /**
         * ^[{@link Class}]łB
         * f̕ύX𔺂ȂR}h̃NXCX^Xێ܂B
         */
    	private final Set<Class<? extends Command4NodeList>> ignoreCommandSet_;
    	
    	public ChangeListener() {
            super();
            ignoreCommandSet_ = new HashSet<Class<? extends Command4NodeList>>();
            ignoreCommandSet_.add(SelectVariation.class);
            ignoreCommandSet_.add(StartSolvingProblem.class);
        }
    	
    	/**
    	 * fҏWR}h̏ꍇAtrueԂ܂B
    	 * ȂAlĂŒ͏falseԂ܂B
    	 */
    	private boolean isChangedCommand(Command command) {
    	    return !(isSolvingProblem_ || ignoreCommandSet_.contains(command.getClass()));
    	}
    	
    	/* (non-Javadoc)
         * @see org.unitarou.cmd.CommandDriverListener#undoExecuted(org.unitarou.cmd.Command)
         */
        public void undoExecuted(Command command) {
            if (isChangedCommand(command)) {
                --changedCounter_;
            }
            if (command instanceof Command4NodeList) {
            	nodeList_.setPositionIndex(((Command4NodeList)command).getUndoneNodeIndex());
            } else {
            	nodeList_.setPositionIndex(0);
            }
             
        }

        /* (non-Javadoc)
         * @see org.unitarou.cmd.CommandDriverListener#executed(org.unitarou.cmd.Command)
         */
        public void executed(Command command) {
            if (isChangedCommand(command)) {
                ++changedCounter_;
            }
            if (command instanceof Command4NodeList) {
            	nodeList_.setPositionIndex(((Command4NodeList)command).getExecutedNodeIndex());
            } else {
            	nodeList_.setPositionIndex(0);
            }
        }

        /* (non-Javadoc)
         * @see org.unitarou.cmd.CommandDriverListener#redoExecuted(org.unitarou.cmd.Command)
         */
        public void redoExecuted(Command command) {
            if (isChangedCommand(command)) {
                ++changedCounter_;
            }
            if (command instanceof Command4NodeList) {
            	nodeList_.setPositionIndex(((Command4NodeList)command).getExecutedNodeIndex());
            } else {
            	nodeList_.setPositionIndex(0);
            }
        }
    }
}
