/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.sgf.GameTree;
import org.unitarou.sgf.Node;
import org.unitarou.sgf.Property;
import org.unitarou.sgf.PropertyType;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.type.GameType;
import org.unitarou.sgf.type.SgfColor;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.sgf.type.SgfReal;
import org.unitarou.sgf.type.TimeLapsed;
import org.unitarou.sgf.util.BasicFinder;
import org.unitarou.sgf.util.Stone;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.model.board.IgoBoard;
import org.unitarou.yukinoshita.model.board.InheritableMarker;
import org.unitarou.yukinoshita.model.board.MarkupIgoBoard;

/**
 * {@link Node}CX^X{@link EditableNodeList}
 * ߁E삵₷`ɃbvNXłB
 * 
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public final class NodeEntity {
	static private final Log log_s_ = LogFactory.getLog(NodeEntity.class);

	/** Ȃꍇ{@link #selectedVariationIndex_}ƂlłB*/
	static public final int NO_VARIATION = -1;

	
	/**
	 * {@link #inheritPointMap_}̃L[gړŁAǉӖ܂B
	 */
	static private final String PRE_APPENDED = "appended"; //$NON-NLS-1$
	
	/**
	 * {@link #inheritPointMap_}̃L[gړŁA폜Ӗ܂B
	 */
	static private final String PRE_REMOVED = "removed"; //$NON-NLS-1$
	
	/**
	 * ̔z΂̒ǉwL[łB
	 * @see {@link #inheritPointMap_}
	 */
	static private final String APPEND_SETUP_BLACK 
			= PRE_APPENDED + SgfId.ADD_BLACK.id();
	
	/**
	 * ̔z΂̍폜wL[łB
	 * @see {@link #inheritPointMap_}
	 */
	static private final String REMOVE_SETUP_BLACK 
			= PRE_REMOVED + SgfId.ADD_BLACK.id();

	/**
	 * ̔z΂̒ǉwL[łB
	 * @see {@link #inheritPointMap_}
	 */
	static private final String APPEND_SETUP_WHITE 
			= PRE_APPENDED + SgfId.ADD_WHITE.id();

	/**
	 * ̔z΂̍폜wL[łB
	 * @see {@link #inheritPointMap_}
	 */
	static private final String REMOVE_SETUP_WHITE
			= PRE_REMOVED + SgfId.ADD_WHITE.id();
	
	static private final Set<SgfId> idSet4Problem_s_
			= Collections.unmodifiableSet(new HashSet<SgfId>(Arrays.asList(
					new SgfId[]{SgfId.BLACK, SgfId.KO, SgfId.SET_MOVE_NUMBER, SgfId.WHITE,
								SgfId.ADD_BLACK, SgfId.ADD_EMPTY, SgfId.ADD_WHITE, SgfId.PLAYER_TO_PLAY,
								SgfId.COMMENT, SgfId.GOOD_FOR_BLACK, SgfId.GOOD_FOR_WHITE,
								SgfId.BAD_MOVE,
								SgfId.ARROW, SgfId.CIRCLE, SgfId.DIM_POINTS, SgfId.LABEL, 
								SgfId.LINE, SgfId.MARK_WITH_X, SgfId.SELECTED, SgfId.SQUARE, SgfId.TRIANGLE,
								SgfId.APPLICATION, SgfId.CHARSET, SgfId.FILE_FORMAT, SgfId.GAME_MODE,
								SgfId.GAME_TYPE, SgfId.STYLE, SgfId.SIZE,
								SgfId.BLACK_RANK, SgfId.COPYRIGHT, SgfId.DATE, SgfId.GAME_COMMENT,
								SgfId.GAME_NAME, SgfId.RESULT, SgfId.SOURCE, SgfId.TIMELIMIT, SgfId.USER,
								SgfId.VIEW,
								})));

	static private final Set<SgfId> idSet4Drill_s_
			= Collections.unmodifiableSet(new HashSet<SgfId>(Arrays.asList(
					new SgfId[]{SgfId.APPLICATION, SgfId.CHARSET, SgfId.FILE_FORMAT, SgfId.GAME_MODE,
								SgfId.GAME_TYPE, SgfId.STYLE, SgfId.BLACK_RANK, SgfId.GAME_COMMENT,
								SgfId.GAME_NAME, SgfId.INPUT_FILES, SgfId.PROBLEM_PROPERTIES,
								})));

	/**
	 * CX^XێĂ郊XgłB
	 */
	private final EditableNodeList nodeList_;
	
	/**
	 * ̃m[hŒǉꂽ΂InheritƁA
	 * (Õm[hƂ̔rŋ߂)폜ꂽ΂Ƒ̏WłB<br>
	 * L[łꂼ̏WԂ܂F
	 * <pre>
	 * key: "appended" +  {@link SgfId#id()}	value: ǉꂽInherit̏W
	 * key: "removed" +  {@link SgfId#id()}	value: ǉꂽInherit̏W
	 * </pre> 
	 * ΂̏ꍇ{@link SgfId#ADD_BLACK}, {@link SgfId#WHITE}Ȃǂ܂B
	 * ADD_BLACKBLACK͕ʁXɕۊǂ܂B
	 * ̓AQn}̌vZɂ͒ɂ鎀΂݂̂vłB
	 */
	private final Map<String, SgfPoint[]> inheritPointMap_; 
	
	/**
	 * ̃GeBeB̃Xgł̈ÕCX^XłB
	 * {@link #forwardUpdate(NodeEntity, MarkupIgoBoard)}ōXV܂B
	 * ̃CX^XXg̐擪̏ꍇ́Anulll܂B
	 */
	private NodeEntity previous_;
	
	/**
	 * bvĂNodełB
	 */
	private final Node node_;
	
	/**
	 * GT{SE(node_) (GT)(GT)(GT)}<br>
	 *  IIndex~~~~~~~~~~~~~
	 */
	private int selectedVariationIndex_;
	
	
	private final NodeTree nodeTree_;

	/** DD, VWƂAp鑮łB*/
	private final InheritableMarker inheritableDecoration_;
	
	/** ̃GeBeB̒łB*/
	private Stone move_;
	
	/** (킸)̒ł̃AQn}܂B*/
	private final List<Stone> capturedStones_;
	
	/** ΂dȂĂȂǕsȃ|Cg̏WłB*/
	private final Set<SgfPoint> invalidPoints_;
	
	/**
	 * 
	 */
	private NodeView nodeView_;

	
	/**
	 * @param node bvm[h
	 * @param gameTree node̒ڂ̐eɂȂc[
	 * @throws org.unitarou.lang.NullArgumentException node, treenull̏ꍇ
	 */
	public NodeEntity(Node node, GameTree gameTree, EditableNodeList nodeList, NodeEntity previous) {
	    ArgumentChecker.throwIfNull(node, gameTree, nodeList);

		node_ = node;
		nodeTree_ = new NodeTree(node_, gameTree);
		nodeList_ = nodeList;
		previous_ = previous;
		nodeView_ = null;

		// inheritPointMap_̏
		inheritPointMap_ = new HashMap<String, SgfPoint[]>();
		inheritPointMap_.put(APPEND_SETUP_BLACK, SgfPoint.EMPTY_ARRAY);
		inheritPointMap_.put(REMOVE_SETUP_BLACK, SgfPoint.EMPTY_ARRAY);
		inheritPointMap_.put(APPEND_SETUP_WHITE, SgfPoint.EMPTY_ARRAY);
		inheritPointMap_.put(REMOVE_SETUP_WHITE, SgfPoint.EMPTY_ARRAY);
		for (SgfId sgfType : PropertyType.INHERIT.getSgfTypeSet()) {
			inheritPointMap_.put(PRE_APPENDED + sgfType.id(), SgfPoint.EMPTY_ARRAY);
			inheritPointMap_.put(PRE_REMOVED  + sgfType.id(), SgfPoint.EMPTY_ARRAY);
		}
		
		
		selectedVariationIndex_ = 
		    	(nodeTree_.getVariationSize() == 0)
				? NO_VARIATION : 0;
		
		inheritableDecoration_ = new InheritableMarker();
		move_ = Stone.NULL_STONE;
		composeMove();
		
		// ̓AQn}͌vłȂ̂0
		capturedStones_ = new ArrayList<Stone>(0);
		invalidPoints_ = new HashSet<SgfPoint>(1);
	}

	

	/**
	 * @param index
	 */
	public void setSelectedVariationIndex(int index) {
		int size = nodeTree_.getVariationSize();
		if (0 < size && (index < 0 || nodeTree_.getVariationSize() <= index)) {
				throw new IndexOutOfBoundsException();
		} else if (size == 0 && index != NO_VARIATION) {
			throw new IndexOutOfBoundsException();
		}
		selectedVariationIndex_ = index;
	}
	
	/**
	 * {@link EditableNodeList}ŃXg̍XVƂ
	 * Ăяo܂B
	 * @param previous
	 */
	void setPrevious(NodeEntity previous) {
		previous_ = previous;
	}
	
	/**
	 * previousViewIgoBoardx[XɁA
	 * Node_Ŏw肳Ă钅Az΁AԍApǉ܂B
	 * @param previous CX^ẌÕGeBeBłB
	 *         ̃CX^X[gm[hłꍇnullw肳܂B
	 */
	void forwardUpdate(NodeEntity previous, MarkupIgoBoard igoBoard) {
	    previous_ = previous;
	    
	    // ω}̍XVs
	    // FIXME łȂXV̂Ȃ
		updateVariationIndex();
	    
	    // node_inheritPointMap_XVB
	    
	    // z΂̒ǉ
	    setStones(SgfColor.BLACK, igoBoard);
	    setStones(SgfColor.WHITE, igoBoard);
	    
	    // z΂̍폜
	    setEmpties(igoBoard);

	    // ̒ǉƍ폜
	    moveStone(igoBoard);
	    
	    // s_̍XV
	    invalidPoints_.clear();
	    invalidPoints_.addAll(igoBoard.validate());
	    
	    //fR[V̍XV
		updateInheritedDecoration();
	}
	
	/**
	 * ̃m[hɂigoBoardւ̕ύXSăNA[܂B
	 * 
	 * @param entity
	 */
	void backwardUpdate(MarkupIgoBoard igoBoard) {
	    // ω}̍XVs
	    // FIXME łȂXV̂Ȃ
		updateVariationIndex();
	    
	    // inheritPointMap_igoBoardXV
	    // nodeŒǉz΂Sč폜A
	    // 폜z΂SĒǉB
	    for (SgfPoint point : inheritPointMap_.get(APPEND_SETUP_BLACK)) {
	    	igoBoard.removeStone(point);
	    }

	    for (SgfPoint point : inheritPointMap_.get(APPEND_SETUP_WHITE)) {
	    	igoBoard.removeStone(point);
	    }
	    
	    for (SgfPoint point : inheritPointMap_.get(REMOVE_SETUP_BLACK)) {
	    	igoBoard.setStone(new Stone(SgfColor.BLACK, point, null));
	    }

	    for (SgfPoint point : inheritPointMap_.get(REMOVE_SETUP_WHITE)) {
	    	igoBoard.setStone(new Stone(SgfColor.WHITE, point, null));
	    }
	    
	    // nodeŒǉSč폜
	    // 폜(łグꂽ)΂SĒǉB
	    if (move_.isValid()) {
		    igoBoard.removeStone(move_.getPoint());
	    
		    for (Stone stone : capturedStones_) {
		    	igoBoard.setStone(stone);
		    }
		    igoBoard.removeCaptured(capturedStones_);
	    }
	}
	
	/**
	 * ω}̃CfbNX̍XVs܂B
	 * ω}̒ǉE폜ɌĂяoKv܂B 
	 */
	public void updateVariationIndex() {
	    int variations = nodeTree_.getVariationSize(); 
	    if (variations == 0) {
	        selectedVariationIndex_ = NO_VARIATION;
	    } else if ( (variations <= selectedVariationIndex_) 
	            	|| (selectedVariationIndex_ == NO_VARIATION)){
	        selectedVariationIndex_ = 0;
	    }
	}

	/**
	 * igoBoard_NodeŎw肳ꂽ΂zu܂B
	 */
	private void setStones(SgfColor color, IgoBoard igoBoard) {
		Property property = node_.getProperty(color.setupType());
		if (property == null) {
			return;
		}
		
		// zΒǉp̏W쐬
		Set<SgfPoint> set = new HashSet<SgfPoint>();
		
		// ՂƏWɐ΂ǉ
		String[] values = property.getStrings();
		for (SgfPoint sgfPoint : SgfPoint.parse(nodeList_.getSize(), values)) {
			set.add(sgfPoint);
			// validateɈȂ悤ɈȔꏊɂ΂폜
			igoBoard.removeStone(sgfPoint);
			igoBoard.setStone(new Stone(color, sgfPoint, null));
		}
		
		if (set.isEmpty()) {
			inheritPointMap_.put(
					PRE_APPENDED + color.setupType().id(), 
					SgfPoint.EMPTY_ARRAY);
			
		} else {
			inheritPointMap_.put(
					PRE_APPENDED + color.setupType().id(), 
					set.toArray(new SgfPoint[set.size()]));
		}
	}

	/**
	 * igoBoard_Node̋u܂B
	 */
	private void setEmpties(IgoBoard igoBoard) {
		Property property = node_.getProperty(SgfId.ADD_EMPTY);
		if (property == null) {
			return;
		}

		// zΒǉp̏W쐬
		Set<SgfPoint> blacks = new HashSet<SgfPoint>();
		Set<SgfPoint> whites = new HashSet<SgfPoint>();
		
		String[] values = property.getStrings();
		for (SgfPoint sgfPoint : SgfPoint.parse(nodeList_.getSize(), values)) {
			SgfColor color = igoBoard.removeStone(sgfPoint);
			if (SgfColor.BLACK.equals(color)) {
				blacks.add(sgfPoint);
			} else if (SgfColor.WHITE.equals(color)) {
				whites.add(sgfPoint);
			}
		}
		
		inheritPointMap_.put(
				REMOVE_SETUP_BLACK, 
				blacks.isEmpty() 
					? SgfPoint.EMPTY_ARRAY 
					: blacks.toArray(new SgfPoint[blacks.size()]));
		inheritPointMap_.put(
				REMOVE_SETUP_WHITE,
				whites.isEmpty() 
					? SgfPoint.EMPTY_ARRAY
					: whites.toArray(new SgfPoint[whites.size()]));
	}

	/**
	 * igoBoardcolor̐΂ł܂B
	 */
	private void moveStone(MarkupIgoBoard igoBoard) {
		composeMove();
		if (!move_.isValid()) {
			return;
		}
	    
	    // AQn}̏Wւ̒ǉ
		capturedStones_.clear();
		capturedStones_.addAll(Arrays.asList(igoBoard.moveStone(move_)));
	}
	
	/**
	 * ݂{@link #node_}璅\܂B
	 * ʂ{@link #move_}ɐݒ肳܂B
	 */
	private void composeMove() {
		move_ = Stone.NULL_STONE;
		for (SgfColor sgfColor : new SgfColor[]{SgfColor.BLACK, SgfColor.WHITE}) {
			Property property = node_.getProperty(sgfColor.moveType());
			if (property == null) {
				// 肪
				continue;
			}
			
			// ꏊ̕]
		    SgfPoint point = SgfPoint.parseMoveQuietly(
		    		nodeList_.getSize(), property.getString());
		    if (point == null) {
		    	continue;
		    }
		    
		    // ԍ̕]
		    Integer mn = composeMoveNumber();

		    // lʎԂ̕]
		    Double tl = composeTimeLapsed(sgfColor);
		    
	    	move_ = new Stone(sgfColor, point, mn, tl);
	    	return;
		}
	}
	
	/**
	 * {@link #node_}̒ԍ߂ĕԂ܂B
	 * @return {@link #node_}̒ԍ
	 */
	private Integer composeMoveNumber() {
	    Integer mn = new Integer(1); // ȂƂ1ӂB

	    // Õm[h𒅎肪܂łǂ肻甲o
	    // CNgB
	    NodeEntity lastMoveNode = previous_;
	    while (lastMoveNode != null) {
	    	if (!lastMoveNode.move_.isValid()) {
	    		lastMoveNode = lastMoveNode.previous_;
	    		continue;
	    	}
	    	Integer lastMn = lastMoveNode.move_.getNumber();
	    	if (lastMn != null) {
	    		mn = new Integer(lastMn.intValue() + 1);
	    	}
    		break;
	    }
	    
	    // MNvpeB㏑B
	    Property property = node_.getProperty(SgfId.SET_MOVE_NUMBER);
    	if (property != null) {
    	    try {
    	    	mn = new Integer(property.getString());
    	    } catch (NumberFormatException ignore) {
    	        //TODO O
    	        ignore.printStackTrace();
    	    }
	    }
    	return mn;
	}
	
	/**
	 * {@link #node_}̍lʎԂ߂ĕԂ܂B
	 * @return {@link #node_}̍lʎԁA߂Ȃꍇnull
	 */
	private Double composeTimeLapsed(SgfColor sgfColor){
		// TLvpeBꍇ͂D悳B
		Property property = node_.getProperty(SgfId.TIME_LAPSED);
		if (property != null) {
			TimeLapsed tl = TimeLapsed.parseQuietly(property.getString());
			if (tl != null) {
				return new Double(tl.getTimeLapsed());
			}
		}
		
		property = node_.getProperty(sgfColor.timeType());
		if (property == null) {
			return null;
		}
		
		SgfReal nowTimeLeft = SgfReal.parseQuietly(property.getString());
		if (nowTimeLeft == null) {
			return null;
		}
		
	    // Õm[hWLBL܂łǂ
		SgfReal lastTimeLeft = null;
	    NodeEntity lastMoveNode = previous_;
	    while (lastMoveNode != null) {
	    	property = lastMoveNode.node_.getProperty(sgfColor.timeType());
	    	if (property != null) {
				lastTimeLeft = SgfReal.parseQuietly(property.getString());
	    		break;
	    	}
    		lastMoveNode = lastMoveNode.previous_;
	    }
	    
	    if (lastTimeLeft == null) {
		    // O̎Ԃ킩ȂƂTMvpeBTB
	    	NodeEntity entity = findAbove(PropertyType.GAME_INFO);
	    	if (entity == null) {
	    		return null;
	    	}
	    	
	    	property = entity.node_.getProperty(SgfId.TIMELIMIT);
	    	if (property == null) {
	    		return null;
	    	}
	    	lastTimeLeft = SgfReal.parseQuietly(property.getString());
	    	if (lastTimeLeft == null) {
	    		return null;
	    	}
	    }
	    
	    // lԂɂ͎̂vǉꂽꍇ
	    // łnullԂB
	    double tl = lastTimeLeft.getReal() - nowTimeLeft.getReal();
	    if (tl < 0) {
	    	return null;
	    }
    	return new Double(tl);
	}

    /**
     * {@link #inheritableDecoration_}inheritableDecorationɒuA
     * nodeɊ܂܂Ăpǉ܂B
     */
	private void updateInheritedDecoration() {
		if (previous_ == null) {
		    inheritableDecoration_.clear();
		} else {
			inheritableDecoration_.set(previous_.inheritableDecoration_);
		}
		Property[] properties = node_.getProperties();
		for (int i = 0; i < properties.length; ++i) {
            inheritableDecoration_.set(properties[i]);
        }
    }


	// Getter
	
    /**
     * EntitÿOEntityԂ܂B
     * ݂Ȃ(܂A擪̏ꍇnullԂ܂B
     */
    public NodeEntity getPrevious() {
        return previous_;
    }
    
	/**
	 * bvĂNodeԂ܂B<br>
	 * null͌ĕԂ܂B
	 */
	public Node getNode() {
		return node_;
	}
	/**
	 * {@link org.unitarou.sgf.Node}Ɛec[̃CX^XԂ܂B
	 */
	public NodeTree getNodeTree() {
	    return nodeTree_;
	}

	
	/**
	 * ݑIĂ镪}̃CfbNXԂ܂B
	 * 򂪖ꍇ-1Ԃ܂B
	 * @throws IllegalStateException ω}̐ƃCfbNXɕsꍇB 
	 */
	public int getSelectedVariationIndex() {
		int size = nodeTree_.getVariationSize();
		if ((size == 0 && selectedVariationIndex_ != NO_VARIATION) 
				|| (size <= selectedVariationIndex_)
				|| (size != 0 && selectedVariationIndex_ == NO_VARIATION)) {
			throw new IllegalStateException("Variation miss match. Variation size: " + size + ", index:" + selectedVariationIndex_); //$NON-NLS-1$ //$NON-NLS-2$
		}
		return selectedVariationIndex_;
	}
	
	/**
	 * {@link #selectedVariationIndex_}`FbNɂ̂܂ܕԂ܂B
	 * @return
	 */
	int getUnmodifiedSelectVariationIndex() {
		return selectedVariationIndex_;
	}
	
	/**
	 * ̃m[h̋ǖʂ\ՂԂ܂B<br>
	 * <b></b>Ղ{@link NodeList}Ŝ1łB
	 * ʂNodeEntity{@link #getIgoBoard()}ĂяoƁA
	 * ̃\bhԂ{@link IgoBoard}̓eς܂B
	 * @return
	 */
	public IgoBoard getIgoBoard() {
		if (!nodeList_.contains(this)) {
			if (nodeList_.size() == 0) {
				return null;
			}
			int index = Math.min(nodeList_.getPositionIndex(), nodeList_.size() - 1);
			return nodeList_.updateIgoBoard(nodeList_.get(index));
		}
		return nodeList_.updateIgoBoard(this);
	}
	
	/**
	 * ̃CX^Xێ{@link Node}XVŌĂяo܂B
	 * ŕێĂ΂inheritȑXV܂B
	 */
	public void update() {
		nodeList_.update(this);
	}
	
	/**
	 * ݂̒Ԃ܂B
	 * null͕Ԃ܂B
	 */
	public Stone getMove() {
	    return move_;
	}
	

	/**
	 * CX^X{@link InheritableMarker}Ԃ܂B
	 */
	public InheritableMarker getInheritableDecoration() {
		//TODO ܂肨sV͂悭Ȃ
		nodeList_.updateIgoBoard(this);
	    return inheritableDecoration_;
	}

	/** 
	 * (킸)SAQn}Ԃ܂B
	 */
	public Stone[] getCaptured() {
	    return capturedStones_.toArray(new Stone[capturedStones_.size()]);
	}

	/**
     * ݂̒őłグꂽAQn}Ԃ܂B
     */
    public Stone[] getLastCaptured() {
        Set<Stone> captured = new HashSet<Stone>(capturedStones_);
        NodeEntity previous = getPrevious();
        if (previous != null) {
            captured.removeAll(previous.capturedStones_);
        }
        return captured.toArray(new Stone[captured.size()]);
    }
	

    /**
	 * ω}̐擪̃m[h̔zԂ܂B
	 * ω}݂Ȃꍇ͒O̔zԂ܂B
	 * ω}Children, Sibling̃^Cvɍ킹č쐬Ă܂B
	 */
	public NodeEntity[] getVariations() {
	    GameTree[] children = nodeTree_.getVariation();
	    NodeEntity[] ret = new NodeEntity[children.length];
	    for (int i = 0; i < ret.length; ++i) {
            ret[i] = new NodeEntity(
                    		children[i].getSequence().getFirst(),
                    		children[i],
                    		nodeList_,
                    		previous_);
            ret[i].composeMove();
        }
	    return ret;
	}
	
	/**
	 * ΂dȂĂȂǕsȃ|Cg̏WԂ܂B
	 * @return ߂l͕ҏWsłB
	 */
	public Set<SgfPoint> getInvalid() {
		return Collections.unmodifiableSet(invalidPoints_);
	}

	
	/**
     * propertyType܂񂾃m[h㗬܂ŌČꍇ͂NodeEntityԂ܂B
	 * @return ݂ȂꍇnullԂ܂B
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
     */
    public NodeEntity findAbove(PropertyType propertyType) {
        ArgumentChecker.throwIfNull(propertyType);
        NodeEntity entity = this;
        while(entity != null) {
        	if (entity.getNode().contains(propertyType)) {
        		return entity;
        	}
            entity = entity.getPrevious();
        }
        return entity;
    }
    
    /**
     * ̃CX^Xbv{@link NodeView}Ԃ܂B
     * @return
     */
    NodeView getNodeView() {
    	if (nodeView_ == null) {
    		nodeView_ = new NodeView(this);
    	}
    	return nodeView_;
    }
    
    /**
     * SGFȂтɃLmV^̃[ɑāA
     * Ɏw肵sgfIdCX^XbvĂ{@link Node}
     * ǉłꍇtrueԂ܂B<br>
     * ̓Iɂ́F
     * <ol>
     * <li>[SGF]{@link PropertyType#MOVE}̑{@link SgfId}w肵ꍇA
     * @  {@link SgfId#WHITE}܂{@link SgfId#BLACK}鎞̂trueԂB</li>
     * <li>[SGF]{@link PropertyType#ROOT}̑{@link SgfId}w肵ꍇA
     *     bvĂ{@link Node}{@link org.unitarou.sgf.RootGameTree}̍ŏ{@link Node}
     *     ̏ꍇɂ̂trueԂB</li>
     * <li>[SGF]{@link PropertyType#SETUP}̑{@link SgfId}w肵ꍇA
     *     bvĂ{@link Node}{@link PropertyType#MOVE}̑{@link SgfId}
     *     Ȃꍇɂ̂trueԂB</li>
     * <li>[LmV^]{@link GameType#PROBLEM}̏ꍇAǉ\{@link SgfId}͎̂ƂF
     *     <ol>
     *     <li>Move PropertiesJeS[SS</li>
     *     <li>Setup PropertiesJeS[SS</li>
     *     <li>Node Annotation PropertiesJeS[̓{@link SgfId#COMMENT}
     *         {@link SgfId#GOOD_FOR_WHITE}A{@link SgfId#GOOD_FOR_BLACK}</li>
     *     <li>Move Annotation PropertiesJeS[{@link SgfId#BAD_MOVE}̂</li>
     *     <li>Markup PropertiesJeS[S</li>
     *     <li>Root PropertiesJeS[S</li>
     *     <li>Game Info PropertiesJeS[̓{@link SgfId#BLACK_RANK}A
     *         {@link SgfId#COPYRIGHT}A{@link SgfId#DATE}A{@link SgfId#GAME_COMMENT}A
     *         {@link SgfId#GAME_NAME}A{@link SgfId#RESULT}A{@link SgfId#SOURCE}A
     *         {@link SgfId#TIMELIMIT}A{@link SgfId#USER}</li>
     *     <li>Timing PropertiesJeS[͑Sėps</li>
     *     <li>Miscellaneous PropertiesJeS[{@link SgfId#VIEW}̂</li>
     *     </ol>
     * <li>[LmV^]{@link GameType#DRILL}̏ꍇAǉ\{@link SgfId}͎̂ƂF
     *     <ol>
     *     <li>Move PropertiesJeS[͑Sėps</li>
     *     <li>Setup PropertiesJeS[͑Sėps</li>
     *     <li>Node Annotation PropertiesJeS[͑Sėps</li>
     *     <li>Move Annotation PropertiesJeS[͑Sėps</li>
     *     <li>Move PropertiesJeS[͑Sėps</li>
     *     <li>Setup PropertiesJeS[͑Sėps</li>
     *     <li>Markup PropertiesJeS[͑Sėps</li>
     *     <li>Root PropertiesJeS[{@link SgfId#SIZE}đSĉ</li>
     *     <li>Game Info PropertiesJeS[{@link SgfId#BLACK_RANK}A
     *         {@link SgfId#GAME_COMMENT}A{@link SgfId#GAME_NAME}̂</li>
     *     <li>Timing PropertiesJeS[͑Sėps</li>
     *     <li>Miscellaneous PropertiesJeS[͑Sėps</li>
     *     <li>IWi{@link SgfId#INPUT_FILES}A{@link SgfId#PROBLEM_PROPERTIES}</li>
     *     </ol>
     * </li>
     * </ol>
     * @param sgfId
     * @return
     * @throws org.unitarou.lang.NullArgumentException  <code>null</code>̏ꍇB
     */
    public boolean appends(SgfId sgfId) {
    	ArgumentChecker.throwIfNull(sgfId);
    	switch (sgfId.propertyType()) {
    	case MOVE:
    		if ( (null == node_.getProperty(SgfId.WHITE)) 
    				&& (null == node_.getProperty(SgfId.BLACK))) {
    			return false;
    		}
    		break;
    		
    	case ROOT:
    		if (nodeTree_.getRootGameTree().getSequence().getFirst() != node_) {
    			return false;
    		}
    		break;
    		
    	case SETUP:
    		if (node_.contains(PropertyType.MOVE)) {
    			return false;
    		}
    		break;
    		
  		default:
  			// ȂB
    	}
    	
    	
    	
    	GameType gameType = nodeTree_.getRootGameTree().getGameType();
    	switch (gameType) {
    	case GAME:
    		return true;
    		
    	case PROBLEM:
    		if (!idSet4Problem_s_.contains(sgfId)) {
    			return false;
    		}
    		SgfColor color = BasicFinder.firstMove(nodeTree_.getRootGameTree());
    		if (SgfId.GOOD_FOR_BLACK.equals(sgfId) && color != SgfColor.BLACK) {
    			return false;
    		}
    		if (SgfId.GOOD_FOR_WHITE.equals(sgfId) && color != SgfColor.WHITE) {
    			return false;
    		}
    		return true;
    		
    	case DRILL:
    		return idSet4Drill_s_.contains(sgfId);
    		
    	default:
    		assert false : "Unknown GameType: " + gameType; //$NON-NLS-1$
    		log_s_.warn("Unknown GameType: " + gameType); //$NON-NLS-1$
    	}
    	return true;
    }

    
    /**
     * {@link #node_}bvĂꍇtrueԂ܂B
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
	@Override
	public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if ((obj == null) || !obj.getClass().equals(this.getClass())) {
            return false;
        }
        return node_ == ((NodeEntity)obj).node_;
    }
    
    
    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
	@Override
	public int hashCode() {
        return node_.hashCode();
    }
}
