/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.model;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import org.unitarou.sgf.Node;
import org.unitarou.sgf.Property;
import org.unitarou.sgf.PropertyType;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.type.GameType;
import org.unitarou.sgf.type.SgfColor;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.sgf.type.SgfSize;
import org.unitarou.sgf.util.SgfPointType;
import org.unitarou.sgf.util.Stone;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.model.board.IgoBoard;
import org.unitarou.yukinoshita.model.board.IgoBoardView;
import org.unitarou.yukinoshita.model.board.InheritableMarker;
import org.unitarou.yukinoshita.model.board.OverblockMarker;

/**
 * {@link NodeEntity}View
 * Kvȃf[^݂̂𔲂o悤ɃbvNXłB
 * 
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public final class NodeView {

    private final NodeEntity nodeEntity_;
	
    private final RootView rootView_;
	
	/**
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
	 */
	public NodeView(NodeEntity nodeEntity) {
		super();
		ArgumentChecker.throwIfNull(nodeEntity);
		nodeEntity_ = nodeEntity;
		rootView_ = new RootView(nodeEntity_.getNodeTree().getRootGameTree());
	}
	
	/**
	 * @return null͕Ԃ܂B
	 */
	public RootView getRootView() {
		return rootView_;
	}
	

	public boolean isSameNodeEntity(NodeView nodeView) {
	    ArgumentChecker.throwIfNull(nodeView);
	    return nodeEntity_.equals(nodeView.nodeEntity_);
	}

	/**
	 * ̃m[h̋ǖʂ\ՂԂ܂B<br>
	 * <b></b>Ղ{@link NodeList}Ŝ1łB
	 * ʂNodeEntity{@link #getIgoBoard()}ĂяoƁA
	 * ̃\bhԂ{@link IgoBoard}̓eς܂B
	 * @return
	 */
	public IgoBoardView getIgoBoard() {
		return nodeEntity_.getIgoBoard();
	}
	
	public InheritableMarker getInheritableDecoration() {
	    return nodeEntity_.getInheritableDecoration();
	}
	
	/**
	 * propertyTypȇ{@link Property}{@link org.unitarou.sgf.Node}
	 * ꍇtrueԂ܂B
	 * @param propertyType
	 * @return
	 */
	public boolean containsPropertyType(PropertyType propertyType) {
		ArgumentChecker.throwIfNull(propertyType);
		for (SgfId sgfId : nodeEntity_.getNode().getPropertyIds()) {
			if (sgfId.propertyType() == propertyType) {
				return true;
			}
		}
		return false;
	}

	/** 
	 * ̃m[hɑ݂ω}̐Ԃ܂B
	 * ω}݂Ȃꍇ0Ԃ܂B
	 */
	public int variationSize() {
		return nodeEntity_.getNodeTree().getVariationSize();
	}
	
	/**
	 * ω}̐擪̃m[h̔zԂ܂B
	 * 0Ԗڂ̗vf͖{łB
	 * ω}݂Ȃꍇ͒O̔zԂ܂B
	 * ω}Children, Sibling̃^Cvɍ킹č쐬Ă܂B
	 */
	public NodeView[] getVariations() {
	    NodeEntity mle[] = nodeEntity_.getVariations();
	    NodeView[] ret = new NodeView[mle.length];
	    for (int i = 0; i < ret.length; ++i) {
            ret[i] = mle[i].getNodeView();
        }
	    return ret;
	}
	
	
	/**
	 * ݂̒Ԃ܂B
	 * null͕Ԃ܂B
	 */
	public Stone getMove() {
	    return nodeEntity_.getMove();
	}

 	/**
 	 * idɕRÂPropertyԂ܂B<br>
 	 * ȂꍇnullԂ܂B
 	 * ܂ÃCX^XێĂProperty𕡎ʂĕԂ܂̂ŁA
 	 * ʂɑĂÃCX^X͉e󂯂܂B
 	 * {@link #addProperty(Property)}ŕύXsĂB
	 * @throws org.unitarou.lang.NullArgumentException id<code>null</code>̏ꍇB
 	 */
	public Property getProperty(SgfId id) {
		return nodeEntity_.getNode().getProperty(id);
	}
	
	/**
	 * typeL[ƂăvpeB܂B
	 * {@link #getProperty(SgfId)}Ƃ̈Ⴂ́A
	 * ̃\bh͂̃m[hɊYvpeBꍇA
	 * V[PX̏㗬ɌɍsƂłB
	 * @́A܂{@link SgfId#propertyType()}܂񂾃m[h
	 * ݂̃m[h㗬ɂǂ܂BčŏɌm[hA
	 * type܂B
	 *  
	 * @return ݂ȂꍇnullԂ܂B
	 * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
	 */
	public Property findProperty(SgfId sgfId) {
	    ArgumentChecker.throwIfNull(sgfId);
	    
	    NodeEntity neGameInfo_ = nodeEntity_.findAbove(sgfId.propertyType());
	    if (neGameInfo_ == null) {
	        return null;
	    }
	    
	    return neGameInfo_.getNode().getProperty(sgfId);
	}


    /**
     * ω}childrenX^C(V[PX̍Ōŕω}ʒm)
     * \ƂtrueԂ܂B
     * ܂siblingsX^C(V[PX̍ŏŕω}ʒm)
     * \ƂfalseԂ܂B
     */
	public boolean isChildrenStyle() {
	    return nodeEntity_.getNodeTree().isChildrenStyle();
	}
	
    /**
     * ω}ǉۂ̐FԂ܂B
     * ω}̐eqEZ^ɑΉĎ̒̐FԂ܂B 
     * fs\̏ꍇ<code>null</code>Ԃ܂B
     */
	public SgfColor getOppositeColor() {
        return isChildrenStyle()
        			? getNextMoveColor()
        	        : getMove().getColor();
    }

	/**
	 * ݑIĂ镪}̃CfbNXԂ܂B
	 * 򂪖ꍇ-1Ԃ܂B
	 */
    public int getSelectedVariationIndex() {
        return nodeEntity_.getSelectedVariationIndex();
    }
    
    /**
     * @return
     */
    public int getVariationHash() {
        NodeEntity previous = nodeEntity_;
        int hash = NodeEntity.NO_VARIATION;
        while (previous != null) {
            int index = previous.getSelectedVariationIndex();
            if (index != NodeEntity.NO_VARIATION) {
                hash += index;
            }
            previous = previous.getPrevious();
        }
        return hash;
    }

    
    public OverblockMarker getOverblockMarker() {
    	return new OverblockMarker(nodeEntity_.getNode());
    }

    public SgfSize getSize() {
        return nodeEntity_.getNodeTree().getSize();
    }

	/**
	 * AQn}̏W𕡎ʂĕԂ܂B
	 * @param playerColor vC[̐F(Ȃ甒΂AȂ獕΂Ԃ)
	 * @return
	 */
	public List<Stone> getCaptured(SgfColor playerColor) {
		ArgumentChecker.throwIfNull(playerColor);
		
		return nodeEntity_.getIgoBoard().getCaptured(playerColor);
	}

	/**
	 * ɁuׂvԂԂ܂B
	 * ܂ۂɃf[^Ƃđ݂鎟̎ԂƂ͖֌WɁuׂvԂԂ܂B
	 * Ďۂɂ(ˍ)ƂȂĂĂ(m[h́eˁf)A
	 * ߂l{@link SgfColor#WHITE}ɂȂ܂
	 * 
	 * m[hɒ肪ꍇ͈Õm[h܂B
	 * z΃m[h̏ꍇ(܂̓m[ĥڂĔz΃m[hɂȂꍇ)
	 * PL^OTĂ̒lD悵ĕԂ܂B
	 * PL^O݂ȂgǕs\̏ꍇnullԂ܂B
	 * 
	 * Xg̐擪܂Ŗ߂Ăǂ炩ȂꍇnullԂ܂B<br>
	 */
    public SgfColor getNextMoveColor() {
        NodeEntity entity = nodeEntity_;
        do {
        	if (entity.getNode().contains(PropertyType.SETUP)) {
                Property property = entity.getNode().getProperty(SgfId.PLAYER_TO_PLAY);
                if (property != null) {
                    return SgfColor.parseQuietly(property.getString());
                }
                return null;                
        	}
            SgfColor curColor = entity.getMove().getColor();
            if (curColor != null) {
                return curColor.opposite();
            }
            entity = entity.getPrevious(); 
        } while(entity != null);
        return null;
    }
    
	/**
	 * @return
	 */
	public boolean isFirst() {
		return nodeEntity_.getPrevious() == null;
	}
    /**
     * m[hXg̖łƂtrueԂ܂B
     */
    public boolean isLast() {
        NodeTree nodeTree = nodeEntity_.getNodeTree();
        return (nodeTree.getGameTree().getChildrenSize() == 0) 
        		&& nodeTree.isLastNode();
    }
    

    /**
     * poinẗʒu({@link SgfPointType#WHITE})
     * ({@link SgfPointType#BLACK})A
     * Ζ({@link SgfPointType#EMPTY})̂ꂩԂ܂B
     * ܂֎~n_̏ꍇ{@link SgfPointType#FORBIDDEN}Ԃ܂B
     * 
     * @throws org.unitarou.lang.NullArgumentException pointnull̏ꍇB
     */
    public EnumSet<SgfPointType> getPointTypes(SgfPoint point) {
        ArgumentChecker.throwIfNull(point);
        
        EnumSet<SgfPointType> ret = EnumSet.noneOf(SgfPointType.class);
        IgoBoardView igoBoard = nodeEntity_.getIgoBoard();
        if (igoBoard == null) {
            ret.add(SgfPointType.FORBIDDEN);
            ret.add(SgfPointType.EMPTY);
            return ret;
        }
        Stone stone = igoBoard.getStone(point);
        if (stone == null) {
    		ret.add(SgfPointType.EMPTY);
            
        } else if (SgfColor.BLACK.equals(stone.getColor())) {
		    ret.add(SgfPointType.BLACK);
        
        } else if (SgfColor.WHITE.equals(stone.getColor())) {
		    ret.add(SgfPointType.WHITE);
            
        } else {
            assert false : "Known types are found:" + stone.getColor(); //$NON-NLS-1$
        }

        Stone move = nodeEntity_.getMove();
        if (move.isValid() 
		        && igoBoard.isForbiddenMove(
		                point, 
		                move.getColor().opposite(), 
		                move.getPoint(), 
		                nodeEntity_.getLastCaptured())) {
		    ret.add(SgfPointType.FORBIDDEN);
		}
		return ret;
    }

    /**
     * pointɂω}ǉ邽߂̏(΂̗LE\̗L)Ԃ܂B
     * @return {@link SgfPointType#EMPTY}{@link SgfPointType#FORBIDDEN}
     *          ꂩԂ܂B
     * 
     * @throws org.unitarou.lang.NullArgumentException null̏ꍇB
     */
    public SgfPointType getPointTypes4Variation(SgfPoint point) {
        ArgumentChecker.throwIfNull(point);
        if (isChildrenStyle()) {
        	EnumSet<SgfPointType> set = getPointTypes(point); 
            return set.contains(SgfPointType.FORBIDDEN) 
            		? SgfPointType.FORBIDDEN 
            		: SgfPointType.EMPTY;
        }
        
        
        // ݂1OIgoBoard璅\_ݒ肷B
        // ̃m[h[gm[h̏ꍇA
        // OIgoBoard̓ꏊɐ΂ꍇ͕s
        NodeEntity previous = nodeEntity_.getPrevious();
        if (previous == null ||
                previous.getIgoBoard().getStone(point) != null) {
            return SgfPointType.FORBIDDEN;
        }
        
        // L𖞂
        // Õm[hɒ肪ꍇ͒\B
        if (!previous.getMove().isValid()){
            return SgfPointType.EMPTY;
        }
        
        
        // ŌIgoBoard#isForbiddenMoveŃ`FbNB
        return previous.getIgoBoard().isForbiddenMove(
                		point,
                		previous.getMove().getColor().opposite(),
                		previous.getMove().getPoint(),
                		previous.getLastCaptured()
                		)
                	? SgfPointType.FORBIDDEN : SgfPointType.EMPTY;
        
    }


	/**
	 * ΂dȂĂȂǕsȃ|Cg̏WԂ܂B
	 * @return ߂l͕ҏWsłB
	 */
	public Set<SgfPoint> getInvalid() {
		return nodeEntity_.getInvalid();
	}

    /**
     * SGFȂтɃLmV^̃[ɑāA
     * Ɏw肵sgfIdCX^XbvĂ{@link Node}
     * ǉłꍇtrueԂ܂B<br>
     * ̓Iɂ́F
     * <ol>
     * <li>[SGF]{@link PropertyType#MOVE}̑{@link SgfId}w肵ꍇA
     * @  {@link SgfId#WHITE}܂{@link SgfId#BLACK}鎞̂trueԂB</li>
     * <li>[SGF]{@link PropertyType#ROOT}̑{@link SgfId}w肵ꍇA
     *     bvĂ{@link Node}{@link org.unitarou.sgf.RootGameTree}̍ŏ{@link Node}
     *     ̏ꍇɂ̂trueԂB</li>
     * <li>[SGF]{@link PropertyType#SETUP}̑{@link SgfId}w肵ꍇA
     *     bvĂ{@link Node}{@link PropertyType#MOVE}̑{@link SgfId}
     *     Ȃꍇɂ̂trueԂB</li>
     * <li>[LmV^]{@link GameType#PROBLEM}̏ꍇAǉ\{@link SgfId}͎̂ƂF
     *     <ol>
     *     <li>Move PropertiesJeS[SS</li>
     *     <li>Setup PropertiesJeS[SS</li>
     *     <li>Node Annotation PropertiesJeS[̓{@link SgfId#COMMENT}
     *         {@link SgfId#GOOD_FOR_WHITE}A{@link SgfId#GOOD_FOR_BLACK}</li>
     *     <li>Move Annotation PropertiesJeS[{@link SgfId#BAD_MOVE}̂</li>
     *     <li>Markup PropertiesJeS[S</li>
     *     <li>Root PropertiesJeS[S</li>
     *     <li>Game Info PropertiesJeS[̓{@link SgfId#BLACK_RANK}A
     *         {@link SgfId#COPYRIGHT}A{@link SgfId#DATE}A{@link SgfId#GAME_COMMENT}A
     *         {@link SgfId#GAME_NAME}A{@link SgfId#RESULT}A{@link SgfId#SOURCE}A
     *         {@link SgfId#TIMELIMIT}A{@link SgfId#USER}</li>
     *     <li>Timing PropertiesJeS[͑Sėps</li>
     *     <li>Miscellaneous PropertiesJeS[{@link SgfId#VIEW}̂</li>
     *     </ol>
     * <li>[LmV^]{@link GameType#DRILL}̏ꍇAǉ\{@link SgfId}͎̂ƂF
     *     <ol>
     *     <li>Move PropertiesJeS[͑Sėps</li>
     *     <li>Setup PropertiesJeS[͑Sėps</li>
     *     <li>Node Annotation PropertiesJeS[͑Sėps</li>
     *     <li>Move Annotation PropertiesJeS[͑Sėps</li>
     *     <li>Move PropertiesJeS[͑Sėps</li>
     *     <li>Setup PropertiesJeS[͑Sėps</li>
     *     <li>Markup PropertiesJeS[͑Sėps</li>
     *     <li>Root PropertiesJeS[{@link SgfId#SIZE}đSĉ</li>
     *     <li>Game Info PropertiesJeS[{@link SgfId#BLACK_RANK}A
     *         {@link SgfId#GAME_COMMENT}A{@link SgfId#GAME_NAME}̂</li>
     *     <li>Timing PropertiesJeS[͑Sėps</li>
     *     <li>Miscellaneous PropertiesJeS[͑Sėps</li>
     *     <li>IWi{@link SgfId#INPUT_FILES}A{@link SgfId#PROBLEM_PROPERTIES}</li>
     *     </ol>
     * </li>
     * </ol>
     * @param sgfId
     * @return
     * @throws org.unitarou.lang.NullArgumentException  <code>null</code>̏ꍇB
	 * @see org.unitarou.yukinoshita.model.NodeEntity#appends(org.unitarou.sgf.SgfId)
     */
	public boolean appends(SgfId sgfId) {
		return nodeEntity_.appends(sgfId);
	}
	
	
}

