/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.model.board;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import org.unitarou.lang.NullArgumentException;
import org.unitarou.lang.UEnum;
import org.unitarou.sgf.Property;
import org.unitarou.sgf.PropertyType;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.ValueType;
import org.unitarou.sgf.type.Label;
import org.unitarou.sgf.type.SgfLine;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.sgf.type.SgfSize;
import org.unitarou.sgf.util.SgfRectangle;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.model.NodeView;
import org.unitarou.yukinoshita.view.jface.board.BlockPainter;

/**
 * ՂubNɋ؂Ĉȉ̍ڂ𓱏oNXłB<br>
 * <ol>
 * <li>ubÑTCYBTCY͈ȉ̍ڂ瓱oF
 *   <ol>
 *     <li>Ղ̃TCY(PXHAXHAetc...)</li>
 *     <li>VWɂ\</li>
 *     <li>Wx̃TCY</li>
 *   </ol>
 * </li>
 * <li>ĕ`悷ׂubNBȉ̍ڂ瓱oF
 *   <ol>
 *     <li>΂̕ω(AAQn}AȊOɊ_)</li>
 *     <li>L̕ω</li>
 *     <li>etc...̃NX瓱ô͓̂ŁAOubNw肳邾ɂƂǂ߂B</li>
 *   </ol>
 * </li>
 * </ol>
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
public class BlockedBoardDesigner {
	/** 
	 * `ΏۂƂ݂ȂvpeB̌^̔złB
	 */
    static private final SgfId[] drawTypes_s_
    		= new SgfId[]{SgfId.MARK_WITH_X, SgfId.CIRCLE,
            				SgfId.SQUARE, SgfId.TRIANGLE,
            				SgfId.SELECTED, SgfId.TERRITORY_WHITE,
            				SgfId.TERRITORY_BLACK, SgfId.DIM_POINTS};

    
	/** 
	 * ACêʒuƂ̎ނi[܂B
	 */
    private final Map<SgfPoint, Set<SgfId>> markerMap_;

    /** p^Cv̑񂪓Ă܂B*/
	private final InheritableMarker inheritableMarker_;
	
    /** C⃉xȂǁAubNɂ܂񂪓Ă܂B*/
	private final OverblockMarker overblockMarker_;
    
    
    /**
     * `͈͂vZ邽߂ɕێA`IuWFNgłB
     */
    private final List<BlockPainter> blockPainters_;
        
    /**
     * ݕ`ΏۂƂȂĂNodeViewێ܂B
     */
    private NodeView nodeView_;
    
    
    /**
     * ^̍ĕ`̈ێ܂B 
     */
    private final Set<SgfPoint> durablePaintings_;
    
    /**
     * ꎞ^̍ĕ`̈ێ܂B 
     */
    private final Set<SgfPoint> transientPaintings_;

    /**
     * ȑÖꎞ^ĕ`̈ێ܂B 
     */
    private final Set<SgfPoint> lastTransientPaintings_;
    
    /**
     * VWŎw肳ꂽ`w|Cg̏Wێ܂B
     */
    private final Set<SgfPoint> vwPoints_;
    
    /**
     * 
     */
    public BlockedBoardDesigner(IgoBoard igoBoard) {
        super();
        markerMap_ = new TreeMap<SgfPoint, Set<SgfId>>();
		inheritableMarker_ = new InheritableMarker();
		overblockMarker_ = new OverblockMarker();
        
        
        blockPainters_ = new ArrayList<BlockPainter>();
        nodeView_ = null;
        durablePaintings_ = new TreeSet<SgfPoint>();
        transientPaintings_ = new TreeSet<SgfPoint>();
        lastTransientPaintings_ = new TreeSet<SgfPoint>();
        vwPoints_ = new HashSet<SgfPoint>(SgfSize.DEFAULT.height() * SgfSize.DEFAULT.width());
    }
    
    /**
     * blockPaintero^܂B<br>
     * {@link #update(NodeView)}ɓŗp܂B
     * 
     * @param blockPainter
     * @throws NullArgumentException <code>null</code>̏ꍇ
     */
    public void addBlockPainter(BlockPainter blockPainter) {
    	ArgumentChecker.throwIfNull(blockPainter);
    	blockPainters_.add(blockPainter);
    }
    
    /**
     * blockPainter폜܂B<br>
     * 
     * @param blockPainter
     * @throws NullArgumentException <code>null</code>̏ꍇ
     */
    public void removeBlockPainter(BlockPainter blockPainter) {
    	ArgumentChecker.throwIfNull(blockPainter);
    	blockPainters_.remove(blockPainter);
    }

    /**
     * pointsgfTypeݒ肵܂B
     * 
     * @throws NullArgumentException ̉ꂩłnull̏ꍇB
     */
    private void addMarker(SgfPoint point, SgfId sgfType) {
        ArgumentChecker.throwIfNull(point, sgfType);

        Set<SgfId> markerSet = markerMap_.get(point);
        if (markerSet == null) {
            markerSet = new TreeSet<SgfId>();//TODO \Rp[^KvB
            markerMap_.put(point, markerSet);
        }
        markerSet.add(sgfType);
    }


    /**
     * ĕ`̑ΏۂƂpointo^܂B
     * ͈ȑO̕`悪ȂȂꍇɌĂяo܂B
     * 
     * @throws NullArgumentException null̏ꍇB
     */
    public void addRedrawPoint(SgfPoint point) {
        ArgumentChecker.throwIfNull(point);
        transientPaintings_.add(point);
    }


    /**
     * pointɂsgfType̗vf`揇()ŕԂ܂B
     * ݂Ȃꍇ͗vf[̔zԂ܂B
     * 
     * @throws NullArgumentException pointnull̏ꍇB
     */
    public SgfId[] getMarkers(SgfPoint point) {
    	ArgumentChecker.throwIfNull(point);

    	Set<SgfId> markerSet = markerMap_.get(point);
        if (markerSet == null) {
            return new SgfId[0];
        }

        return markerSet.toArray(new SgfId[markerSet.size()]);
    }
    

    /**
     * sgfType݂Point̃XgԂ܂B
     */
    public SgfPoint[] getMarkerPoints(SgfId sgfType) {
        if (sgfType == null) {
            throw new NullArgumentException("sgfType must not be null."); //$NON-NLS-1$
        }
        List<SgfPoint> ret = new ArrayList<SgfPoint>(markerMap_.size());
        for (Map.Entry<SgfPoint, Set<SgfId>> entry : markerMap_.entrySet()) {
            Set<SgfId> markers = entry.getValue();
            if ((markers != null) && markers.contains(sgfType)) {
                ret.add(entry.getKey());
            }
        }
        return ret.toArray(new SgfPoint[ret.size()]);
    }

    
    public boolean hasOverblockMark() {
        return !overblockMarker_.isEmpty();
    }
    
    

    /**
     * nodeViewœԂXV܂B
     */
    public void update(NodeView newNodeView) {
        ArgumentChecker.throwIfNull(newNodeView);
        
        nodeView_ = newNodeView;
        markerMap_.clear();
        
		durablePaintings_.clear();
		lastTransientPaintings_.clear();
		lastTransientPaintings_.addAll(transientPaintings_);
		transientPaintings_.clear();
        for (BlockPainter blockPainter : blockPainters_) {
        	Set<SgfPoint> set = blockPainter.getDurablePaintings(nodeView_);
        	if (set == null) {
        		continue;
        	}
        	durablePaintings_.addAll(set);
        			
        	transientPaintings_.addAll(blockPainter.getTransientPaintings(nodeView_));
        }
        
		updateMarkers(nodeView_);
        updateInheritableMarkers(nodeView_.getInheritableDecoration());
        updateOverblockMarker(nodeView_);
    }


    

    /**
     * PubNɑ݂S}[J[XV܂B
     */
    private void updateMarkers(NodeView nodeView) {
        for (int i = 0; i < drawTypes_s_.length; ++i) {
            Property property = nodeView.getProperty(drawTypes_s_[i]);
            if ((property == null) || 
                    !property.sgfId().valueType().equals(ValueType.POINT)) {
                continue;
            }
            String[] values = property.getStrings();
            SgfPoint[] points = SgfPoint.parse(nodeView_.getSize(), values);
            for (int j = 0; j < points.length; ++j) {
                addMarker(points[j], drawTypes_s_[i]);
            }
        }
    }


    /**
	 * ̌pvpeB\nXV܂B
     */
    private void updateInheritableMarkers(InheritableMarker inheritableMarker) {
        UEnum[] enums = UEnum.getAll(SgfId.class);
        for (int i = 0; i < enums.length; ++i) {
            SgfId sgfId = (SgfId)enums[i];
            if (!sgfId.propertyType().equals(PropertyType.INHERIT)) {
                continue;
            }
            if (SgfId.VIEW.equals(sgfId)) {
            	updateInheritableMarkerImpl4VW(inheritableMarker);
            } else {
                updateInheritableMarkerImpl(inheritableMarker, sgfId);
            }
        }
        inheritableMarker_.set(inheritableMarker);
    }
    
    /**
	 * sgfTypeŎw肳ꂽpvpeBXV܂B
     */
    private void updateInheritableMarkerImpl(InheritableMarker inheritable, SgfId sgfType) {
        String[] oldValues = inheritableMarker_.get(sgfType);
        String[] newValues = inheritable.get(sgfType);
        SgfPoint[] oldPoints = getPoints(oldValues);
        SgfPoint[] newPoints = getPoints(newValues);
        
        if (oldPoints != null) {
            for (int i = 0; i < oldPoints.length; ++i) {
                addRedrawPoint(oldPoints[i]);
            }
        }
        
        if (newPoints == null) {
            return;
        }
        for (int i = 0; i < newPoints.length; ++i) {
            addMarker(newPoints[i], sgfType);
        }
    }
    
    /**
     * VWĕ`̃[ςKv(̏ꍇ͑Sĕ`ɂȂ)̂ŁA
     * {@link #updateInheritableMarkerImpl(InheritableMarker, SgfId)}Ƃ͕ʂ
     * \bhpӂĂ܂B
     * @param inheritable
     */
    private void updateInheritableMarkerImpl4VW(InheritableMarker inheritable) {
        String[] oldValues = inheritableMarker_.get(SgfId.VIEW);
        String[] newValues = inheritable.get(SgfId.VIEW);
        SgfPoint[] oldPoints = getPoints(oldValues);
        SgfPoint[] newPoints = getPoints(newValues);
        
        vwPoints_.clear();
        if ((oldPoints == null) && (newPoints == null)) {
        	vwPoints_.addAll(nodeView_.getSize().all());
        	return;
        }
        
        if ((oldPoints == null) || (oldPoints.length == 0)){
    		Set<SgfPoint> all = nodeView_.getSize().all();
    		oldPoints = all.toArray(new SgfPoint[all.size()]);
        }
        for (int i = 0; i < oldPoints.length; ++i) {
            addRedrawPoint(oldPoints[i]);
        }

        if ((newPoints == null) || (newPoints.length == 0)) {
        	vwPoints_.addAll(nodeView_.getSize().all());
    		newPoints = vwPoints_.toArray(new SgfPoint[vwPoints_.size()]);
    	}
    	for (SgfPoint sgfPoint : newPoints) {
            addMarker(sgfPoint, SgfId.VIEW);
            addRedrawPoint(sgfPoint);
            vwPoints_.add(sgfPoint);
        }
    }

    private SgfPoint[] getPoints(String[] values) {
        return (values == null) ? null : SgfPoint.parse(nodeView_.getSize(), values);
    }


    /**
     * @param nodeView
     */
    private void updateOverblockMarker(NodeView nodeView) {
        OverblockMarker newMarker = nodeView.getOverblockMarker();
        boolean isEmpty = overblockMarker_.isEmpty() && newMarker.isEmpty();
        
        overblockMarker_.set(newMarker);
        if (isEmpty) {
            return;
        }
    }

    /**
     * ݂View͈̔͂Ԃ܂B
     * ݒ̏ꍇsize͈̔͂̂܂ܕԂ܂B
     */
    public SgfRectangle calcView() {
        SgfRectangle ret = new SgfRectangle();
//        SgfPoint[] points = getMarkerPoints(SgfId.VIEW);
        SgfPoint[] points = vwPoints_.toArray(new SgfPoint[vwPoints_.size()]);
        for (int i = 0; i < points.length; ++i) {
            ret.intersect(points[i]);
        }
        
        if (points.length == 0) {
            SgfSize size = nodeView_.getSize();
            ret.set(SgfPoint.create(size, 1, 1), 
                    SgfPoint.create(size, size.width(), size.height()));
        }
        return ret;
    }
    
    /**
     * pointVWŎw肳ꂽ`|Cg(܂VWw̏ꍇ͔Փ)̏ꍇtrueԂ܂B
     * @param point
     * @return
     */
    public boolean isDrawArea(SgfPoint point) {
    	return vwPoints_.contains(point);
    }
    
    
    /**
     * `ΏۂƂȂubN̏WԂ܂B
     * @return
     */
    public Set<SgfPoint> getRedrawBlocks() {
    	Set<SgfPoint> allPoints = new TreeSet<SgfPoint>();
    	allPoints.addAll(durablePaintings_);
    	allPoints.addAll(transientPaintings_);
    	allPoints.addAll(lastTransientPaintings_);
        return allPoints;
    }

	

    /**
     * @return
     */
    public Label[] getLabels() {
        return overblockMarker_.getLabels(nodeView_.getSize());
    }

    /**
     * @param sgfType
     * @return
     */
    public SgfLine[] getLine(SgfId sgfType) {
        return overblockMarker_.getLine(sgfType, nodeView_.getSize());
    }
}