/* TextViewer.java
 * 
 * Copyright 2004 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.model.board;

import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;

import org.unitarou.lang.NullArgumentException;
import org.unitarou.sgf.type.SgfColor;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.sgf.type.SgfSize;
import org.unitarou.sgf.util.SgfPointType;

/**
 * {@link org.unitarou.yukinoshita.model.board.IgoBoard}̓eA
 * eLXǧ`ŕ\܂B
 * 
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public class TextViewer {
	static public final class CharMap {
		static public final CharMap DEFAULT_CHARMAP; 
		static public final CharMap NO_BOLD_CHARMAP;
		static {
			SortedMap<SgfPointType, Character> map = new TreeMap<SgfPointType, Character>();
			map.put(SgfPointType.BLACK, 		new Character(''));
			map.put(SgfPointType.WHITE, 		new Character(''));
			map.put(SgfPointType.EMPTY, 		new Character(''));
			map.put(SgfPointType.TOP_LEFT, 		new Character(''));
			map.put(SgfPointType.TOP, 			new Character(''));
			map.put(SgfPointType.TOP_RIGHT,		new Character(''));
			map.put(SgfPointType.LEFT, 			new Character(''));
			map.put(SgfPointType.RIGHT, 		new Character(''));
			map.put(SgfPointType.BOTTOM_LEFT,		new Character(''));
			map.put(SgfPointType.BOTTOM, 			new Character(''));
			map.put(SgfPointType.BOTTOM_RIGHT,	new Character(''));
			map.put(SgfPointType.STAR, 			new Character(''));
			DEFAULT_CHARMAP = new CharMap(map);

			map.put(SgfPointType.TOP_LEFT, 		new Character(''));
			map.put(SgfPointType.TOP, 			new Character(''));
			map.put(SgfPointType.TOP_RIGHT,		new Character(''));
			map.put(SgfPointType.LEFT, 			new Character(''));
			map.put(SgfPointType.RIGHT, 		new Character(''));
			map.put(SgfPointType.BOTTOM_LEFT,		new Character(''));
			map.put(SgfPointType.BOTTOM, 			new Character(''));
			map.put(SgfPointType.BOTTOM_RIGHT,	new Character(''));
			map.put(SgfPointType.STAR, 			new Character(''));
			NO_BOLD_CHARMAP = new CharMap(map);
		}
		/** ^[PointType, Character] */
		private final SortedMap<SgfPointType, Character> charMap_;
		
		/** charMap̌^[PointType, Character] */
		private CharMap(SortedMap<SgfPointType, Character> charMap) {
			charMap_ = new TreeMap<SgfPointType, Character>(charMap);
		}
		public CharMap(CharMap charMap) {
			charMap_ = new TreeMap<SgfPointType, Character>(charMap.charMap_);
		}
		public char getChar(SgfPointType type) {
			return charMap_.get(type).charValue();
		}
		public void put(SgfPointType type, char c) {
			if (type == null) {
				throw new NullArgumentException();
			}
			charMap_.put(type, new Character(c));
		}
	}
	


	private final CharMap charMap_;
	/**
	 * 
	 */
	public TextViewer() {
		super();
		charMap_ = new CharMap(CharMap.DEFAULT_CHARMAP);
	}
	
	public void setCharacter(SgfPointType type, char c) {
		if (type == null) {
			throw new NullArgumentException();
		}
		charMap_.put(type, c);
	}
	
	public String[] write(IgoBoardView goBoard) {
		SgfSize size = goBoard.size();
		String[] ret = new String[size.height()];
		
		SortedSet blackStones = goBoard.position(SgfColor.BLACK);
		SortedSet whiteStones = goBoard.position(SgfColor.WHITE);
		for (int y = 1; y <= size.height(); ++y) {
		    ret[y - 1] = writeOneLine(size, y, blackStones, whiteStones);
		}
		
		return ret;
	}
	
	private String writeOneLine(
	        SgfSize size, 
	        int y, 
	        SortedSet blackStones, 
	        SortedSet whiteStones) 
	{
		StringBuilder sb = new StringBuilder();
		for (int x = 1; x <= size.width(); ++x) {		    
			SgfPoint point = SgfPoint.create(size, x, y);
			if (blackStones.contains(point)) {
				sb.append(charMap_.getChar(SgfPointType.BLACK));
				
			} else if (whiteStones.contains(point)) {
				sb.append(charMap_.getChar(SgfPointType.WHITE));
				
			} else if (size.isStarPoint(point)) {
				sb.append(charMap_.getChar(SgfPointType.STAR));
			
			} else {
			    sb.append(charMap_.getChar(SgfPointType.classify(point)));
			}
		}
		return sb.toString();
	}
	
	
	public char getCharacter(SgfPointType pointType) {
		return charMap_.getChar(pointType);
	}
	
}
