/* 
 * Copyright 2003-2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.view;

import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.unitarou.sgf.Collection;
import org.unitarou.sgf.PropertyType;
import org.unitarou.sgf.io.CollectionRepository;
import org.unitarou.sgf.type.GameType;
import org.unitarou.util.Adaptable;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.util.Provider;
import org.unitarou.util.StopWatch;
import org.unitarou.yukinoshita.Yukinoshita;
import org.unitarou.yukinoshita.context.ContextListener;
import org.unitarou.yukinoshita.context.CurrentContext;
import org.unitarou.yukinoshita.events.ActionListener;
import org.unitarou.yukinoshita.events.EventBroker;
import org.unitarou.yukinoshita.events.EventBrokerImpl;
import org.unitarou.yukinoshita.model.CollectionEditor;
import org.unitarou.yukinoshita.model.GameMediator;
import org.unitarou.yukinoshita.model.NodeView;
import org.unitarou.yukinoshita.model.cmd.Command4NodeList;
import org.unitarou.yukinoshita.model.cmd.ModelInfluence;
import org.unitarou.yukinoshita.view.cmd.ChangeEditModeCommand;
import org.unitarou.yukinoshita.view.cmd.Command4View;
import org.unitarou.yukinoshita.view.cmd.WrapperCommand;
import org.unitarou.yukinoshita.view.monitor.CollectionMonitor;
import org.unitarou.yukinoshita.view.monitor.ContextMonitor;
import org.unitarou.yukinoshita.view.monitor.GameInfoNodeMonitor;
import org.unitarou.yukinoshita.view.monitor.GameMonitor;
import org.unitarou.yukinoshita.view.monitor.NodeMonitor;

/**
 * CollectionŜ́A܂SGFt@CControllerłB
 * Collection̂̂̕ҏW̓t@T[hNX{@link org.unitarou.yukinoshita.model.CollectionEditor}
 * S܂B瑤UISłB
 * 
 * View({@link org.unitarou.yukinoshita.view.ModelFrame})Ɏgo^
 * EventBrokern܂B
 * 
 * TODO@EventBrokerImplp~āÃNX̎qNX
 *        EventBrokerĂĂȂB
 * 
 * @author UNITAROU &lt;boss@unitarou.org&gt;
 */
public class GameFrameController implements Adaptable {

	static private final Log log_s_ = LogFactory.getLog(GameFrameController.class);
	private final CollectionEditor collectionEditor_;
	
	private final ListenerAdapter listenerAdapter_;
	private final ModelFrame modelFrame_;
	private final EventBrokerImpl eventBrokerImpl_;
	private boolean isEditMode_;
	
	/** 
	 * ^[{@link Class},{@link ModelFrame}]łB
	 */
	private final Map<Class, ModelFrame> modelFrameMap_;
		
	public GameFrameController(Collection collection, ModelFrame modelFrame) {
		this(collection, modelFrame, false);
	}
	/**
	 * @throws org.unitarou.lang.NullArgumentException NULL̏ꍇ
	 * @throws IllegalArgumentException collection gameȂꍇB
	 */
	public GameFrameController(Collection collection, ModelFrame modelFrame, boolean setChangedFlag) {
		super();
		ArgumentChecker.throwIfNull(collection, modelFrame);

		if (collection.size() == 0) {
		    throw new IllegalArgumentException("collection must have one or more games."); //$NON-NLS-1$
		}
        listenerAdapter_ = new ListenerAdapter();
        
        collectionEditor_ = new CollectionEditor(collection, setChangedFlag);
		eventBrokerImpl_ = new EventBrokerImpl();
		isEditMode_ = false;
		
		eventBrokerImpl_.addController(this);
		eventBrokerImpl_.addView(this);
		modelFrameMap_ = new HashMap<Class, ModelFrame>();

		//TODO ʈoɂB
		modelFrame_ = modelFrame;
		modelFrame_.setEventBroker(eventBrokerImpl_);
		
		// Drill̏ꍇ͍ŏ̖oB
		if (collection.get(0).getGameType().equals(GameType.DRILL)) {
			ViewerUtils.setupDriiRootDirectory(collection);
			CollectionRepository.instance().loadProblem(collection);
			if (1 < collection.size()) {
				collectionEditor_.setActiveGame(1);
			}
		}
		
		fireCollectionViewerUpdate();
		fireGameViewerUpdate();
		fireNodeViewerUpdate();
		Yukinoshita.context().addListener(listenerAdapter_);
	}
	
	
	/**
	 * modelFramẽRg[ɓo^܂B
	 * ȑOɓNXœo^ĂmodelFrameꍇ́A
	 * CX^X폜Ăo^܂B
	 * @param key
	 * @param modelFrame
	 */
	public void registerModelFrame(ModelFrame modelFrame) {
	    ArgumentChecker.throwIfNull(modelFrame);
	    
	    modelFrame.setEventBroker(eventBrokerImpl_);
	    modelFrameMap_.put(modelFrame.getClass(), modelFrame);
		fireCollectionViewerUpdate();
		fireGameViewerUpdate();
		eventBrokerImpl_.executeCommand(new ChangeEditModeCommand(isEditMode_));
	}
	

	
    /* (non-Javadoc)
     * @see org.unitarou.lang.Adaptable#getAdapter(java.lang.Class)
     */
    public Object getAdapter(Class<?> adapter) {
        if (adapter == null) {
            return null;
        }
        if (adapter.isAssignableFrom(listenerAdapter_.getClass())) {
            return listenerAdapter_;
        }
        return null;
    }

    public void dispose() {
	    collectionEditor_.dispose();
	    modelFrame_.dispose();
	    eventBrokerImpl_.removeController(this);
	    eventBrokerImpl_.dispose();
	}
    
    /**
     * Rg[[ǗĂ{@link CollectionEditor}Ԃ܂B
     *  <code>null</code>͕Ԃ܂B 
     */
    public CollectionEditor getCollectionEditor() {
    	return collectionEditor_;
    }
    
    /**
     * Rg[[ǗĂ{@link Collection}Ԃ܂B<br> 
     * <b></b>̃IuWFNgɒڕҏWȂłB<br>
     * ViewɕύXʒmsȂ̂ŁAfƕ`̑ΉȂȂ܂B
     */
    public Collection getCollection() {
        return collectionEditor_.getCollection();
    }
	
    /**
     * ǗĂCollectionɕύXꍇtrueԂ܂B
     */
    public boolean isChanged() {
        boolean[] flags = collectionEditor_.getChangedFlags();
        for (int i = 0; i < flags.length; ++i) {
            if (flags[i]) {
                return true;
            }
        }        
        return false;
    }
    
    /**
     * ύXtONA[܂B
     */
    public void clearChangedFlag() {
        collectionEditor_.clearChangedFlags();
        fireNodeViewerUpdate(); // TODO m[hx̐؂ւł̂ȁH
    }
    
    /**
     * ǂ̃Q[ҏWĂ邩zԂ܂B<br>
     * ߂l͒{@link Collection#size()}Ŋevftruȅꍇ
     * RootGameTreen܂Q[ҏWĂ邱Ƃ܂B
     */
    public boolean[] getChangedFlags() {
        return collectionEditor_.getChangedFlags();
    }
    
    /**
     * ҏW[h̏ꍇtrueԂ܂B 
     * @return
     */
    public boolean isEditMode() {
    	return isEditMode_;
    }
    
    /** 
     * Redo/Undoɂf̕ωɑΉ܂B
     * @param name 
     */
    public void updateByRedoUndo(EnumSet<ModelInfluence> name) {
    	for (ModelInfluence influence : name) {
    		switch (influence) {
			case COLLECTION:
				fireCollectionViewerUpdate();
				break;
			
			case ROOT_GAME_TREE:
				fireGameViewerUpdate();
				break;

			case NODE_PEVIOUS:
				//TODO ƕύX݂̂ɒʒmׂB
				fireGameViewerUpdate();
				break;

			case NODE_DOWNWARD:
				//TODO ƕύX݂̂ɒʒmׂB
				fireGameViewerUpdate();
				break;
				
			case NODE_CURRENT:
				fireNodeViewerUpdate();
				break;
				
			case NODE_GAMEINFO:
				fireGameInfoNodeUpdate();
				break;

			default:
				assert false : "Unknown type:" + influence; //$NON-NLS-1$
			}
    	}

        fireNodeViewerUpdate();
    }
    
    /**
     * ̃Rg[Ă{@link EventBroker}Ԃ܂B
     * @return null͌ĕԂ܂B
     */
    public EventBroker getEventBroker() {
        return eventBrokerImpl_;
    }

    private void fireCollectionViewerUpdate() {
        GameType gameType = collectionEditor_.getActiveGame().getGameType();
        eventBrokerImpl_.connect(gameType, isEditMode_);
        for (CollectionMonitor viewer : eventBrokerImpl_.getListeners(CollectionMonitor.class)) {
        	viewer.update(collectionEditor_);
        }
	}
	
    /**
     * \EҏWQ[̕ύXʒm܂B
     */
	private void fireGameViewerUpdate() {
		ViewerUtils.setupDriiRootDirectory(collectionEditor_.getCollection());
	    GameMediator gameMediator = collectionEditor_.getActiveGame();
        for (Iterator ip = eventBrokerImpl_.getListeners(GameMonitor.class).iterator();
    			ip.hasNext(); ) {
        	GameMonitor gameViewer = (GameMonitor)ip.next();
        	gameViewer.update(gameMediator);
        }
	}
	
	private void fireNodeViewerUpdate() {
		StopWatch stopWatch = new StopWatch();
		NodeView nodeView = collectionEditor_.getActiveGame().getCurrentNodeView();
        for (Iterator<NodeMonitor> ip = eventBrokerImpl_.getListeners(NodeMonitor.class).iterator(); ip.hasNext(); ) {
        	NodeMonitor nodeViewer = ip.next();
        	nodeViewer.currentChanged(nodeView);
    		if (log_s_.isTraceEnabled()) {
    			log_s_.trace(stopWatch.lapSecond() + " secs for " + nodeViewer);  //$NON-NLS-1$
    		}
        }
		if (log_s_.isTraceEnabled()) {
			log_s_.trace(stopWatch.stopSecond() + " secs for fireNodeViewerUpdate");  //$NON-NLS-1$
		}
	}
    /**
     * \EҏWQ[̕ύXʒm܂B<br>
     * {@link #setCollectionEditor(CollectionEditor)}A
     * {@link #setEventBroker(EventBroker)}ŐݒĂȂꍇ
     * {@link IllegalStateException}o܂B
     * 
     * @throws IllegalStateException {@link #collectionEditor_}A{@link #eventBroker_}null̏ꍇ
     */
	private void fireGameInfoNodeUpdate() {
		GameMediator gameMediator = collectionEditor_.getActiveGame();
		
		//GameInfoNodeViewF
		int index = gameMediator.findNodeView(PropertyType.GAME_INFO);
		if (index < 0) {
			index = 0;
		}
		NodeView target = gameMediator.getNodeList().getNodeView(index);
	    for (GameInfoNodeMonitor monitor : 
				eventBrokerImpl_.getListeners(GameInfoNodeMonitor.class)) {
	    	monitor.gameInfoChanged(target);
	    }
	}

    private void fireContextAttributeChanged(Set<String> keySet) {
    	for (ContextMonitor contextViewer 
    			: eventBrokerImpl_.getListeners(ContextMonitor.class)) 
    	{
    		contextViewer.attributeChanged(keySet);
    	}
    }
    
    /**
     * voC_[ύX܂
     * ێĂViewŜ{@link ContextMonitor#attributeChanged(Set)}ʒm܂B
     * @param providerInterface
     * @param provider
     * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
     */
    public <T extends Provider> void changeProvider(
    		Class<T> providerInterface, Provider provider) 
    {
        ArgumentChecker.throwIfNull(providerInterface, provider);
        
        GameMediator gameMediator = collectionEditor_.getActiveGame();
        CurrentContext context = gameMediator.getContext();
        context.setProvider(providerInterface, provider);
        
        Set<String> set =new HashSet<String>();
        set.add(providerInterface.getName());
        Set<String> keySet = Collections.unmodifiableSet(set);
        fireContextAttributeChanged(keySet);
    }

    /**
     * providerClassAbstract strategyƂA
     * ݎgĂvoC_[Ԃ܂B
     * @param providerInterface
     * @return
     */
    public <T extends Provider> T getProvider(Class<T> providerInterface) {
        ArgumentChecker.throwIfNull(providerInterface);
        
        GameMediator gameMediator = collectionEditor_.getActiveGame();
        CurrentContext context = gameMediator.getContext();
        return context.getProvider(providerInterface);
    }
    
    /** \NodeύXꂽƂɕ`ʒm܂B*/
	private class ListenerAdapter implements ActionListener, ContextListener {
	    private HandlerPhase currentStatus_;
        public ListenerAdapter() {
            super();
            currentStatus_ = null;
        }

		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.events.ActionListener#executeCommand(org.unitarou.yukinoshita.model.cmd.Command4NodeList)
		 */
		public void executeCommand(Command4NodeList command) {
			ArgumentChecker.throwIfNull(command);
			WrapperCommand wrapperCommand = new WrapperCommand(command);
			wrapperCommand.getParameter().set(collectionEditor_, eventBrokerImpl_, isEditMode_);
			wrapperCommand.execute();
		}

		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.events.ActionListener#executeCommand(org.unitarou.yukinoshita.view.cmd.Command4View)
		 */
		public void executeCommand(Command4View command) {
			ArgumentChecker.throwIfNull(command);
			Command4View.Parameter parameter = command.getParameter();
			parameter.setCollectionEditor(collectionEditor_);
			parameter.setEventBroker(eventBrokerImpl_);
			// EditModeɂĂ̓R}hw肷ꍇ̂ŁA
			// w̎̂ݐݒ肵Aݒlł̒lXVB
			if (parameter.isEditMode() == null) {
				parameter.setEditMode(isEditMode_);
			} else {
				isEditMode_ = parameter.isEditMode().booleanValue();
			}
			
			// HandlerPhaseĂ̓R}hw肷ꍇ̂ŁA
			// w̎̂ݐݒ肵AR}hsݒlł̒lXVB
			if ((parameter.getCurrentStatus() == null) && (currentStatus_ != null)) {
				parameter.setCurrentStatus(currentStatus_);
			}
			command.execute();
			currentStatus_ = parameter.getCurrentStatus();
		}

        /* (non-Javadoc)
         * @see org.unitarou.yukinoshita.ContextListener#attributeChanged(java.util.Set)
         */
        public void attributeChanged(Set<String> keySet) {
            ArgumentChecker.throwIfNull(keySet);
            fireContextAttributeChanged(keySet);
        }
	}
}
