/* 
 * Copyright 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.view.jface;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.window.WindowManager;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import org.unitarou.lang.Strings;
import org.unitarou.ml.BasicMessages;
import org.unitarou.ml.MessageResource;
import org.unitarou.sgf.Collection;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.Application;
import org.unitarou.yukinoshita.FileManager;
import org.unitarou.yukinoshita.context.Context;
import org.unitarou.yukinoshita.view.WindowController;
import org.unitarou.yukinoshita.view.jface.board.sp.StonePainter;

/**
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
public class JFaceApplication implements Application {
    /** ̃NX̃K[łB*/
	static private final Log log_s_ = LogFactory.getLog(JFaceApplication.class);	

	private final WindowManager windowManager_;
	
	private final JFaceContext context_;
	
	private final Display display_;
	private final Shell shell_;

	/**
	 * LmV^ŃG[Ă܂B
	 * AvP[VIAf[^댯łB
	 * \łΌݕҏWĂt@Cʖŕۑ邱Ƃ߂܂B
	 * ܂̉P̂߁AOt@CJ҂ɑĂƍKłB
	 */
	static private final MessageResource MSG_INNER_ERROR 
			= new MessageResource(JFaceApplication.class,"msgInnerError"); //$NON-NLS-1$
	
	/**
	 * LmV^ŕAs\ȃG[܂B
	 * \󂠂܂Bۑ̃f[^͎댯ɍłB
	 * ꂩ炱ꂩ疢ۑ̃t@Cɂĕۑ_CAOo܂A
	 * \Ȍ茻ݕҏWĂt@CʖŕۑĂB
	 * ܂̉P̂߁AOt@CJ҂ɑĂƍKłB
	 */
	static private final MessageResource MSG_FATAL_INNER_ERROR 
			= new MessageResource(JFaceApplication.class,"msgFatalInnerError"); //$NON-NLS-1$
	
	/**
     * 
     */
    public JFaceApplication() {
        super();
        display_ = new Display();
        windowManager_ = new WindowManager();
        context_ = new JFaceContext();
        context_.registerInterface(StonePainter.class);
        Window.setExceptionHandler(new ExceptionHandler());
        shell_ = new Shell(display_);
   }

    
    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.Application#dispose()
     */
    public void dispose() {
    	display_.dispose();
    }

    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.Application#start()
     */
    public void start(File[] files) {
        context_.registerProvider(StonePainter.CONTEXT.defaultProvider().getClass());
        try {
            createWindow();
            JFaceWindow window = (JFaceWindow)windowManager_.getWindows()[0];
            for (File file : files) {
            	Collection collection = FileManager.instance().open(file);
            	if (collection != null) {
                	window.register(collection, Strings.EMPTY, false);
            	}
            }
            while (!shell_.isDisposed()) {
            	if (!display_.readAndDispatch()) {
                	display_.sleep();
            	}
            	if (windowManager_.getWindowCount() == 0) {
            		shell_.dispose();
            	}
            }
            context_.saveContext();
            
        } catch (Throwable e) {
			log_s_.error("Fatal internal error occurred.", e); //$NON-NLS-1$
			MessageDialog dig = new MessageDialog(
			        Display.getCurrent().getActiveShell(),
			        BasicMessages.NT_ERROR.get(),
			        null,
			        JFaceApplication.MSG_FATAL_INNER_ERROR.get(),
			        MessageDialog.ERROR,
			        new String[]{BasicMessages.CLB_CLOSE.get()},
			        0);
		    dig.open();
		    windowManager_.close();
	    
	    } finally {
            Display.getCurrent().dispose();
        }
    }


    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.Application#createWindow()
     */
    public void createWindow() {
        JFaceWindow window = new JFaceWindow(null);
        windowManager_.add(window);
        window.setBlockOnOpen(false);
        window.open();
    }

    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.Application#closeAllWindows()
     */
    public void closeAllWindows() {
    	windowManager_.close();
    }


    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.Application#getContext()
     */
    public Context getContext() {
        return context_;
    }

   
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.Application#openMessageDialog(org.unitarou.yukinoshita.Application.MessageLevel, java.lang.String, java.lang.String)
	 */
	public void openMessageDialog(MessageLevel level, String title, String message) {
		ArgumentChecker.throwIfNull(level, title, message);
		Display display = Display.getCurrent();
		Shell shell = (display != null) ? display.getActiveShell() : new Shell();
		switch (level) {
		case ERROR:
	        MessageDialog.openError(shell, title, message);
			break;

		case WARNING:
	        MessageDialog.openWarning(shell, title, message);
			break;

		case INFORMATION:
	        MessageDialog.openInformation(shell, title, message);
			break;
			
		default:
			assert false : "Unknown type:" + level; //$NON-NLS-1$
			break;
		}
	}

	/**
	 * SWT̃[`ŔG[NXłB
	 * Â悤̂ŁA_CAOoɂ܂B 
	 */
	private class ExceptionHandler implements Window.IExceptionHandler {

		public void handleException(Throwable t) {
			if (t instanceof ThreadDeath) {
				throw (ThreadDeath)t;
			}

			log_s_.error("SWT internal error occurred.", t); //$NON-NLS-1$
			MessageDialog dig = new MessageDialog(
			        Display.getCurrent().getActiveShell(),
			        BasicMessages.NT_ERROR.get(),
			        null,
			        JFaceApplication.MSG_INNER_ERROR.get(),
			        MessageDialog.ERROR,
			        new String[]{BasicMessages.CLB_CLOSE.get()},
			        0);
		    dig.open();
		}
    	
    }

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.Application#openDirectoryDialog(java.lang.String, java.lang.String, java.io.File)
	 */
	public File openDirectoryDialog(String title, String message, File currentDirectory) {
		ArgumentChecker.throwIfNull(title, message, currentDirectory);
		DirectoryDialog directoryDialog = new DirectoryDialog(Display.getCurrent().getActiveShell());
		directoryDialog.setText(title);
		directoryDialog.setMessage(message);
		String path = (currentDirectory.isDirectory()) 
								? currentDirectory.getAbsolutePath()
								: currentDirectory.getParentFile().getAbsolutePath();
		directoryDialog.setFilterPath(path);
		path = directoryDialog.open();
		return (path == null) ? null : new File(path);
	}


	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.Application#getAllController()
	 */
	public WindowController[] getAllController() {
		Window[] windows = windowManager_.getWindows();
		List<WindowController> controllers = new ArrayList<WindowController>(windows.length - 1);
		for (Window window : windows) {
			if (window instanceof WindowController) {
				controllers.add((WindowController)window);
			}
		}
		return controllers.toArray(new WindowController[controllers.size()]);
	}
}
