/*  
 * Copyright 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.view.jface;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;
import java.util.WeakHashMap;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.preference.PreferenceStore;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;

import org.unitarou.lang.Strings;
import org.unitarou.ml.BasicMessages;
import org.unitarou.ml.MessageResource;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.util.Provider;
import org.unitarou.yukinoshita.Yukinoshita;
import org.unitarou.yukinoshita.Application.MessageLevel;
import org.unitarou.yukinoshita.context.BooleanContextValue;
import org.unitarou.yukinoshita.context.Context;
import org.unitarou.yukinoshita.context.ContextListener;
import org.unitarou.yukinoshita.context.ContextValue;
import org.unitarou.yukinoshita.context.CurrentContext;
import org.unitarou.yukinoshita.context.DefaultContext;
import org.unitarou.yukinoshita.context.FileContextValue;
import org.unitarou.yukinoshita.context.IntArrayContextValue;
import org.unitarou.yukinoshita.context.IntContextValue;

/**
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
public class JFaceContext extends PreferenceStore implements Context {
	
	/**
	 * lz{@link PreferenceStore}ɕۑۂ̋؂qłB<br>
	 * ۂ̒l̓J}łB 
	 */
	static private String DELIMITER_ARRAY = ","; //$NON-NLS-1$
	
	/** ̃NX̃JeS[̃K[ł */
	static private final Log log_s_ = LogFactory.getLog(JFaceContext.class);
	
    
	/**
	 * uݒt@C̓ǂݍ݂Ɏs܂BSďݒl̗p܂B\n
	 * VXẽG[bZ[WF{0}v
	 */
	static private final MessageResource MSG_IO_ERROR_IN_RESTORE
		= new MessageResource(JFaceContext.class, "msgIoErrorInRestore"); //$NON-NLS-1$

	/**
	 * uݒt@C̏oɎs܂Bݒ͕ۑ܂B\n
	 * VXẽG[bZ[WF{0}v
	 */
	static private final MessageResource MSG_IO_ERROR_IN_SAVE
		= new MessageResource(JFaceContext.class, "msgIoErrorInSave"); //$NON-NLS-1$

	private final DefaultContext defaultContext_;
    
	/**
     * ^[Object:scope, {@link CurrentContext}:currentContext]łB
     */
    private final WeakHashMap<Object, CurrentContext> contextMap_;

    /**
     * ۑ̃t@CłB 
     */
    private File file_;
    
    /**
     * 
     */
    public JFaceContext() {
        super();
        setFilename(Yukinoshita.USER_NAME + ".preference"); //$NON-NLS-1$
        defaultContext_ = new DefaultContext();
        contextMap_ = new WeakHashMap<Object, CurrentContext>();
        addPropertyChangeListener(new IPropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent event) {
				propertyChangeImpl(event.getProperty(), event.getNewValue());
			}
        });
    }
    
    private void propertyChangeImpl(String key, Object value) {
    	if (value == null) {
    		return;
    	}
		Object lastVal = getAttribute(key, null);
		Object newVal = null;
		if (lastVal == null || lastVal.getClass().isAssignableFrom(value.getClass())) {
			newVal = value;
			
		} else if (lastVal instanceof Integer) {
			newVal = new Integer(value.toString());

		} else if (lastVal instanceof int[]) {
			newVal = parseIntArray(value);

		} else if (lastVal instanceof Boolean) {
			newVal = new Boolean(value.toString());

		} else if (lastVal instanceof Enum) {
			for (Object obj : lastVal.getClass().getEnumConstants()){
				if (((Enum)obj).name().equals(value.toString())) {
					newVal = obj;
				}
			}

		} else if (lastVal instanceof Provider) {
			try {
				newVal = Class.forName(value.toString()).newInstance();

			} catch (Exception e) {
				log_s_.warn(e.getMessage() + " Bad class name: " + value ); //$NON-NLS-1$
			}
			
		} else if (lastVal instanceof File) {
			newVal = new File(lastVal.toString());
			
		} else {
			log_s_.warn(composeWarnMsgUnknownTypeValue(key, lastVal));
		}
		if (newVal == null) {
			return;
		}
		defaultContext_.setAttribute(key, newVal, null);
		defaultContext_.fireAttributeChanged(new String[]{key});
		if (log_s_.isDebugEnabled()) {
	    	StringBuilder builder = new StringBuilder();
	    	builder.append("Property changed. "); //$NON-NLS-1$
	    	builder.append(key).append("=").append(value); //$NON-NLS-1$
	    	log_s_.debug(builder.toString());
		}
    }
    
	/**
	 * valuel̔złΕʂĕԂ܂B
	 * valueReLXgɕۑꂽJ}؂̐lłΐzɕϊĕԂ܂B
	 * ǂłȂꍇnullԂ܂B
	 * @param value
	 * @return
	 */
	private int[] parseIntArray(Object value) {
    	if (value instanceof int[]) {
    		int[] newArray = new int[((int[])value).length];
    		System.arraycopy(value, 0, newArray, 0, newArray.length);
    		return newArray;
    	}
    	
    	if (!(value instanceof String)) {
        	return null;
    	}
    	List<Integer> list = new ArrayList<Integer>();
		StringTokenizer tokenizer = new StringTokenizer((String)value, DELIMITER_ARRAY);
		while(tokenizer.hasMoreTokens()){ 
			try {
				String token = tokenizer.nextToken();
				Integer integer = Integer.decode(token.trim());
				list.add(integer);
			} catch (NumberFormatException e) {
				log_s_.warn("Bad value.", e); //$NON-NLS-1$
				return null;
			}
		}
		
		int[] ret = new int[list.size()];
		for (int i = 0; i < ret.length; ++i) {
			ret[i] = list.get(i).intValue();
		}
		return ret;
	}

	/**
	 * ̔zReLXg̕ۑł镶(J}؂)ɕϊ܂B
	 * @param value
	 * @return
	 */
	private String formatIntArray(int[] value) {
		if (value.length == 0) {
			return Strings.EMPTY;
		}
		StringBuilder builder = new StringBuilder();
		for (int element : value) {
			builder.append(element).append(DELIMITER_ARRAY);
		}
		builder.delete(builder.length() - DELIMITER_ARRAY.length(), builder.length());
		return builder.toString();
	}

	/**
     * ^ϊɎsɏo郁bZ[W쐬ĕԂ܂B
     * @param key
     * @param value
     * @return
     */
    private String composeWarnMsgUnknownTypeValue(String key, Object value) {
    	StringBuilder builder = new StringBuilder();
    	builder.append("Unknown value type, ignore this attribute: ("); //$NON-NLS-1$
    	builder.append(key).append("=").append(value); //$NON-NLS-1$
    	builder.append(") class=").append(value.getClass()); //$NON-NLS-1$
    	return builder.toString();
    }
    
	/**
	 * GetterŕۑꂽlƗvꂽ^قȂƂbZ[W쐬ĕԂ܂B
	 * @param key
	 * @param value
	 * @param clazz
	 */
	private String composeMsgMismatchValueType(String key, Object value, Class clazz) { 
		StringBuilder builder = new StringBuilder();
		builder.append("Bad value type, use default value. key=\'").append(key);  //$NON-NLS-1$
		builder.append("\', value=\'").append(value);  //$NON-NLS-1$
		builder.append("\', class=" + clazz); //$NON-NLS-1$
		return builder.toString();
	}

    
    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.context.Context#restore()
     */
    public void loadContext() {
    	file_ = new File(Yukinoshita.USER_NAME + ".preference").getAbsoluteFile(); //$NON-NLS-1$
        FileInputStream inputStream = null;
        try {
        	FileUtils.touch(file_);
        	inputStream = new FileInputStream(file_);
			load(inputStream);
			for (String key : preferenceNames()) {
				propertyChangeImpl(key, getString(key));
			}
		} catch (IOException e) {
			log_s_.warn(e.getMessage() + " File: " + file_.toString()); //$NON-NLS-1$
        	Yukinoshita.application().openMessageDialog(
        			MessageLevel.INFORMATION,
        			BasicMessages.NT_ERROR_FILE_INPUT.get(),
        			MSG_IO_ERROR_IN_RESTORE.get(e.getMessage()));
			
		} finally {
			IOUtils.closeQuietly(inputStream);
		}
        setFilename(file_.getAbsolutePath());
    }

    /* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.context.Context#saveContext()
	 */
	public void saveContext() {
		try {
			super.save();
		} catch (IOException e) {
			log_s_.warn("Cant save context. Reason: " + e.getMessage()); //$NON-NLS-1$
        	Yukinoshita.application().openMessageDialog(
        			MessageLevel.INFORMATION,
        			BasicMessages.NT_ERROR_FILE_OUTPUT.get(),
        			MSG_IO_ERROR_IN_SAVE.get(e.getMessage()));
			
		}
	}
    
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#addListener(org.unitarou.yukinoshita.ContextListener)
	 */
	public void addListener(ContextListener listener) {
		defaultContext_.addListener(listener);
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#clearCurrent(java.lang.Object)
	 */
	public Object[] clearCurrent(Object scope) {
        contextMap_.remove(scope);
		return defaultContext_.clearCurrent(scope);
	}

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#fireAttributeChanged(java.lang.Object[])
	 */
	public void fireAttributeChanged(String[] keys) {
		defaultContext_.fireAttributeChanged(keys);
	}
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#getAttribute(java.lang.Object, java.lang.Object)
	 */
	public Object getAttribute(String key, Object scope) {
		return defaultContext_.getAttribute(key, scope);
	}

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#getCurrent(java.lang.Object)
	 */
	public CurrentContext getCurrent(Object scope) {
        CurrentContext ret = contextMap_.get(scope);
        if (ret == null) {
            ret = new CurrentContextImpl(scope);
            contextMap_.put(scope, ret);
        }
        return ret;
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#getProvider(java.lang.Class, java.lang.Object)
	 */
	public <T extends Provider> T getProvider(Class<T> provider, Object scope) {
		return defaultContext_.getProvider(provider, scope);
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#getProviders(java.lang.Class)
	 */
	public <T extends Provider> T[] getProviders(Class<T> provider) {
		return defaultContext_.getProviders(provider);
	}

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#registerInterface(java.lang.Class)
	 */
	public boolean registerInterface(Class<?> provider) {
		return defaultContext_.registerInterface(provider);
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.context.Context#getInterfaces()
	 */
	public Class<?>[] getInterfaces() {
		return defaultContext_.getInterfaces();
	}

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#registerProvider(java.lang.Class)
	 */
	public Class<?>[] registerProvider(Class<?> concreteProvider) {
		Class[] classes =  defaultContext_.registerProvider(concreteProvider);
		for (Class clazz : classes) {
			String key = clazz.getName();
			if (!contains(key)){
				setDefault(key, concreteProvider.getName());
			}
		}
		return classes;
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#removeListener(org.unitarou.yukinoshita.ContextListener)
	 */
	public void removeListener(ContextListener listener) {
		defaultContext_.removeListener(listener);
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#setAttribute(java.lang.Object, java.lang.Object, java.lang.Object)
	 */
	public void setAttribute(String key, Object value, Object scope) {
		defaultContext_.setAttribute(key, value, scope);
		
		if (scope != null) {
			return;
		}

		//AvP[Vx̐ݒ̎̂݁Avt@Xɓo^B
		if (value instanceof Boolean) {
			setValue(key, ((Boolean)value).booleanValue());

		} else if (value instanceof Double) {
			setValue(key, ((Double)value).doubleValue());
			
		} else if (value instanceof Float) {
			setValue(key, ((Float)value).floatValue());

		} else if (value instanceof Long) {
			setValue(key, ((Long)value).longValue());

		} else if (value instanceof Integer) {
			setValue(key, ((Integer)value).intValue());
			
		} else if (value instanceof int[]) {
			setValue(key, formatIntArray((int[])value));

		} else if (value instanceof Enum) {
			setValue(key, ((Enum)value).name());

		} else if (value instanceof String) {
			setValue(key, (String)value);

		} else if (value instanceof Provider) {
			setValue(key, value.getClass().getName());
			
		} else if (value instanceof File) {
			setValue(key, ((File)value).getAbsolutePath());
			
		} else {
			log_s_.warn(composeWarnMsgUnknownTypeValue(key, value));
			return;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.context.DefaultContext#setAttribute(org.unitarou.yukinoshita.context.ContextValue, java.lang.Object)
	 */
	public void setAttribute(ContextValue attribute, Object scope) {
		ArgumentChecker.throwIfNull(attribute);
		String key = attribute.id();
		Object value = attribute.defaultValue();
		setAttribute(key, value, scope);
		setDefault(key, value);
	}
	
	private void setDefault(String key, Object value) {
		if (value instanceof Boolean) {
			setDefault(key, ((Boolean)value).booleanValue());

		} else if (value instanceof Double) {
			setDefault(key, ((Double)value).doubleValue());
			
		} else if (value instanceof Float) {
			setDefault(key, ((Float)value).floatValue());

		} else if (value instanceof Long) {
			setDefault(key, ((Long)value).longValue());

		} else if (value instanceof Integer) {
			setDefault(key, ((Integer)value).intValue());
			
		} else if (value instanceof Enum) {
			setDefault(key, ((Enum)value).name());

		} else if (value instanceof String) {
			setDefault(key, (String)value);

		} else if (value instanceof Provider) {
			setDefault(key, value.getClass().getName());
			
		} else if (value instanceof File) {
			setDefault(key, ((File)value).getAbsolutePath());

		} else {
			log_s_.warn(composeWarnMsgUnknownTypeValue(key, value));
			return;
		}
	}

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.context.Context#rootKeys()
	 */
	public String[] rootKeys() {
		return defaultContext_.rootKeys();
	}

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.DefaultContext#setProvider(java.lang.Class, java.lang.Class, java.lang.Object)
	 */
	public void setProvider(Class<?> providerInterface, Class<?> provider, Object scope) {
		defaultContext_.setProvider(providerInterface, provider, scope);
		String key = providerInterface.getName();
		if ((scope != null) || contains(key)){
			return;
		}
		setValue(key, provider.getName());
	}

    /**
     * {@link DefaultContext#getCurrent(Object)}ŕԂNXłB
     * XR[vQƂŕێ̂ŁAOŃXR[vGCꂽuԂɁA
     * ̃CX^X̓ReLXg̃[g悤ɂȂ܂B 
     */
    private class CurrentContextImpl implements CurrentContext {
        private final WeakReference<Object> reference_;
        private CurrentContextImpl(Object scope) {
            reference_ = new WeakReference<Object>(scope);
        }
        /* (non-Javadoc)
         * @see org.unitarou.yukinoshita.CurrentContext#getProvider(java.lang.Class)
         */
        public <T extends Provider> T getProvider(Class<T> providerInterface) {
            return JFaceContext.this.getProvider(providerInterface, reference_.get());
        }
        
        /* (non-Javadoc)
         * @see org.unitarou.yukinoshita.CurrentContext#setProvider(java.lang.Class, java.lang.Class)
         */
        public <T extends Provider> void setProvider(
        		Class<T> providerInterface, Provider provider) 
        {
            setAttribute(providerInterface.getName(), provider, reference_.get());
        }
        
        /* (non-Javadoc)
         * @see org.unitarou.yukinoshita.CurrentContext#getBoolean(java.lang.Object)
         */
        public boolean getBoolean(String key) {
            Object value = getAttribute(key, reference_.get());
            if (value == null || !(value instanceof Boolean)) {
            	throwIllegalStateException(key, value);
            }
            return ((Boolean)value).booleanValue();
        }
        
		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.context.CurrentContext#getBoolean(org.unitarou.yukinoshita.context.BooleanContextValue)
		 */
		public boolean getBoolean(BooleanContextValue value) {
			ArgumentChecker.throwIfNull(value);
            Object attr = getAttribute(value.id(), reference_.get());
            if (attr == null || !(attr instanceof Boolean)) {
            	return value.defaultBoolean();
            }
            return ((Boolean)attr).booleanValue();
		}

        
		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.context.CurrentContext#getInteger(org.unitarou.yukinoshita.context.IntContextValue)
		 */
		public int getInteger(IntContextValue value) {
			ArgumentChecker.throwIfNull(value);
			return getInteger(value.id(), value.defaultInt());
		}
		
		
		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.context.CurrentContext#getInteger(java.lang.String, int)
		 */
		public int getInteger(String key, int defaultValue) {
            Object value = getAttribute(key, reference_.get());
            if (value == null) {
            	return defaultValue;
            }
            if (value instanceof Integer) {
                return ((Integer)value).intValue();
            }
            if (value instanceof String) {
            	try {
            		return Integer.parseInt((String)value);
            	} catch (NumberFormatException e) {
            		log_s_.warn("Cant parse value: " + value); //$NON-NLS-1$
				}
            }
            log_s_.warn(composeMsgMismatchValueType(key, value, int[].class));
			return defaultValue;
		}
		
		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.context.CurrentContext#setInteger(java.lang.String, int)
		 */
		public void setInteger(String key, int value) {
			setAttribute(key, new Integer(value), reference_.get());
		}
		
		
		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.context.CurrentContext#getIntArray(java.lang.String, int[])
		 */
		public int[] getIntArray(String key, int[] defaultValue) {
			Object value = getAttribute(key, reference_.get());
            if (value == null) {
            	return defaultValue;
            }
            int[] ret = parseIntArray(value);
            if (ret != null) {
            	return ret;
            }
            log_s_.warn(composeMsgMismatchValueType(key, value, int[].class));
			return defaultValue;
		}

		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.context.CurrentContext#getIntArray(org.unitarou.yukinoshita.context.IntArrayContextValue)
		 */
		public int[] getIntArray(IntArrayContextValue value) {
			ArgumentChecker.throwIfNull(value);
			return getIntArray(value.id(), value.defaultIntArray());
		}

		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.context.CurrentContext#setIntArray(java.lang.String, int[])
		 */
		public void setIntArray(String key, int[] value) {
			ArgumentChecker.throwIfNull(value);
			setAttribute(key, value, reference_.get());			
		}

		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.context.CurrentContext#getFile(org.unitarou.yukinoshita.context.FileContextValue)
		 */
		public File getFile(FileContextValue value) {
			ArgumentChecker.throwIfNull(value);
            Object obj = getAttribute(value.id(), reference_.get());
            if (obj == null) {
            	return value.defaultFile();
            }
            if (obj instanceof String) {
            	obj = new File((String)obj);
            	
            }
            if (!(obj instanceof File)) {
            	log_s_.warn(composeMsgMismatchValueType(value.id(), obj, File.class));
            	return value.defaultFile();
            }
            return (File)obj;
		}
		
		/* (non-Javadoc)
		 * @see org.unitarou.yukinoshita.context.CurrentContext#setFile(java.lang.String, java.io.File)
		 */
		public void setFile(String key, File file) {
			ArgumentChecker.throwIfNull(file);
			setAttribute(key, file, reference_.get());			
		}

		/**
		 * keyɑ΂valuesƂ{@link IllegalStateException}𓊂܂B
		 * @param key
		 * @param value
		 */
		private void throwIllegalStateException(String key, Object value) {
            throw new IllegalStateException(
                    "Bad value. key=\'" + key  //$NON-NLS-1$
                    + "\', value=\'" + value + "\' " //$NON-NLS-1$ //$NON-NLS-2$
                    + "class=" + ((value == null) ? "null" : value.getClass().getName()));  //$NON-NLS-1$ //$NON-NLS-2$
		}
    }
}