/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.view.jface.board;


import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.util.Geometry;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;

import org.unitarou.jface.ColorResource;
import org.unitarou.lang.Strings;
import org.unitarou.sgf.Property;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.type.Label;
import org.unitarou.sgf.type.SgfColor;
import org.unitarou.sgf.type.SgfLine;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.sgf.type.SgfSize;
import org.unitarou.sgf.type.TypeParseException;
import org.unitarou.sgf.util.SgfPointType;
import org.unitarou.sgf.util.SgfRectangle;
import org.unitarou.sgf.util.Stone;
import org.unitarou.sgf.util.provider.crdlp.CoordinatesLabelProvider;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.context.Context;
import org.unitarou.yukinoshita.context.CurrentContext;
import org.unitarou.yukinoshita.events.EventBroker;
import org.unitarou.yukinoshita.events.ModelEventNotifier;
import org.unitarou.yukinoshita.model.GameMediator;
import org.unitarou.yukinoshita.model.NodeView;
import org.unitarou.yukinoshita.model.board.BlockedBoardDesigner;
import org.unitarou.yukinoshita.model.board.IgoBoard;
import org.unitarou.yukinoshita.view.jface.board.mp.SimpleMarkerPainter;
import org.unitarou.yukinoshita.view.jface.board.sp.StonePainter;
import org.unitarou.yukinoshita.view.jface.board.sp.StonePainterParameter;
import org.unitarou.yukinoshita.view.monitor.ContextMonitor;
import org.unitarou.yukinoshita.view.monitor.GameMonitor;
import org.unitarou.yukinoshita.view.monitor.NodeMonitor;
import org.unitarou.yukinoshita.view.provider.blklp.BlockLabelProvider;
import org.unitarou.yukinoshita.view.provider.vlp.VariationLabelProvider;

/**
 * OtBJɌՂ`悷plłB
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
public class IgoGraphicalBoardPanel implements IgoBoardPanel {
	/**
	 * K[łB
	 */
	static private final Log log_s_ = LogFactory.getLog(IgoGraphicalBoardPanel.class);
	
	/**
	 * `ΏۂƂubN̍ŏl(2)łB
	 */
	static private final int MINIMUM_BLOCK_SIZE = 2;

	/**
	 * 
	 */
	static private final ColorResource FOREGROUND_CANT_PAINT = new ColorResource(0xFF,0x00,0x00);
	
	/**
	 * ΂̕`Ɋ֘AID̔złB
	 */
	static private final SgfId[] IDS_STONE 
		= new SgfId[]{SgfId.WHITE, SgfId.BLACK, SgfId.ADD_WHITE, SgfId.ADD_BLACK, SgfId.ADD_EMPTY};
    
	/** ۂɕ`ΏۂƂȂLoXłB*/
    private Canvas canvas_;
    
	/** VMŋ؂ꂽ`GAłB*/
    private SgfRectangle view_;
    
	
    /** Ղ̖ؒn`悷IuWFNgłB*/
    private BoardPainter boardPainter_;
    
    /** r`悷IuWFNgłB*/
    private RuledLinePainter ruledLinePainter_;
    
    /** Ղ̍W`悷IuWFNgłB*/
    private CoordinatesPainter coordinatesPainter_;
    
    /** ΂`悷IuWFNgłB*/
    private StonePainter stonePainter_;
    
    /** L`悷IuWFNgł*/
    private MarkPainter markPainter_;
    
    /** updateɍXVPubÑTCY(sNZP)łB */
    private int blockSize_;
    
    /** W\tHg̃TCYłB */
    private Point labelSize_;

    /** ォ̃ItZbg̃TCYłB */
    private Point offsetSize_;

    
    private EventBroker eventBroker_;
    
    /** ݕ\Ăǖʂ̃m[hłB*/
    private NodeView nodeView_;

	/** `pɌՂ̏ԂێǗĂ܂B*/
	private final BlockedBoardDesigner blockedBoardDesigner_;

    /** ݂̃ReLXgłB */
	private CurrentContext context_;

    /**
     * ^[{@link SgfPoint}, {@link BlockStatus}]
     * RXĝ{@link BlockStatus}̍쐬LbVNXłB
     * Ղ̍XVɂăNA[܂B
     * L[null̂HashMapŒłB
     */
    private final HashMap<SgfPoint, BlockStatus> blockStatusMap_;

   
    /** true̎Ƀ_uobt@[s܂B*/
    private boolean usesDoubleBuffer_;
    
    
    /**
     * bIɕ`悷vpeBłB
     * ݂Ȃꍇnullێ܂B
     */
    private Property transientProperty_;
    
    private Adapter adapter_;
    
    private BlockLabelProvider blockLabelProvider_;
    private VariationLabelProvider variationLabelProvider_;
    
    /**
     * ̃tOtruêƂ́AŏIɂƂ킩}[N܂B
     * ftHgłtruełB
     */
    private boolean showLastMoveMark_;
    
    
    

    /**
     * 
     */
    public IgoGraphicalBoardPanel() {
        super();
        canvas_ = null;
        
        view_ = new SgfRectangle();
		boardPainter_ 		= new SimpleBoardPainter();
		ruledLinePainter_ 	= new SimpleRuledLinePainter();
		coordinatesPainter_ = new SimpleCoordinatesPainter();
		stonePainter_		= StonePainter.CONTEXT.defaultProvider();
		markPainter_ 		= new SimpleMarkerPainter();
		usesDoubleBuffer_ 	= true;// TODO Ƃ肠
		labelSize_ = new Point(1,1);
		offsetSize_ = new Point(0,0);
		eventBroker_ = EventBroker.NULL_BROKER;
		blockedBoardDesigner_ = new BlockedBoardDesigner(new IgoBoard(SgfSize.DEFAULT));
		blockStatusMap_ = new HashMap<SgfPoint, BlockStatus>();
		transientProperty_ = null;
		adapter_ = new Adapter();
		blockLabelProvider_ = BlockLabelProvider.CONTEXT.defaultProvider();
		variationLabelProvider_ = VariationLabelProvider.CONTEXT.defaultProvider();
		showLastMoveMark_ = true;
		context_ = CurrentContext.nullContext;
		
		blockedBoardDesigner_.addBlockPainter(boardPainter_);
		blockedBoardDesigner_.addBlockPainter(ruledLinePainter_);
		blockedBoardDesigner_.addBlockPainter(stonePainter_);
		blockedBoardDesigner_.addBlockPainter(markPainter_);
		blockedBoardDesigner_.addBlockPainter(blockLabelProvider_);
    }
   
    /* (non-Javadoc)
     * @see org.unitarou.lang.Adaptable#getAdapter(java.lang.Class)
     */
    public Object getAdapter(Class<?> adapter) {
        if (adapter == null) {
            return null;
        }
        if (/*adapter.isInstance(adapter_)*/ adapter.isAssignableFrom(adapter_.getClass())){
            return adapter_;
        }
        return null;
    }

    /**
     * parenteƂăpl\܂B
     * 
     * @throws org.unitarou.lang.NullArgumentException null̏ꍇ
     */
    public Control createContents(Composite parent) {
        ArgumentChecker.throwIfNull(parent);

        canvas_ = new Canvas(parent, SWT.NO_BACKGROUND);
        canvas_.addPaintListener(new PaintListener() {
            public void paintControl(PaintEvent e) {
                Rectangle paintArea = new Rectangle(e.x, e.y, e.width, e.height);
                if (usesDoubleBuffer_) {
                    paintByDoubleBuffer(e.gc, paintArea);
                } else {
                    paint(e.gc, paintArea);
                }
            }
        });
        canvas_.addControlListener(new ControlAdapter() {

			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ControlListener#controlResized(org.eclipse.swt.events.ControlEvent)
			 */
			@Override
			public void controlResized(ControlEvent e) {
				blockStatusMap_.clear();
			}
        
        });
       return canvas_;
    }
    

    /* (non-Javadoc)
     * @see org.unitarou.swt.WidgetContainer#dispose()
     */
    public void dispose() {
        canvas_.dispose();
        eventBroker_ = EventBroker.NULL_BROKER;
        nodeView_ = null;
    }

    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.view.jface.board.IgoBoardPanel#getCanvas()
     */
    public Canvas getCanvas() {
        return canvas_;
    }


    /**
     * Double Buffer`ŕ`s܂B
     */
    private void paintByDoubleBuffer(GC gc, Rectangle paintArea) {
        Image image = null;
        GC imageGc = null;
        try {
            image = new Image(canvas_.getDisplay(), canvas_.getBounds());
            imageGc = new GC(image);
            paint(imageGc, paintArea);
    		gc.drawImage(image, 0, 0);
    		
        } finally {
            if (imageGc != null) {
                imageGc.dispose();
            }
            if (image != null) {
                image.dispose();
            }
        }
    }
    
    /**
     * LoXŜ̕`s\bhłB
     */
    private void paint(GC gc, Rectangle paintArea) {
        // f̏O͕`悵Ȃ
        if (nodeView_ == null) {
        	log_s_.info("Call paint(GC gc, Rectangle paintArea) before init model."); //$NON-NLS-1$
            return;
        }
        SgfSize size = nodeView_.getSize();
		view_ = blockedBoardDesigner_.calcView();
        calcDrawParameters();
        if (blockSize_ < MINIMUM_BLOCK_SIZE) {
        	paintCantDraw(gc);
        	return;
        }
        
        Rectangle boardRect = calcDrawArea(view_);
        
        // wi̕`
        paintBackground(gc, boardRect);
        coordinatesPainter_.paint(
                gc, 
                boardRect,
                view_,
                new boolean[]{true, true, false, false}
                );
		for (int y = view_.getStart().y(); y <= view_.getEnd().y(); ++y) {
			for (int x = view_.getStart().x(); x <= view_.getEnd().x(); ++x) {
				SgfPoint point = SgfPoint.create(size, x, y);
				if (blockedBoardDesigner_.isDrawArea(point)) {
					paintImpl(gc, paintArea, point);
				} else {
					paintBackground(gc, point);
				}
			}
		}
		paintLabel(gc);
		paintLine(gc);
		paintVariation(gc);
		paintTransientStone(gc, boardRect);
		paintTransientMarkers(gc);
    }


    /**
     * ubNĕ`s\ł邱Ƃ}`܂B
	 * @param gc
	 */
	private void paintCantDraw(GC gc) {
		Point point = canvas_.getSize();
		int size = Math.min(point.x, point.y);
		gc.setForeground(FOREGROUND_CANT_PAINT.get());
		gc.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		
		gc.fillRectangle(0,0,point.x, point.y);
		gc.drawRectangle(0,0,size - 1, size - 1);
		gc.drawLine(0,0,size, size);
		gc.drawLine(0,size,size, 0);
	}
	
	/**
	 * Ղ̊O̔wi`悵܂B
	 * @param gc
	 * @param boardRect
	 */
	private void paintBackground(GC gc, Rectangle boardRect) {
        gc.setBackground(canvas_.getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
        Rectangle rect =  canvas_.getBounds();
        gc.fillRectangle(0, 0, boardRect.x, rect.height);
        gc.fillRectangle(boardRect.x, 0, rect.width, boardRect.y);
        gc.fillRectangle(boardRect.x + boardRect.width, boardRect.y, rect.width, rect.height);
        gc.fillRectangle(boardRect.x, boardRect.y + boardRect.height, rect.width, rect.height);
	}
	
	/**
	 * VWŌڂɂȂ̌wiƂĕ`悵܂B
	 * @param gc
	 * @param blockPoint
	 */
	private void paintBackground(GC gc, SgfPoint blockPoint) {
		gc.setBackground(canvas_.getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		Rectangle rectangle = calcBlockArea(blockPoint);
		// SWT͂ꂼI[܂߂Ȃ̂
		++rectangle.width;
		++rectangle.height;
		gc.fillRectangle(rectangle);
	}

	/**
     * pointR}`悵܂B
     */
    private void paintImpl(GC gc, Rectangle paintArea, SgfPoint point) {
        Rectangle rectangle = calcBlockArea(point);
        
        // ĕ`͈͂ɓȂꍇ͕`悵ȂB
        if ((rectangle.x + rectangle.width < paintArea.x) || (paintArea.x + paintArea.width < rectangle.x)
             || (rectangle.y + rectangle.height < paintArea.y) || (paintArea.y + paintArea.height < rectangle.y)) {
            return;
        }
        // wi`
        boardPainter_.paint(gc, point, rectangle);
        
        // r`
        ruledLinePainter_.paint(gc, point, rectangle);
        
        // ΂`
        paintStone(gc, point, rectangle);
        
        // L`
        paintMarks(gc, point, rectangle);
        
        // VXeXe[^X\
        paintStatus(gc, blockStatusMap_.get(point));
    }
    
	/**
     * ݂̃LoX̑傫VWɂ鐧̈悩
     * R}̑傫vZĕԂ܂B
     */
    private void calcDrawParameters() {
    	Rectangle canvasSize = canvas_.getBounds();
        labelSize_ = coordinatesPainter_.calcLabelSize(view_);
        blockSize_ = Math.min((canvasSize.width - labelSize_.x) / view_.width(), 
        	    		 	  (canvasSize.height - labelSize_.y) / view_.height());
        blockSize_ = Math.min(blockSize_, context_.getInteger(Context.MAX_STONE_SIZE));
        offsetSize_.x = (canvasSize.width - (blockSize_ * view_.width() + labelSize_.x)) / 2;
        offsetSize_.y = (canvasSize.height - (blockSize_ * view_.height() + labelSize_.y)) / 2;
    }
    
    /**
     * VMɂ̈搧poinẗʒuAblockSize
     * PubN̈ʒuvZ܂B 
     */
    private Rectangle calcBlockArea(SgfPoint point) {
		return new Rectangle(
		        labelSize_.x + (point.x() - view_.getStart().x()) * blockSize_ + offsetSize_.x, 
		        labelSize_.y + (point.y() - view_.getStart().y()) * blockSize_ + offsetSize_.y,
        		blockSize_, blockSize_);
    }
    
    /**
     * areaۂ̃LoX̕`̈ɕϊ܂B
     * @param area
     * @return
     */
    private Rectangle calcDrawArea(SgfRectangle area) {
		return new Rectangle(
	              labelSize_.x + (area.getStart().x() - view_.getStart().x()) * blockSize_ + offsetSize_.x,
	              labelSize_.y + (area.getStart().y() - view_.getStart().y()) * blockSize_ + offsetSize_.y,
	              area.width() * blockSize_,
	              area.height() * blockSize_);

    }

    
    // Ōɐ΂`
    private void paintStone(GC gc, SgfPoint point, Rectangle rectangle) {
    	EnumSet<SgfPointType> pointTypes = getBlockStatusImpl(point).getPointTypes();
        StonePainterParameter parameter;
        if (pointTypes.contains(SgfPointType.WHITE)) {
        	parameter = new StonePainterParameter();
        	parameter.setSgfColor(SgfColor.WHITE);
		    
        } else if (pointTypes.contains(SgfPointType.BLACK)) {
        	parameter = new StonePainterParameter();
        	parameter.setSgfColor(SgfColor.BLACK);
        	
        } else {
        	return;
        }
        if (transientProperty_ != null) {
        	for (SgfId id : IDS_STONE) {
        		if (transientProperty_.contains(id, point.getString())) {
        			parameter.setTransient(true);
        			break;
        		}
        	}
        }
        Stone move = nodeView_.getMove();
        String label = blockLabelProvider_.getLabel(nodeView_, point);
        parameter.setLastMove(showLastMoveMark_ && point.equals(move.getPoint()));
        parameter.setLabel(label);
	    stonePainter_.paintStone(gc, rectangle, parameter);
    }
    
    
    private void paintMarks(GC gc, SgfPoint point, Rectangle rectangle) {
        for (SgfId sgfId : blockedBoardDesigner_.getMarkers(point)) {
        	if ( (transientProperty_ != null)
        			&& transientProperty_.contains(sgfId, point.getString())) {
        		break;
        	} 
            markPainter_.paintMark(gc, rectangle, sgfId, false);
        }
    }
    
    /**
	 * @param gc
	 * @param status
	 */
	private void paintStatus(GC gc, BlockStatus status) {
		markPainter_.paintStatus(gc, status);
	}

	/**
     * @param gc
     */
    private void paintTransientStone(GC gc, Rectangle boardRect) {
    	if (transientProperty_ == null) {
    	    return;
    	}
    	
    	SgfColor sgfColor = SgfColor.getStoneColor(transientProperty_.sgfId());
    	if (sgfColor == null) {
    		return;
    	}

    	StonePainterParameter parameter = new StonePainterParameter();
        parameter.setLabel(null);
        parameter.setLastMove(false);
        parameter.setTransient(true);
        parameter.setSgfColor(sgfColor);
        
        

        for (String string : transientProperty_.getStrings()) {
            SgfPoint point = SgfPoint.parseMoveQuietly(nodeView_.getSize(), string);
            if (point == null) {
            	continue;
            }
            Rectangle block = calcBlockArea(point);
            
            // Ղ̒̂ݕ`悷B
            if (0 == Geometry.getRelativePosition(boardRect, Geometry.centerPoint(block))) 
            {
            	stonePainter_.paintStone(gc, block, parameter);
            }
        }
    }

    private void paintTransientMarkers(GC gc) {
        // uuvL̕`
        if (transientProperty_ == null) {
            return;
        }
        SgfId[] sgfTypes = markPainter_.markerScope();
        for (int i = 0; i < sgfTypes.length; ++i) {
            if (!sgfTypes[i].equals(transientProperty_.sgfId())) {
                continue;
            }
            for (String datum : transientProperty_.getStrings()) {
                SgfPoint point = SgfPoint.parseMoveQuietly(nodeView_.getSize(), datum);
                if (point == null) {
                	continue;
                }
                Rectangle block = calcBlockArea(point);
                markPainter_.paintMark(gc, block, sgfTypes[i], true);
            	
            }
        }
    }
    
    /**
     * @param gc
     */
    private void paintLabel(GC gc) {
        Label[] labels = blockedBoardDesigner_.getLabels();
        for (int i = 0; i < labels.length; ++i) {
            Rectangle rectangle = calcBlockArea(labels[i].getPoint());
            markPainter_.paintLabel(gc, rectangle, labels[i], false);
        }
        
        // ꎞIȏp̓߃x̕\
        if ((transientProperty_ == null) || !transientProperty_.sgfId().equals(SgfId.LABEL)) {
        	return;
        }
        Label label = Label.parseQuietly(nodeView_.getSize(), transientProperty_.getStrings()[0]);
        Rectangle rectangle = calcBlockArea(label.getPoint());
        markPainter_.paintLabel(gc, rectangle, label, true);
    }

    /**
     * @param gc
     */
    private void paintLine(GC gc) {
    	for (SgfId id : new SgfId[]{SgfId.LINE, SgfId.ARROW}) {
    		String string = Strings.EMPTY;
    		if ( (transientProperty_ != null) && id.equals(transientProperty_.sgfId())) {
    			string = transientProperty_.getStrings()[0];
    		}
	        paintLineImpl(gc, id, string);
    	}
    }
    
    
    private void paintLineImpl(GC gc, SgfId sgfType, String string) {
        for (SgfLine line : blockedBoardDesigner_.getLine(sgfType)) {
            Rectangle start = calcBlockArea(line.getStart());
            Rectangle end = calcBlockArea(line.getEnd());
            markPainter_.paintLine(gc, start, end, sgfType, line.getString().equals(string));
        }
        if ((transientProperty_ == null) || !sgfType.equals(transientProperty_.sgfId())) {
            return;
        }
        String[] data = transientProperty_.getStrings();
        for (int i = 0; i < data.length; ++i) {
            SgfLine line;
            try {
                line = SgfLine.parse(nodeView_.getSize(), data[i]);
                Rectangle start = calcBlockArea(line.getStart());
                Rectangle end = calcBlockArea(line.getEnd());
                markPainter_.paintLine(gc, start, end, sgfType, true);
            } catch (TypeParseException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }

    /**
     * ω}ւ̋L(A`AaAb)\܂B
     * @param gc
     */
    private void paintVariation(GC gc) {
        NodeView[] nodeViews = nodeView_.getVariations();
        if (nodeViews.length == 0) {
        	return;
        }
        
        // ꏊɕω}̏肪ꍇ̑Ή 
        Map<SgfPoint, List<String>> variationMap = new HashMap<SgfPoint, List<String>>(nodeViews.length);
        
        for (int i = 0; i < nodeViews.length; ++i) {
            // Siblings̏ꍇ͍ŏIƕω}LԂ̂
            // ĂяoȂB
            if (!nodeView_.isChildrenStyle() 
                    && (i == nodeView_.getSelectedVariationIndex())) 
            {
                continue;
            }
            SgfPoint point = nodeViews[i].getMove().getPoint();
            if ((point == null) || (SgfPointType.PASS.equals(point.condition()))){
                continue;
            }
            
            List<String> strings = variationMap.get(point);
            if (strings == null) {
            	strings = new ArrayList<String>(nodeViews.length);
            	variationMap.put(point, strings);
            }
            strings.add(variationLabelProvider_.getVariationLabel(i));
        }
        
        for (Map.Entry<SgfPoint, List<String>> entry : variationMap.entrySet()) {
            Rectangle rectangle = calcBlockArea(entry.getKey());
            List<String> list = entry.getValue();
            String[] strings = list.toArray(new String[list.size()]);
            markPainter_.paintVariation(gc, rectangle, strings);
        	
        }
    }

	private void redraw() {
		SgfRectangle lastView = view_;
		view_ = blockedBoardDesigner_.calcView();
		if (!lastView.equals(view_)) {
			refresh();
			return;
		}
		
		calcDrawParameters();
		Set<SgfPoint> redrawBlocks = blockedBoardDesigner_.getRedrawBlocks();
		SgfRectangle redrawArea = SgfRectangle.create(redrawBlocks);
		redrawArea.union(view_);
        Rectangle rectangle = calcDrawArea(redrawArea);
		canvas_.redraw(rectangle.x, rectangle.y, rectangle.width, rectangle.height, false);
		if (log_s_.isTraceEnabled()) {
			StringBuilder sb = new StringBuilder();
			sb.append("Redraw rectangle is: ") //$NON-NLS-1$
			  .append(rectangle).append('[')
			  .append((redrawArea.getStart().x() - view_.getStart().x()) + 1)
			  .append(",") //$NON-NLS-1$
			  .append((redrawArea.getStart().y() - view_.getStart().y()) + 1)
			  .append("]-[") //$NON-NLS-1$
			  .append((redrawArea.getStart().x() - view_.getStart().x()) + redrawArea.width())
			  .append(",") //$NON-NLS-1$
			  .append((redrawArea.getStart().y() - view_.getStart().y()) + redrawArea.height())
			  .append("]"); //$NON-NLS-1$
			log_s_.trace(sb);
		}
	}
	
	/** 
	 * ύX_̊ǗƖ֌WɔՑŜ`悵܂B
	 */
	private void refresh() {
		canvas_.redraw(0, 0, canvas_.getBounds().width, canvas_.getBounds().height, false);
		log_s_.trace("Refreshed."); //$NON-NLS-1$
	}

    public boolean isUsesDoubleBuffer() {
        return usesDoubleBuffer_;
    }
    
    public void setUsesDoubleBuffer(boolean usesDoubleBuffer) {
        usesDoubleBuffer_ = usesDoubleBuffer;
    }



    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.view.jface.board.IgoBoardPanel#setTransient(org.unitarou.sgf.Property[])
     */
    public void paintInTransient(Property property) {
        boolean needsRedraw;
        if (property != null) {
            needsRedraw = !property.equals(transientProperty_);
        } else {
            needsRedraw = property != transientProperty_;
        }
        if (!needsRedraw) {
            transientProperty_ = property;
        } else {
            addRedrawPoint(property);
            addRedrawPoint(transientProperty_);
            transientProperty_ = property;
    		redraw();
        }
    }
    
    /**
     * 
     * @param property
     */
    private void addRedrawPoint(Property property) {
        if (property == null) {
            return;
        }
        if (addDdRedrawPoint(property)){
        	return;
        }
        switch (property.sgfId().valueType()) {
        case CPOINT:
        case POINT:
        case MOVE:
            SgfPoint[] points = SgfPoint.parse(nodeView_.getSize(), property.getStrings());
            for (int i = 0; i < points.length; ++i) {
                blockedBoardDesigner_.addRedrawPoint(points[i]);
            }
            break;

        default:
        	// Ȃ
        }
        return;
    }
    
    /**
     * DD vpeBɂĕ`̈𔻒肵܂B
     * @param property
     * @return
     */
    private boolean addDdRedrawPoint(Property property) {
        if (!SgfId.DIM_POINTS.equals(property.sgfId())) {
        	return false;
        }
        SgfPoint[] points = SgfPoint.parse(nodeView_.getSize(), property.getStrings());
        if (points.length != 1 || !points[0].condition().equals(SgfPointType.PASS)) {
            for (int i = 0; i < points.length; ++i) {
                blockedBoardDesigner_.addRedrawPoint(points[i]);
            }
            return true;
        }
        
        // DD̃NA[̏ꍇ͑SẴ|CgNA[B
    	for (int x = view_.getStart().x(); x < view_.getEnd().x(); ++x) {
        	for (int y = view_.getStart().y(); y < view_.getEnd().y(); ++y) {
    	    	blockedBoardDesigner_.addRedrawPoint(
    	    			SgfPoint.create(points[0].size(), x, y));
    		}
    	}

        return true;
    }

    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.view.jface.board.IgoBoardPanel#getNodeView()
     */
    public NodeView getNodeView() {
        return nodeView_;
    }

    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.view.jface.board.IgoBoardPanel#getBlockStatus(org.eclipse.swt.graphics.Point)
     */
    public BlockStatus getBlockStatus(Point canvasPoint) {
        ArgumentChecker.throwIfNull(canvasPoint);
        		
        int x = (int)Math.floor(
        			(canvasPoint.x - labelSize_.x - offsetSize_.x) 
        				/ (double)blockSize_) + view_.getStart().x();
        int y = (int)Math.floor(
        			(canvasPoint.y - labelSize_.y - offsetSize_.y) 
        				/ (double)blockSize_) + view_.getStart().y();
        SgfSize size = nodeView_.getSize();
        if (!size.check(x, y).equals(SgfPointType.IN)) {
            return getBlockStatusImpl(null);
        }
        SgfPoint point = SgfPoint.create(size, x, y);
        if (!blockedBoardDesigner_.isDrawArea(point)) {
            return getBlockStatusImpl(null);
        }
        return getBlockStatusImpl(point);
    }
    
    private BlockStatus getBlockStatusImpl(SgfPoint point) {
        BlockStatus ret = blockStatusMap_.get(point);
        if (ret != null) { 
            return ret;
        }
        ret = createBlockStatus(point);
        blockStatusMap_.put(point, ret);
        return ret;
    }
    
    private BlockStatus createBlockStatus(SgfPoint point) {
    	BlockStatus ret = new BlockStatus();
        if (point == null) {
        	return ret;
        }
        ret.setPoint(point);
        ret.setPointTypes(nodeView_.getPointTypes(point));
        ret.setRectangle(calcBlockArea(point));
        return ret;
    }
    
    
	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.view.jface.board.IgoBoardPanel#setContext(org.unitarou.yukinoshita.CurrentContext)
	 */
	public void setContext(CurrentContext context) {
		ArgumentChecker.throwIfNull(context);
		
		context_ = context;
        boolean needsRefresh = false;
        needsRefresh |= updateBlockLabelProvider();
        needsRefresh |= updateCoordinatesLabelProvider();
        needsRefresh |= updateVariationLabelProvider();
        if (showLastMoveMark_ != context_.getBoolean(Context.SHOW_LAST_MOVE_MARK.id())) {
        	showLastMoveMark_ = !showLastMoveMark_;
            needsRefresh = true;
        }
        if (needsRefresh) {
            refresh();
        }
	}

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.view.jface.board.IgoBoardPanel#update(org.unitarou.yukinoshita.model.NodeView)
	 */
	public void update(NodeView nodeView) {
	    ArgumentChecker.throwIfNull(nodeView);
	    nodeView_ = nodeView;
		blockedBoardDesigner_.update(nodeView);
		blockStatusMap_.clear();
		for (SgfPoint invalid : nodeView.getIgoBoard().validate()) {
			BlockStatus blockStatus = createBlockStatus(invalid);
			EnumSet<SgfPointType> set = blockStatus.getPointTypes();
			set.add(SgfPointType.OVERLAP);
			blockStatus.setPointTypes(set);
			blockStatusMap_.put(invalid, blockStatus);
			blockedBoardDesigner_.addRedrawPoint(invalid);
		}
		
		redraw();
	}

    /**
     * {@link IgoGraphicalBoardPanel#stonePainter__}XV܂B
     * @return VlɍXVꂽtrue
     */
    private boolean updateStonePainter() {
        StonePainter newProvider = context_.getProvider(StonePainter.class);
        if (!stonePainter_.equals(newProvider)) {
        	blockedBoardDesigner_.removeBlockPainter(stonePainter_);
        	stonePainter_ = newProvider;
            blockedBoardDesigner_.addBlockPainter(stonePainter_);
            return true;
        }
        return false;
    }

    /**
     * {@link IgoGraphicalBoardPanel#blockLabelProvider_}XV܂B
     * @return VlɍXVꂽtrue
     */
    private boolean updateBlockLabelProvider() {
        BlockLabelProvider newProvider = context_.getProvider(BlockLabelProvider.class);
        if (!blockLabelProvider_.equals(newProvider)) {
        	blockedBoardDesigner_.removeBlockPainter(blockLabelProvider_);
            blockLabelProvider_ = newProvider;
            blockedBoardDesigner_.addBlockPainter(blockLabelProvider_);
            return true;
        }
        return false;
    }

    /**
     * {@link IgoGraphicalBoardPanel#coordinatesPainter_}
     * {@link CoordinatesLabelProvider}XV܂B
     * @return trueԂ܂B
     */
    private boolean updateCoordinatesLabelProvider() {
        CoordinatesLabelProvider newProvider = 
            context_.getProvider(CoordinatesLabelProvider.class);
        coordinatesPainter_.setLabelProvider(newProvider);
        return true;
    }
    
    private boolean updateVariationLabelProvider() {
        VariationLabelProvider newProvider = 
            context_.getProvider(VariationLabelProvider.class);
        if (!variationLabelProvider_.equals(newProvider)) {
            variationLabelProvider_ = newProvider;
            return true;
        }
        return false;
    }

    /**
     * ̃pl󂯎Cxg܂Ƃ߂NXłB
     * {@link IgoGraphicalBoardPanel#getAdapter(Class)}ŉL̃C^[tF[X
     * Ԃ܂B
     */
    private class Adapter 
    		implements GameMonitor, NodeMonitor, ModelEventNotifier, ContextMonitor {


        /* (non-Javadoc)
         * @see org.unitarou.yukinoshita.view.GameMonitor#update(org.unitarou.yukinoshita.model.GameMediator)
         */
        public void update(GameMediator gameMediator) {
            ArgumentChecker.throwIfNull(gameMediator);
            
            context_ = gameMediator.getContext();
            updateStonePainter();
            updateBlockLabelProvider();
            updateCoordinatesLabelProvider();
            updateVariationLabelProvider();
            showLastMoveMark_ = context_.getBoolean(Context.SHOW_LAST_MOVE_MARK.id());
            refresh();
        }

        /* (non-Javadoc)
         * @see org.unitarou.yukinoshita.view.NodeMonitor#update(org.unitarou.yukinoshita.model.NodeView)
         */
        public void currentChanged(NodeView nodeView) {
    	    IgoGraphicalBoardPanel.this.update(nodeView);
        }

        /* (non-Javadoc)
         * @see org.unitarou.yukinoshita.events.ModelEventNotifier#setEventBroker(org.unitarou.yukinoshita.events.EventBroker)
         */
        public void setEventBroker(EventBroker broker) {
            ArgumentChecker.throwIfNull(broker);
            eventBroker_.removeView(IgoGraphicalBoardPanel.this);
            eventBroker_.disconnect();
            eventBroker_ = broker;
            eventBroker_.addView(IgoGraphicalBoardPanel.this);
        }


        /* (non-Javadoc)
         * @see org.unitarou.yukinoshita.view.ContextMonitor#attributeChanged(java.util.Set)
         */
        public void attributeChanged(Set<String> keySet) {
            ArgumentChecker.throwIfNull(keySet);
            
            boolean needsRefresh = false;
            if (keySet.contains(StonePainter.class.getName())) {
                needsRefresh |= updateStonePainter();
            }

            if (keySet.contains(BlockLabelProvider.class.getName())) {
                needsRefresh |= updateBlockLabelProvider();
            }
            
            if (keySet.contains(CoordinatesLabelProvider.class.getName())) {
                needsRefresh |= updateCoordinatesLabelProvider();
            }
            
            if (keySet.contains(VariationLabelProvider.class.getName())) {
                needsRefresh |= updateVariationLabelProvider();
            }

            if (keySet.contains(Context.SHOW_LAST_MOVE_MARK.id())) {
                showLastMoveMark_ = context_.getBoolean(Context.SHOW_LAST_MOVE_MARK.id());
                needsRefresh = true;
            }

            if (keySet.contains(Context.MAX_STONE_SIZE.id())) {
                needsRefresh = true;
            }

            if (needsRefresh) {
                refresh();
            }
        }
    }
}
