/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.view.jface.board.mp;

import java.util.HashSet;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;

import org.eclipse.jface.util.Geometry;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Display;

import org.unitarou.jface.ColorResource;
import org.unitarou.jface.FontResource;
import org.unitarou.sgf.Property;
import org.unitarou.sgf.PropertyType;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.type.Label;
import org.unitarou.sgf.type.SgfLine;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.sgf.type.SgfSize;
import org.unitarou.sgf.util.SgfPointType;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.model.NodeView;
import org.unitarou.yukinoshita.model.board.InheritableMarker;
import org.unitarou.yukinoshita.model.board.OverblockMarker;
import org.unitarou.yukinoshita.view.jface.board.BlockStatus;
import org.unitarou.yukinoshita.view.jface.board.MarkPainter;

/**
 * LmV^WŒ񋟂ȑfȕ`NXłB
 * 
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
public class SimpleMarkerPainter implements MarkPainter {
    
    /** VariatioñtHgp̐FL[ł*/
    static private final ColorResource COLOR_VARIATION_FONT = new ColorResource(0x00, 0x66, 0x00);
    
    static private final ColorResource COLOR_LINE = new ColorResource(0x00, 0x66, 0x00);

    static private final ColorResource COLOR_LINE_NOGDI = new ColorResource(0x00, 0x99, 0x00);
    
    static private final ColorResource COLOR_LINE_NOGDI_TRANSIENT = new ColorResource(0x00, 0xFF, 0x00);

    /**
     * ΂̏ɍڂ鐔ɗptHg\[XłB
     * TODO tHg̎w
     */
    static private final FontResource FONT_VARIATION
    		= new FontResource("lr oSVbN", 24, SWT.BOLD); //$NON-NLS-1$

    /**
     * xp̃tHg\[XłB
     * TODO tHg̎w
     */
    static private final FontResource FONT_LABEL
    		= new FontResource("lr oSVbN", 24, SWT.BOLD); //$NON-NLS-1$

    
    /** ^[SgfId, Painter]łB*/
    private final SortedMap<SgfId, Painter> painterMap_;
    
    private final SgfId[] scope_;
    
    
    private final Set<SgfPoint> lastDd_;
    /**
     * 
     */
    public SimpleMarkerPainter() {
        super();
        painterMap_ = new TreeMap<SgfId, Painter>();
        painterMap_.put(SgfId.MARK_WITH_X, new CrossPainter());
        painterMap_.put(SgfId.CIRCLE, new CirclePainter());
        painterMap_.put(SgfId.SQUARE, new SquarePainter());
        painterMap_.put(SgfId.TRIANGLE, new TrianglePainter());
        painterMap_.put(SgfId.SELECTED, new SelectedPainter());
        painterMap_.put(SgfId.TERRITORY_WHITE, new TerritoryWhitePainter());
        painterMap_.put(SgfId.TERRITORY_BLACK, new TerritoryBlackPainter());
        painterMap_.put(SgfId.DIM_POINTS, new DimPointsPainter());
        
        scope_ = painterMap_.keySet().toArray(new SgfId[painterMap_.size()]);
        lastDd_ = new HashSet<SgfPoint>();
    }
    
    

    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.view.swt.board.SquareItemPainter#scope()
     */
    public SgfId[] markerScope() {
        SgfId[] ret = new SgfId[scope_.length];
        System.arraycopy(scope_, 0, ret, 0, ret.length);
        return ret;
    }

    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.view.swt.board.SquareItemPainter#paint(org.eclipse.swt.graphics.GC, org.eclipse.swt.graphics.Rectangle, org.unitarou.sgf.Property)
     */
    public void paintMark(GC gc, Rectangle blockArea, SgfId sgfType, boolean isTransient) {
        Painter painter = painterMap_.get(sgfType);
        if (painter != null) {
            painter.paint(gc, blockArea, isTransient);
        }
    }


    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.view.jface.board.MarkPainter#paintLine(org.eclipse.swt.graphics.GC, org.eclipse.swt.graphics.Rectangle, org.eclipse.swt.graphics.Rectangle, org.unitarou.sgf.SgfId)
     */
    public void paintLine(
            GC gc, Rectangle start, Rectangle end, 
            SgfId sgfType, boolean isTransient) 
    {
        ArgumentChecker.throwIfNull(gc, start, end, sgfType);
        if (!SgfId.LINE.equals(sgfType) && !SgfId.ARROW.equals(sgfType)) {
            return;
        }

        Point s = Geometry.centerPoint(start);
        Point e = Geometry.centerPoint(end);
        gc.setAdvanced(true);
        if (!gc.getAdvanced()) {
            gc.setForeground(isTransient ? COLOR_LINE_NOGDI_TRANSIENT.get() : COLOR_LINE_NOGDI.get());
        } else {
            gc.setAntialias(SWT.ON);
            gc.setAlpha(isTransient ? 0x30 : 0xFF);
            gc.setForeground(COLOR_LINE.get());
        }
        gc.setLineWidth(3);
        gc.drawLine(s.x, s.y, e.x, e.y);
        if (sgfType.equals(SgfId.ARROW)) {
        	final double theta = (s.x == e.x) 
        			? Math.signum(e.y - s.y) * Math.PI / 2.0 
        			: Math.atan2(e.y - s.y, e.x - s.x);
        	final double tar1 = theta + Math.PI * 3.0 / 4.0;
        	final double tar2 = theta - Math.PI * 3.0 / 4.0;
        	final double arSize = start.width / 4.0;
        	Point ar1 = new Point((int)(e.x + Math.cos(tar1) * arSize), (int)(e.y + Math.sin(tar1) * arSize)); 
        	Point ar2 = new Point((int)(e.x + Math.cos(tar2) * arSize), (int)(e.y + Math.sin(tar2) * arSize));
        	gc.drawLine(e.x, e.y, ar1.x, ar1.y);
        	gc.drawLine(e.x, e.y, ar2.x, ar2.y);
        }
    }


    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.view.jface.board.MarkPainter#paintLabel(org.eclipse.swt.graphics.GC, org.eclipse.swt.graphics.Rectangle, org.unitarou.sgf.type.Label)
     */
    public void paintLabel(GC gc, Rectangle block, Label label, boolean isTransient) {
        ArgumentChecker.throwIfNull(gc, block, label);

        gc.setFont(FONT_LABEL.get((int)(block.height / 1.41)));
        Point point = gc.textExtent(label.getLabel().getText());
        int x = (block.x + block.width / 2) - point.x / 2;
        int y = (block.y + block.height / 2) - point.y / 2;
        gc.setForeground(Display.getCurrent().getSystemColor(
        		isTransient ? SWT.COLOR_CYAN : SWT.COLOR_BLUE));
        gc.drawText(label.getLabel().getText(), x, y, true);
    }

    /* (non-Javadoc)
     * @see org.unitarou.yukinoshita.view.jface.board.MarkPainter#paintVariation(org.eclipse.swt.graphics.GC, org.eclipse.swt.graphics.Rectangle, int)
     */
    public void paintVariation(GC gc, Rectangle blockArea, String[] labels) {
    	ArgumentChecker.throwIfNull(gc, blockArea, labels);
    	
    	final int cols;
    	if (labels.length <= 1) {
    		cols = 1;
    	} else if (labels.length <= 4) {
    		cols = 2;
    	} else if (labels.length <= 9) {
    		cols = 3;
    	} else {
    		cols = 4;
    	}
    	final Point size = Geometry.divide(Geometry.getSize(blockArea), cols); 
    	final int center = blockArea.width / (cols * 2);
    	
    	for (int i = 0; i < Math.min(labels.length, 16); ++i) {
    		String label = labels[i];
            gc.setFont((cols == 1) 
            		? FONT_VARIATION.getInCircle(gc, label, size.x)
            		: FONT_VARIATION.getInRect(gc, label, size));
            Point p = gc.textExtent(label);
            gc.setForeground(COLOR_VARIATION_FONT.get());
            gc.drawText(label, 
                    blockArea.x + center + size.x * (i % cols) - p.x / 2,
                    blockArea.y + center + size.y * (i / cols) - p.y / 2,
                    true);
    	}
    }


	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.view.jface.board.BlockPainter#getDurablePaintings(org.unitarou.yukinoshita.model.NodeView)
	 */
	public Set<SgfPoint> getDurablePaintings(NodeView now) {
		ArgumentChecker.throwIfNull(now);
		// DD̂
		
		Set<SgfPoint> ret = new HashSet<SgfPoint>();
		ret.addAll(lastDd_);
		Set<SgfPoint> nowDd = new TreeSet<SgfPoint>();
		InheritableMarker inheritableMarker = now.getInheritableDecoration();
		appendInheritable(nowDd, inheritableMarker, SgfId.DIM_POINTS, now.getSize());
		ret.addAll(nowDd);
		lastDd_.retainAll(nowDd);
		ret.removeAll(lastDd_);
		lastDd_.addAll(nowDd);
		return ret;
	}
	
	private void appendInheritable(
			Set<SgfPoint> set, InheritableMarker inheritableMarker, SgfId sgfType, SgfSize sgfSize) 
	{
		String[] values = inheritableMarker.get(sgfType);
		if (values != null) {
			SgfPoint[] points = SgfPoint.parse(sgfSize, values);
			for (SgfPoint point : points) {
				set.add(point);
			}
		}
	}


	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.view.jface.board.BlockPainter#getTransientPaintings(org.unitarou.yukinoshita.model.NodeView)
	 */
	public Set<SgfPoint> getTransientPaintings(NodeView nodeView) {
		ArgumentChecker.throwIfNull(nodeView);
		
		Set<SgfPoint> ret = new TreeSet<SgfPoint>();

		// LINE, LABEL
		calcLbLnArea(nodeView, ret);

		
		// LINE, LABELȊŐL
		for (SgfId sgfType : scope_) {
			if (sgfType.propertyType().equals(PropertyType.INHERIT)) {
				continue;
			}
			Property property = nodeView.getProperty(sgfType);
			if (property == null) {
				continue;
			}
			SgfPoint[] points = SgfPoint.parse(nodeView.getSize(), property.getStrings());
			for (SgfPoint point : points) {
				ret.add(point);
			}
		}
		
		// ω}
		for (NodeView variation : nodeView.getVariations()) {
            SgfPoint point = variation.getMove().getPoint();
            if ((point == null) || (SgfPointType.PASS.equals(point.condition()))){
                continue;
            }
            ret.add(point);
		}
		
		return ret;
	}
	
	private void calcLbLnArea(NodeView nodeView, Set<SgfPoint> redrawArea) {
		OverblockMarker overblockMarker = nodeView.getOverblockMarker();
		if (overblockMarker.isEmpty()) {
			return;
		}
		SgfSize size = nodeView.getSize();
		for (Label label : overblockMarker.getLabels(size)) {
			for (int x = 1; x <= size.width(); ++x) {
				redrawArea.add(SgfPoint.create(size, x, label.getPoint().y()));
			}
		}
		
		for (SgfLine line : overblockMarker.getLine(SgfId.LINE, size)) {
			addLineArea(line, size, redrawArea);
		}

		for (SgfLine line : overblockMarker.getLine(SgfId.ARROW, size)) {
			addLineArea(line, size, redrawArea);
		}
	}
	
	private void addLineArea(SgfLine line, SgfSize size, Set<SgfPoint> redrawArea) {
		final int sx = Math.min(line.getStart().x(), line.getEnd().x()),
					ex = Math.max(line.getStart().x(), line.getEnd().x()),
					sy = Math.min(line.getStart().y(), line.getEnd().y()),
					ey = Math.max(line.getStart().y(), line.getEnd().y());
		for (int x = sx; x <= ex; ++x) {
			for (int y = sy; y <= ey; ++y) {
				redrawArea.add(SgfPoint.create(size,x ,y));
			}
		}
	}



	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.view.jface.board.MarkPainter#paintStatus(org.eclipse.swt.graphics.GC, org.unitarou.yukinoshita.view.jface.board.BlockStatus)
	 */
	public void paintStatus(GC gc, BlockStatus blockStatus) {
		ArgumentChecker.throwIfNull(gc, blockStatus);

		if (!blockStatus.getPointTypes().contains(SgfPointType.OVERLAP)) {
			return;
		}
		Rectangle block = blockStatus.getRectangle();
        gc.setFont(FONT_LABEL.get((block.height / 2)));
        Point point = gc.textExtent("!");
        int x = (block.x + block.width  * 3 / 4) - point.x / 2;
        int y = (block.y + block.height * 1 / 4) - point.y / 2;
        gc.setForeground(Display.getCurrent().getSystemColor(SWT.COLOR_DARK_YELLOW));
        gc.drawText("!", x, y, true);

	}
}