/* 
 * Copyright 2004, 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.view.jface.ns;

import java.util.EnumSet;

import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;

import org.unitarou.jface.ColorResource;
import org.unitarou.jface.ImageResource;
import org.unitarou.lang.Strings;
import org.unitarou.sgf.Property;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.type.SgfPoint;
import org.unitarou.sgf.type.TypeParseException;
import org.unitarou.sgf.util.provider.crdlp.CoordinatesLabelProvider;
import org.unitarou.swt.Swts;
import org.unitarou.util.ArgumentChecker;
import org.unitarou.yukinoshita.Yukinoshita;
import org.unitarou.yukinoshita.model.NodeEntity;
import org.unitarou.yukinoshita.model.NodeView;
import org.unitarou.yukinoshita.view.jface.resource.NodeImage;
import org.unitarou.yukinoshita.view.provider.nlp.NodeViewLabelProvider;

/**
 * {@link org.unitarou.yukinoshita.model.NodeView}󂯎āA
 * K؂ȃC[WƕԂxvoC_[łB
 * 
 * @author unitarou &lt;boss@unitarou.org&gt;
 */
class NodeListLabelProvider implements ITableLabelProvider, ITableColorProvider {
	
	/** Xgɕ\C[Wp̃|Wg\łB*/
	static private final ImageRegistry imageRegistry_s_ = JFaceResources.getImageRegistry();

    /** wiFL[(LIST̔wiWF)ł*/
    static private final ColorResource BACKGROUND = new ColorResource(SWT.COLOR_LIST_BACKGROUND);

    /** {̃tHgFL[(LIST̃tHgWF)ł*/
    static private final ColorResource MAIN_FONT = new ColorResource(SWT.COLOR_LIST_FOREGROUND);
    
    /** ω}wiFp̐FzłB */
    static private final ColorResource[] VARIATIONS;
    
    /** Fݒ܂B*/
    static  {
        int rs[] = new int[]{0xEE, 0xDD};
        int gs[] = new int[]{0xEE, 0xDD};
        int bs[] = new int[]{0xEE, 0xDD};
        VARIATIONS = new ColorResource[rs.length * gs.length * bs.length - rs.length];
        int index = 0;
        for (int r = 0; r < rs.length; ++r) {
            for (int b = 0; b < bs.length; ++b) {
                for (int g = 0; g < gs.length; ++g) {
                    if ((r == b) && (b == g)) {
                        continue;
                    }
                    VARIATIONS[index] = new ColorResource(rs[r], gs[g], bs[b]);
                    ++index;
                }
            }
        }
    }
    
    /**
     * e{@link NodeView}̏Ԃ\񋓌^łB
     */
    private enum NodeStatus {
    	WHITE_MOVE(NodeImage.WHITE),
    	BLACK_MOVE(NodeImage.BLACK),
    	HAS_VARIATION(NodeImage.VNONE),
    	WARNING(NodeImage.WARNING),
    	;
    	
    	private final ImageResource imageResource_;
    	
    	NodeStatus(ImageResource imageResource) {
    		imageResource_ = imageResource;
    	}
    }
    
    private CoordinatesLabelProvider coordinatesLabelProvider_;
    /**
     * 
     */
    public NodeListLabelProvider() {
        super();
        coordinatesLabelProvider_ 
        		=  Yukinoshita.context().getProvider(
        				CoordinatesLabelProvider.class, null);
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
     */
    public void dispose() {
        // do nothing
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
     */
    public Image getColumnImage(Object obj, int column) {
        if (!(obj instanceof NodeView) || (column != 1)) {
            return null;
        }
        // NodeImageォdˍ킹B
        // gݍ킹j[NȕɂāAB
        // W_V_WAW__Ȃ
        NodeView nodeView = (NodeView)obj;
        EnumSet<NodeStatus> statusSet = EnumSet.noneOf(NodeStatus.class);
        if (nodeView.getProperty(SgfId.WHITE) != null) {
        	statusSet.add(NodeStatus.WHITE_MOVE);

        } else if (nodeView.getProperty(SgfId.BLACK) != null) {
        	statusSet.add(NodeStatus.BLACK_MOVE);
        }
        
        if (nodeView.variationSize() != 0) {
        	statusSet.add(NodeStatus.HAS_VARIATION);
        }

        if (!nodeView.getInvalid().isEmpty()) {
        	statusSet.add(NodeStatus.WARNING);
        }
        
        StringBuilder builder = new StringBuilder();
        builder.append(this.getClass().toString()).append('/');
        for (NodeStatus status : statusSet) {
        	builder.append(status.name()).append(':');
        }
        
        String key = builder.toString();
        Image image = imageRegistry_s_.get(key);
        if (image != null) {
        	return image;
        }
        
        GC gc = null;
        try {
        	image = new Image(null, 16, 16);
        	gc = new GC(image);
            for (NodeStatus status : statusSet) {
            	gc.drawImage(status.imageResource_.get(), 0, 0);
            }
            imageRegistry_s_.put(key, image);
            return image;
        } finally {
        	Swts.disposeQuietly(gc);
        }
    }
    

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
     */
    public String getColumnText(Object obj, int column) {
        if (!(obj instanceof NodeView)) {
            return null;
        }
        NodeView nodeView = (NodeView)obj;
        switch (column) {
        case 1: return getMoveNumber(nodeView);
        case 2: return getMoveText(nodeView);
        case 3: return getTreeText(nodeView);
        }
        return Strings.EMPTY;
    }
    
    private String getMoveNumber(NodeView nodeView) {
    	StringBuilder sb = new StringBuilder();
    	
        Integer number = nodeView.getMove().getNumber();
        if (number != null) {
        	sb.append(number);
        }
        Double tl = nodeView.getMove().getTimeLapsed();
        if (tl != null) {
        	if (sb.length()!= 0) {
        		sb.append(' ');
        	}
        	sb.append('[').append(Strings.formatDuration(tl.intValue())).append(']');
        }
        return sb.toString();
    }

    private String getMoveText(NodeView nodeView) {
        try {
            Property property = nodeView.getProperty(SgfId.WHITE);
            if (property != null) {
                SgfPoint sgfPoint;
                sgfPoint = SgfPoint.parseMove(nodeView.getSize(), property.getString());
                return coordinatesLabelProvider_.getMoveLabel(sgfPoint);
            }

            property = nodeView.getProperty(SgfId.BLACK);
            if (property != null) {
                SgfPoint sgfPoint;
                sgfPoint = SgfPoint.parseMove(nodeView.getSize(), property.getString());
                return coordinatesLabelProvider_.getMoveLabel(sgfPoint);
            }
        } catch (TypeParseException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return Strings.EMPTY;
    }

    private String getTreeText(NodeView nodeView) {
    	return NodeViewLabelProvider.CONTEXT.defaultProvider().getLabel(nodeView);
    }
    
    /**
     * @return
     */
    public CoordinatesLabelProvider getProvider() {
    	return coordinatesLabelProvider_;
    }

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ITableColorProvider#getForeground(java.lang.Object, int)
	 */
	public Color getForeground(Object element, int columnIndex) {
        return MAIN_FONT.get();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ITableColorProvider#getBackground(java.lang.Object, int)
	 */
	public Color getBackground(Object element, int columnIndex) {
        if ((columnIndex==0) || !(element instanceof NodeView)) {
            return BACKGROUND.get();
        }
        NodeView nodeView = (NodeView)element;
        int variation = nodeView.getVariationHash();
        if (NodeEntity.NO_VARIATION == variation) {
            return BACKGROUND.get();
        }
        return VARIATIONS[variation % VARIATIONS.length].get();
	}

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.IBaseLabelProvider#addListener(org.eclipse.jface.viewers.ILabelProviderListener)
     */
    public void addListener(ILabelProviderListener arg0) {
        // do nothing
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.IBaseLabelProvider#removeListener(org.eclipse.jface.viewers.ILabelProviderListener)
     */
    public void removeListener(ILabelProviderListener arg0) {
        // do nothing
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.IBaseLabelProvider#isLabelProperty(java.lang.Object, java.lang.String)
     */
    public boolean isLabelProperty(Object arg0, String arg1) {
        return false;
    }

    /**
     * @param provider
     */
    public void setProvider(CoordinatesLabelProvider provider) {
        ArgumentChecker.throwIfNull(provider);
        coordinatesLabelProvider_ = provider;
    }
}
