/*  
 * Copyright 2005 unitarou <boss@unitarou.org>. 
 * All rights reserved.
 * 
 * This program and the accompanying materials are made available under the terms of 
 * the Common Public License v1.0 which accompanies this distribution, 
 * and is available at http://opensource.org/licenses/cpl.php
 * 
 * Contributors:
 *     unitarou - initial API and implementation
 */
package org.unitarou.yukinoshita.view.provider.rgtlp;

import org.unitarou.lang.Strings;
import org.unitarou.sgf.GameTree;
import org.unitarou.sgf.Node;
import org.unitarou.sgf.RootGameTree;
import org.unitarou.sgf.Sequence;
import org.unitarou.sgf.SgfId;
import org.unitarou.sgf.type.GameType;
import org.unitarou.sgf.type.Handicap;
import org.unitarou.sgf.type.Result;
import org.unitarou.sgf.type.SgfNumber;
import org.unitarou.sgf.util.BasicFinder;

/**
 * eNX{@link org.unitarou.yukinoshita.view.provider.rgtlp.DefaultRgtLabelProvider}
 * 󕶎ԂɁu[NxM, HA, L]|l[RE, L]|W[N]vԂ܂B
 * 
 * @author unitarou &lt;boss@unitarou.org&gt; 
 */
public class GreedyRgtLabelProvider extends DefaultRgtLabelProvider {

	/**
	 * 
	 */
	public GreedyRgtLabelProvider() {
		super();
	}

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.view.provider.rgtlp.DefaultRgtLabelProvider#displayName()
	 */
	@Override
	public String displayName() {
		return super.displayName();
	}

	/* (non-Javadoc)
	 * @see org.unitarou.yukinoshita.view.provider.rgtlp.DefaultRgtLabelProvider#getLabel(org.unitarou.sgf.RootGameTree)
	 */
	@Override
	public String getLabel(RootGameTree rootGameTree) {
		String label = super.getLabel(rootGameTree);
		if (label.length() != 0) {
			return label;
		}
		
		switch (rootGameTree.getGameType()) {
		case GAME:
			return getLabelAsGame(rootGameTree);

		case PROBLEM:
			return getLabelAsProblem(rootGameTree);
			
		case DRILL:
			return getLabelAsDrill(rootGameTree);
			
		default:
			break;
		}
		return label;
	}

	/**
	 * @param rootGameTree
	 * @return
	 */
	private String getLabelAsGame(RootGameTree rootGameTree) {
		StringBuilder builder = new StringBuilder();
		builder.append(GameType.GAME.displayName()).append('[')
		       .append(rootGameTree.getSize().toString()).append(',');
		
		String haString = makeHandicapLabel(rootGameTree);
		if (haString.length() != 0) {
			builder.append(haString).append(',');
		}
		
		builder.append(countMoves(rootGameTree)).append("").append(']');
		return builder.toString();
	}

	
	/**
	 * ̍ŏ{@link org.unitarou.sgf.Sequence}
	 * {@link SgfId#HANDICAP}^OTČꍇ
	 * ̒lxĕԂ܂B
	 * Ȃꍇ͋󕶎Ԃ܂B
	 * @param rootGameTree
	 * @return
	 */
	private String makeHandicapLabel(RootGameTree rootGameTree) {
		String datum = BasicFinder.findDatum(rootGameTree.getSequence(), SgfId.HANDICAP);
		if (datum.length() == 0) {
			return Strings.EMPTY;
		}
		
		SgfNumber number = SgfNumber.parseQuietly(datum);
		if (number == null) {
			return Strings.EMPTY;
		}
		
		Handicap handicap = new Handicap(number.getIntValue());
		return handicap.displayName();
	}
	
	/**
	 * ̃CpXMOVEJEgĕԂ܂B
	 * @param rootGameTree
	 * @return
	 */
	private int countMoves(RootGameTree rootGameTree) {
		int ret = 0;
		GameTree gameTree = rootGameTree;
		while (gameTree != null) {
			Sequence sequence = gameTree.getSequence();
			for (Node node : sequence) {
				if (node.contains(SgfId.BLACK) || node.contains(SgfId.WHITE)) {
					++ret;
				}
			}
			gameTree = (gameTree.getChildrenSize() == 0) ? null : gameTree.getChild(0);
		}
		return ret;
	}

	/**
	 * @param rootGameTree
	 * @return
	 */
	private String getLabelAsProblem(RootGameTree rootGameTree) {
		StringBuilder builder = new StringBuilder();
		builder.append(GameType.PROBLEM.displayName()).append('[');
		
		String result = makeResultLabel(rootGameTree);
		if (result.length() != 0) {
			builder.append(result).append(',');
		}
		builder.append(countMoves(rootGameTree)).append("").append(']');
		return builder.toString();
	}

	/**
	 * @param rootGameTree
	 * @return
	 */
	private String makeResultLabel(RootGameTree rootGameTree) {
		String datum = BasicFinder.findDatum(rootGameTree.getSequence(), SgfId.RESULT);
		if (datum.length() == 0) {
			return Strings.EMPTY;
		}
		Result result = Result.parse(datum);
		switch (result.getArchetype()) {
		case OTHER:
			return result.getValue();
			
		case BLACK_WIN:
		case WHITE_WIN:
			return result.getArchetype().displayName() + result.getScore();
			
		
		default:
			return result.getArchetype().displayName();
		}
	}

	/**
	 * @param rootGameTree
	 * @return
	 */
	private String getLabelAsDrill(RootGameTree rootGameTree) {
		StringBuilder builder = new StringBuilder();
		builder.append(GameType.DRILL.displayName()).append('[')
		       .append(countProblems(rootGameTree)).append("").append(']');
		return builder.toString();
	}
	
	/**
	 * Wɓo^ꂽ̐Ԃ܂B
	 * @param rootGameTree
	 * @return
	 */
	private int countProblems(RootGameTree rootGameTree) {
		String[] data = BasicFinder.findData(rootGameTree.getSequence(), SgfId.INPUT_FILES);
		if (data == null) {
			return 0;
		}
		return data.length;
	}
}
