package map;

import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.GeneralPath;
import java.awt.geom.PathIterator;
import java.awt.geom.Rectangle2D;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.regex.Pattern;

/**
 * 地図データを扱うクラスです。
 * @author Kumano Tatsuo
 * 作成日: 2003/12/11
 */
public class MapData implements Comparable {
	/**
	 * 隣接グラフ
	 */
	private Map<String, Collection<PolygonData>> adjacentGraph; // String -> Colleciton<Polygon> の Map

	/**
	 * 弧とポリゴンの対応表
	 */
	private Map<ArcData, Collection<String>> arcPolygonMap; // Arc -> Collection<String> の Map

	/**
	 * ベースディレクトリ
	 */
	private String baseDir;

	/**
	 * 図郭の端にあるポリゴンの一覧
	 */
	private Map<String, String> edgePolygons; // String -> String の Map

	/**
	 * 駅の点データ
	 */
	private Map<String, PointData> eki; // String -> Point の Map

	/**
	 * 行政界の弧データ
	 */
	private Map<String, ArcData> gyousei; // String  -> Arc の Map

	/**
	 * 図葉名
	 */
	private String mapName;

	/**
	 * 内水面のポリゴンデータ
	 */
	private Map<String, PolygonData> mizu; // String -> Polygon の Map

	/**
	 * 内水面の弧データ
	 */
	private Map<String, ArcData> mizuArc; // String -> Arc の Map

	/**
	 * その他の弧データ
	 */
	private Map<String, ArcData> others; // String -> Arc の Map

	/**
	 * この地図が表す領域（仮想座標）
	 */
	private Rectangle2D rectangle; // この地図が表す領域（仮想座標）

	/**
	 * 道路の弧データ
	 */
	private Map<String, ArcData> roadArc; // String -> Arc の Map

	/**
	 * 市町界のポリゴンデータ
	 */
	private Map<String, PolygonData> si_tyo; // String -> Polygon の Map

	/**
	 * 建物のポリゴンデータ
	 */
	private Map<String, PolygonData> tatemono; // String -> Tatemono の Map

	/**
	 * 建物界の弧データ
	 */
	private Map<String, ArcData> tatemonoArc; // String -> Arc の Map

	/**
	 * 町丁目のポリゴンデータ
	 */
	private Map<String, PolygonData> tyome; // String -> Polygon の Map

	/**
	 * 城地のポリゴンデータ
	 */
	private Map<String, PolygonData> zyouti; // String -> Polygon の Map

	/** 地図を初期化します。
	 * @param baseDir 地図データのあるディレクトリ
	 * @param mapName 地図の名前
	 * @throws Exception 例外
	 */
	public MapData(String baseDir, String mapName) throws Exception {
		setBaseDir(baseDir);
		setMapName(mapName);
		loadRectangle();
		this.edgePolygons = new HashMap<String, String>();
		this.arcPolygonMap = new HashMap<ArcData, Collection<String>>();
	}

	/**
	 * テスト用のメソッドです。
	 * @param args コマンドライン引数
	 * @throws Exception 例外
	 */
	public static void main(String[] args) throws Exception {
		MapData mapData = new MapData("/home/kumano/map_data/suchi_chizu_2500/hyougo/", "05OF833");
		System.out.println("eki = " + mapData.getEki());
	}

	/** 丁目の隣接グラフを計算します。
	 * @throws Exception 例外
	 */
	private void calcAdjacencyGraph() throws Exception {
		this.adjacentGraph = new HashMap<String, Collection<PolygonData>>();
		if (this.tyome == null) {
			loadTyome();
		}
		for (Collection<String> polygonNames : this.arcPolygonMap.values()) {
			Collection<PolygonData> adjacencyPolygons = new ArrayList<PolygonData>();
			for (String polygonName : polygonNames) {
				PolygonData polygon = this.tyome.get(polygonName);
				adjacencyPolygons.add(polygon);
			}
			if (adjacencyPolygons.size() == 2) {
				Iterator<PolygonData> iter2 = adjacencyPolygons.iterator();
				PolygonData polygon = iter2.next();
				PolygonData polygon2 = iter2.next();
				if (polygon != null) {
					if (!this.adjacentGraph.containsKey(polygon.getPolygonName())) {
						this.adjacentGraph.put(polygon.getPolygonName(),
								new ArrayList<PolygonData>());
					}
					if (!this.adjacentGraph.containsKey(polygon2.getPolygonName())) {
						this.adjacentGraph.put(polygon2.getPolygonName(),
								new ArrayList<PolygonData>());
					}
					this.adjacentGraph.get(polygon.getPolygonName()).add(polygon2);
					this.adjacentGraph.get(polygon2.getPolygonName()).add(polygon);
				}
			}
		}
	}

	/** このオブジェクトと他のオブジェクトの大小関係を比較します。
	 * @param o 比較対象のオブジェクト
	 * @return 大小関係
	 */
	public int compareTo(Object o) {
		return this.mapName.compareTo(((MapData) o).getMapName());
	}

	/**
	 * 駅の点データを開放します。
	 */
	void freeEki() {
		this.eki = null;
	}

	/**
	 * 行政界の弧データを開放します。
	 */
	void freeGyousei() {
		this.gyousei = null;
	}

	/**
	 * 水のポリゴンデータを開放します。
	 */
	void freeMizu() {
		this.mizu = null;
	}

	/**
	 * 内水面界の弧データを開放します。
	 */
	void freeMizuArc() {
		this.mizuArc = null;
	}

	/**
	 * 鉄道、場地界の弧データを開放します。
	 */
	void freeOthers() {
		this.others = null;
	}

	/**
	 * 道路の弧データを開放します。
	 */
	void freeRoadArc() {
		this.roadArc = null;
	}

	/**
	 * 市町界のポリゴンデータを開放します。
	 */
	void freeSi_tyo() {
		this.si_tyo = null;
	}

	/**
	 * 建物のポリゴンデータを開放します。
	 */
	void freeTatemono() {
		this.tatemono = null;
	}

	/**
	 * 建物の弧データを開放します。
	 */
	void freeTatemonoArc() {
		this.tatemonoArc = null;
	}

	/**
	 * 大字、町丁目のポリゴンデータを開放します。
	 */
	void freeTyome() {
		this.tyome = null;
	}

	/**
	 * 場地界のポリゴンデータを開放します。
	 */
	void freeZyouti() {
		this.zyouti = null;
	}

	/** 丁目の隣接グラフを取得します。
	 * @return 丁目の隣接グラフ
	 * @throws Exception 例外
	 */
	Map<String, Collection<PolygonData>> getAdjacentGraph() throws Exception {
		if (this.adjacentGraph == null) {
			calcAdjacencyGraph();
		}
		return this.adjacentGraph;
	}

	/**
	 * 弧とポリゴンのMapを取得します。
	 * @return 弧とポリゴンの Map
	 */
	Map<ArcData, Collection<String>> getArcPolygonMap() {
		return this.arcPolygonMap;
	}

	/**
	 * 地図データのあるディレクトリを取得します。
	 * @return ディレクトリ
	 */
	String getBaseDir() {
		return this.baseDir;
	}

	/**
	 * 地図が表す領域を取得します。
	 * @return 領域
	 */
	public Rectangle2D getBounds() {
		return this.rectangle;
	}

	/**
	 * 図郭にまたがるポリゴンの一覧を取得します。
	 * Mapのキーは図郭上の線分の文字列表現、
	 * 値はポリゴンの名前です。
	 * @return 図郭にまたがるポリゴンの一覧
	 */
	Map<String, String> getEdgePolygons() {
		return this.edgePolygons;
	}

	/**
	 * 駅の点データを取得します。
	 * @throws Exception 例外
	 * @return 駅の点データ
	 */
	Map<String, PointData> getEki() throws Exception {
		if (this.eki == null) {
			loadEki();
		}
		return this.eki;
	}

	/**
	 * 行政界の弧データを取得します。
	 * @return 行政界の弧データ
	 * @throws Exception 例外
	 */
	Map<String, ArcData> getGyousei() throws Exception {
		if (this.gyousei == null) {
			loadGyousei();
		}
		return this.gyousei;
	}

	/**
	 * 地図のファイル名を取得します。
	 * @return 地図のファイル名
	 */
	String getMapName() {
		return this.mapName;
	}

	/** 内水面のポリゴンデータを取得します。
	 * @return 内水面のポリゴンデータ
	 * @throws Exception 例外
	 */
	Map<String, PolygonData> getMizu() throws Exception {
		if ((this.mizu == null) && (this.mizuArc != null)) {
			loadMizu();
		}
		return this.mizu;
	}

	/**
	 * 内水面界の弧データを取得します。
	 * @return 内水面界の弧データ
	 * @throws Exception 例外
	 */
	Map<String, ArcData> getMizuArc() throws Exception {
		if (this.mizuArc == null) {
			loadMizuArc();
		}
		return this.mizuArc;
	}

	/**
	 * 鉄道、場地界の弧データを取得します。
	 * @throws Exception 鉄道、場地界の弧データ
	 * @return 例外
	 */
	Map<String, ArcData> getOthers() throws Exception {
		if (this.others == null) {
			loadOthers();
		}
		return this.others;
	}

	/**
	 * 道路の弧データを取得します。
	 * @throws Exception 例外
	 * @return 道路の弧データ
	 */
	Map<String, ArcData> getRoadArc() throws Exception {
		if (this.roadArc == null) {
			loadRoadArc();
		}
		return this.roadArc;
	}

	/**
	 * 市町界のポリゴンデータを取得します。
	 * @return 市町界のポリゴンデータ
	 * @throws Exception 例外
	 */
	Map<String, PolygonData> getSi_tyo() throws Exception {
		if (this.gyousei == null) {
			loadGyousei();
		}
		if (this.si_tyo == null) {
			loadSi_tyo();
		}
		return this.si_tyo;
	}

	/**
	 * 建物のポリゴンデータを取得します。
	 * @throws Exception 例外
	 * @return 建物のポリゴンデータ
	 */
	Map<String, PolygonData> getTatemono() throws Exception {
		if (this.tatemonoArc == null) {
			loadTatemonoArc();
		}
		if (this.tatemono == null) {
			loadTatemono();
		}
		return this.tatemono;
	}

	/**
	 * 建物界の弧データを取得します。
	 * @throws Exception 例外
	 * @return 建物界の弧データ
	 */
	Map<String, ArcData> getTatemonoArc() throws Exception {
		if (this.tatemonoArc == null) {
			loadTatemonoArc();
		}
		return this.tatemonoArc;
	}

	/**
	 * 大字、町丁目界のポリゴンデータを取得します。
	 * @return 大字、町丁目界のポリゴンデータ
	 * @throws Exception 例外
	 */
	Map<String, PolygonData> getTyome() throws Exception {
		if (this.gyousei == null) {
			loadGyousei();
		}
		if (this.tyome == null) {
			loadTyome();
		}
		return this.tyome;
	}

	/**
	 * 場地のポリゴンデータを取得します。
	 * @throws Exception 例外
	 * @return 場地のポリゴンデータ
	 */
	Map<String, PolygonData> getZyouti() throws Exception {
		if (this.others == null) {
			loadOthers();
		}
		if (this.zyouti == null) {
			loadZyouti();
		}
		return this.zyouti;
	}

	/**
	 * 駅の点データを持っているかどうかを取得します。
	 * @return 駅の点データを持っているかどうか
	 */
	boolean hasEki() {
		return this.eki != null;
	}

	/**
	 * 行政界の弧データを持っているかどうかを取得します。
	 * @return 行政界の弧データを持っているか
	 */
	boolean hasGyousei() {
		return this.gyousei != null;
	}

	/**
	 * 内水面のポリゴンデータを持っているかどうかを取得します。
	 * @return 内水面のポリゴンデータを持っているかどうか
	 */
	boolean hasMizu() {
		return this.mizu != null;
	}

	/**
	 * 内水面界の弧データを持っているかどうかを取得します。
	 * @return 内水面界の弧データを持っているかどうか
	 */
	boolean hasMizuArc() {
		return this.mizuArc != null;
	}

	/**
	 * 鉄道、場地界の弧データを持っているかどうかを取得します。
	 * @return 鉄道、場地界の弧データを持っているかどうか
	 */
	boolean hasOthers() {
		return this.others != null;
	}

	/**
	 * 道の弧データを持っているかどうかを取得します。
	 * @return 道の弧データを持っているかどうか
	 */
	boolean hasRoadArc() {
		return this.roadArc != null;
	}

	/**
	 * 市町界のポリゴンデータを持っているかどうかを取得します。
	 * @return 市町界のポリゴンデータを持っているかどうか
	 */
	boolean hasSi_tyo() {
		return this.si_tyo != null;
	}

	/**
	 * 建物のポリゴンデータを持っているかどうかを取得します。
	 * @return 建物のポリゴンデータを持っているかどうか
	 */
	boolean hasTatemono() {
		return this.tatemono != null;
	}

	/**
	 * 建物界の弧データを持っているかどうかを取得します。
	 * @return 建物界の弧データを持っているかどうか
	 */
	boolean hasTatemonoArc() {
		return this.tatemonoArc != null;
	}

	/**
	 * 大字、町丁目のポリゴンデータを持っているかどうかを取得します。
	 * @return 大字、町丁目のポリゴンデータを持っているかどうか
	 */
	boolean hasTyome() {
		return this.tyome != null;
	}

	/**
	 * 場地界のポリゴンデータを持っているかどうかを取得します。
	 * @return 場地界のポリゴンデータを持っているかどうか
	 */
	boolean hasZyouti() {
		return this.zyouti != null;
	}

	/**
	 * 弧の情報をファイルから読み込みます。
	 * @param arcs 弧
	 * @param in 入力ストリーム
	 * @throws IOException 例外
	 */
	private void loadArc(Map<String, ArcData> arcs, BufferedReader in) throws IOException {
		String line;
		String arcName = null;
		int type = 0;
		int tag = 0;
		GeneralPath path = null;
		while ((line = in.readLine()) != null) {
			StringTokenizer tokenizer = new StringTokenizer(line, ",");
			if (tokenizer.countTokens() == 4) {
				// 新しい弧が始まるとき
				if (path != null) {
					arcs.put(arcName, new ArcData(arcName, path, type, tag));
				}
				String code = tokenizer.nextToken(); // 図式分類コード
				if (code.equals("L1101")) {
					type = ArcData.TYPE_GYOUSEI_PREFECTURE;
				} else if (code.equals("L1103")) {
					type = ArcData.TYPE_GYOUSEI_CITY;
				} else if (code.equals("L1104")) {
					type = ArcData.TYPE_GYOUSEI_VILLAGE;
				} else if (code.equals("L1106")) {
					type = ArcData.TYPE_GYOUSEI_CHOME;
				} else if (code.equals("L2300")) {
					type = ArcData.TYPE_RAILWAY;
				} else if (code.equals("L2110")) {
					type = ArcData.TYPE_ROAD;
				} else if (code.equals("L6241")) {
					type = ArcData.TYPE_ZYOTI_RAILROAD;
				} else if (code.equals("L6242")) {
					type = ArcData.TYPE_ZYOTI_PARK;
				} else if (code.equals("L6243")) {
					type = ArcData.TYPE_ZYOTI_SCHOOL;
				} else if (code.equals("L6244")) {
					type = ArcData.TYPE_ZYOTI_TEMPLE;
				} else if (code.equals("L6215")) {
					type = ArcData.TYPE_ZYOTI_GRAVEYARD;
				} else if (code.equals("L6200")) {
					type = ArcData.TYPE_ZYOTI_OTHER;
				} else if (code.equals("L5101")) {
					type = ArcData.TYPE_MIZU_INSIDE;
				} else if (code.equals("L5106")) {
					type = ArcData.TYPE_MIZU_SEASHORE;
				} else {
					type = ArcData.TYPE_UNKNOWN;
				}
				tag = Integer.parseInt(tokenizer.nextToken()); // 線種タグ
				arcName = tokenizer.nextToken(); // 個別番号
				path = null;
			} else if (tokenizer.countTokens() == 2) {
				// 弧を構成する座標のとき
				float y = -Float.parseFloat(tokenizer.nextToken()); // y 座標
				float x = Float.parseFloat(tokenizer.nextToken()); // x 座標
				if (path != null) {
					path.lineTo(x + (float) getBounds().getX(), y + (float) getBounds().getMaxY());
				} else {
					path = new GeneralPath();
					path.moveTo(x + (float) getBounds().getX(), y + (float) getBounds().getMaxY());
				}
			}
		}
		if (path != null) {
			arcs.put(arcName, new ArcData(arcName, path, type, tag));
		}
	}

	/**
	 * 弧の情報を属性情報をファイルから読み込みます。
	 * @param arcs 弧
	 * @param in 入力ストリーム
	 * @throws Exception 例外
	 */
	private void loadArcAttribute(Map<String, ArcData> arcs, BufferedReader in) throws Exception {
		String line;
		while ((line = in.readLine()) != null) {
			StringTokenizer tokenizer = new StringTokenizer(line, ",");
			if (tokenizer.countTokens() > 0) {
				String firstToken = tokenizer.nextToken(); // 「FH」or 図式分類コード
				if (!firstToken.equals("FH")) {
					// 属性レコードのとき
					if (tokenizer.countTokens() > 2) {
						String name = tokenizer.nextToken(); // 個別番号
						tokenizer.nextToken(); // 属性フィールドの個数
						String attribute = tokenizer.nextToken(); // 属性
						if (arcs.containsKey(name)) {
							ArcData arc = arcs.get(name);
							attribute = attribute.replaceFirst("電気鉄道", "電鉄");
							arc.setAttribute(attribute);
							if (attribute.indexOf("新幹線") > 0) {
								arc.setRailwayType(ArcData.RAILWAY_JR_SHINKANSEN);
							} else if (attribute.startsWith("ＪＲ")) {
								arc.setRailwayType(ArcData.RAILWAY_JR);
							} else {
								arc.setRailwayType(ArcData.RAILWAY_OTHER);
							}
							if ((attribute.indexOf("高速") > 0) || attribute.endsWith("自動車道")) {
								arc.setRoadType(ArcData.ROAD_HIGHWAY);
							} else if (attribute.startsWith("国道")) {
								arc.setRoadType(ArcData.ROAD_KOKUDO);
							} else if (attribute.startsWith("県道") || attribute.startsWith("府道")
									|| attribute.startsWith("道道") || attribute.startsWith("都道")) {
								arc.setRoadType(ArcData.ROAD_KENDO);
							} else if (attribute.startsWith("主要地方道")) {
								arc.setRoadType(ArcData.ROAD_CHIHODO);
								arc.setAttribute(attribute.replaceFirst("主要地方道", ""));
							} else {
								arc.setRoadType(ArcData.ROAD_MAJOR);
							}
						}
					}
				}
			}
		}
	}

	/** 駅の点データを読み込みます。
	 * @throws Exception 例外
	 */
	void loadEki() throws Exception {
		String fileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "others" + Const.SEPARATOR + "eki.pnt";
		String attributeFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "others" + Const.SEPARATOR + "eki.atr";
		this.eki = new HashMap<String, PointData>();
		if (new File(fileName).canRead()) {
			loadPoint(this.eki, new BufferedReader(new InputStreamReader(new FileInputStream(
					fileName))));
			if (new File(attributeFileName).canRead()) {
				loadPointAttribute(this.eki, new BufferedReader(new InputStreamReader(
						new FileInputStream(attributeFileName), "SJIS")),
						PointData.CLASSIFICATION_STATION);
			}
		}
	}

	/** 行政界の弧ファイルを読み込みます。
	 * @throws Exception 例外
	 */
	void loadGyousei() throws Exception {
		String fileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "gyousei" + Const.SEPARATOR + "gyousei.arc";
		this.gyousei = new HashMap<String, ArcData>();
		if (new File(fileName).canRead()) {
			loadArc(this.gyousei, new BufferedReader(new InputStreamReader(new FileInputStream(
					fileName))));
		}
	}

	/** 内水面のポリゴンファイルを読み込みます。
	 * @throws Exception 例外
	 */
	void loadMizu() throws Exception {
		String polygonFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "mizu" + Const.SEPARATOR + "mizu.pgn";
		if (new File(polygonFileName).canRead()) {
			this.mizu = new HashMap<String, PolygonData>();
			loadPolygon(this.mizu, this.mizuArc, new BufferedReader(new InputStreamReader(
					new FileInputStream(polygonFileName))));
			String attributeFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
					+ Const.SEPARATOR + "mizu" + Const.SEPARATOR + "mizu.atr";
			if (new File(attributeFileName).canRead()) {
				loadPolygonAttribute(this.mizu, new BufferedReader(new InputStreamReader(
						new FileInputStream(attributeFileName), "SJIS")),
						PolygonData.CLASSIFICATION_RIVER);
			}
		}
	}

	/** 水界の弧ファイルを読み込みます。
	 * @throws Exception 例外
	 */
	void loadMizuArc() throws Exception {
		String fileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "mizu" + Const.SEPARATOR + "mizu.arc";
		if (new File(fileName).canRead()) {
			this.mizuArc = new HashMap<String, ArcData>();
			loadArc(this.mizuArc, new BufferedReader(new InputStreamReader(new FileInputStream(
					fileName))));
		}
	}

	/**
	 * 鉄道、場地界の弧データを読み込みます。
	 * @throws Exception 例外
	 */
	void loadOthers() throws Exception {
		this.others = new HashMap<String, ArcData>();
		String attributeFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "others" + Const.SEPARATOR + "tetudou.atr";
		loadArc(this.others, new BufferedReader(new InputStreamReader(new FileInputStream(
				this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase() + Const.SEPARATOR
						+ "others" + Const.SEPARATOR + "others.arc"))));
		if (new File(attributeFileName).canRead()) {
			loadArcAttribute(this.others, new BufferedReader(new InputStreamReader(
					new FileInputStream(attributeFileName), "SJIS")));
		}
	}

	/** 点の情報をファイルから読み込みます。
	 * @param points 点
	 * @param in 入力ストリーム
	 * @throws IOException 例外
	 */
	private void loadPoint(Map<String, PointData> points, BufferedReader in) throws IOException {
		String line;
		while ((line = in.readLine()) != null) {
			StringTokenizer tokenizer = new StringTokenizer(line, ",");
			if (tokenizer.countTokens() == 4) {
				String classificationCode = tokenizer.nextToken(); // 図式分類コード
				int code;
				if (classificationCode.equals("P2420")) {
					code = PointData.CLASSIFICATION_STATION;
				} else if (classificationCode.equalsIgnoreCase("P7301")) {
					code = PointData.CLASSIFICATION_DATUMS;
				} else {
					code = PointData.CLASSIFICATION_UNKNOWN;
				}
				String name = tokenizer.nextToken(); // 個別番号
				double y = -Double.parseDouble(tokenizer.nextToken()); // y 座標
				double x = Double.parseDouble(tokenizer.nextToken()); // x 座標
				PointData point = new PointData(name, code, x + getBounds().getX(), y
						+ getBounds().getMaxY());
				points.put(name, point);
			}
		}
	}

	/** 点データの属性情報をファイルから読み込みます。
	 * @param points 点
	 * @param in 入力ストリーム
	 * @param classificationCode 図式分類コードの例
	 * @throws Exception 例外
	 */
	private void loadPointAttribute(Map<String, PointData> points, BufferedReader in,
			int classificationCode) throws Exception {
		String line;
		while ((line = in.readLine()) != null) {
			StringTokenizer tokenizer = new StringTokenizer(line, ",");
			if (tokenizer.countTokens() > 0) {
				String firstToken = tokenizer.nextToken(); // 「FH」or 図式分類コード
				if (!firstToken.equals("FH")) {
					// 属性レコードのとき
					if (classificationCode == PointData.CLASSIFICATION_STATION) {
						// 駅
						if (tokenizer.countTokens() == 3) {
							String name = tokenizer.nextToken(); // 個別番号
							tokenizer.nextToken(); // 属性フィールドの個数
							String attribute = tokenizer.nextToken(); // 属性
							if (!attribute.endsWith("駅")) {
								attribute = attribute + "駅";
							}
							if (points.containsKey(name)) {
								points.get(name).setAttribute(attribute);
							}
						}
					} else if (classificationCode == PointData.CLASSIFICATION_DATUMS) {
						// 三角点
						if (tokenizer.countTokens() == 13) {
							String name = tokenizer.nextToken(); // 個別番号
							tokenizer.nextToken(); // 属性フィールドの個数
							tokenizer.nextToken(); // 3 次メッシュコード
							tokenizer.nextToken(); // 点コード
							String attribute = tokenizer.nextToken(); // 属性
							if (points.containsKey(name)) {
								points.get(name).setAttribute(attribute);
							}
						}
					}
				}
			}
		}
	}

	/** ポリゴンの情報をファイルから読み込みます。
	 * @param polygons ポリゴン
	 * @param arcs ポリゴンを構成する弧
	 * @param in 入力ストリーム
	 * @throws IOException 例外
	 */
	private void loadPolygon(Map<String, PolygonData> polygons, Map<String, ArcData> arcs,
			BufferedReader in) throws IOException {
		String line;
		String polygonName = null;
		GeneralPath path = null;
		int type = 0;
		double x = 0;
		double y = 0;
		while ((line = in.readLine()) != null) {
			StringTokenizer tokenizer = new StringTokenizer(line, ",");
			if (tokenizer.countTokens() == 5) {
				// 新しいポリゴンが始まるとき
				if ((path != null) && (polygonName != null)) {
					polygons.put(polygonName, new PolygonData(polygonName, new Area(path), type, x
							+ getBounds().getX(), y + getBounds().getMaxY()));
				}
				type = parsePolygonType(tokenizer.nextToken()); // 図式分類コード
				polygonName = tokenizer.nextToken(); // 個別番号
				tokenizer.nextToken(); // 当該ポリゴンを構成するアーク指定レコードの数
				y = -Double.parseDouble(tokenizer.nextToken()); // 代表点の y 座標
				x = Double.parseDouble(tokenizer.nextToken()); // 代表点の x 座標
				path = new GeneralPath();
			} else if (tokenizer.countTokens() == 1) {
				// ポリゴンを構成する弧のとき
				int arcIndex = Integer.parseInt(tokenizer.nextToken());
				// アーク指定レコード
				String arcName = new Integer(Math.abs(arcIndex)).toString();
				ArcData arc = arcs.get(arcName);
				if (arcIndex < 0) {
					path.append(arc.getReversalPath(), true);
				} else {
					path.append(arc.getPath(), true);
				}
				if (!this.arcPolygonMap.containsKey(arc)) {
					this.arcPolygonMap.put(arc, new ArrayList<String>());
				}
				this.arcPolygonMap.get(arc).add(polygonName);
				if (arc.getTag() == ArcData.TAG_EDGE_OF_MAP) {
					PathIterator iter = arc.getPath().getPathIterator(new AffineTransform());
					double[] coords = new double[6];
					iter.currentSegment(coords);
					int x1 = (int) coords[0];
					int y1 = (int) coords[1];
					if (!iter.isDone()) {
						iter.next();
						if (iter.currentSegment(coords) == PathIterator.SEG_LINETO) {
							int x2 = (int) coords[0];
							int y2 = (int) coords[1];
							if ((x1 < x2) || (y1 < y2)) {
								this.edgePolygons.put(x1 + "_" + y1 + "_" + x2 + "_" + y2,
										polygonName);
							} else {
								this.edgePolygons.put(x2 + "_" + y2 + "_" + x1 + "_" + y1,
										polygonName);
							}
							if (!iter.isDone()) {
								iter.next();
								if (iter.currentSegment(coords) == PathIterator.SEG_LINETO) {
									int x3 = (int) coords[0];
									int y3 = (int) coords[1];
									if ((x2 < x3) || (y2 < y3)) {
										this.edgePolygons.put(x2 + "_" + y2 + "_" + x3 + "_" + y3,
												polygonName);
									} else {
										this.edgePolygons.put(x3 + "_" + y3 + "_" + x2 + "_" + y2,
												polygonName);
									}
									if (!iter.isDone()) {
										iter.next();
										if (iter.currentSegment(coords) == PathIterator.SEG_LINETO) {
											int x4 = (int) coords[0];
											int y4 = (int) coords[1];
											if ((x3 < x4) || (y3 < y4)) {
												this.edgePolygons.put(x3 + "_" + y3 + "_" + x4
														+ "_" + y4, polygonName);
											} else {
												this.edgePolygons.put(x4 + "_" + y4 + "_" + x3
														+ "_" + y3, polygonName);
											}
										}
									}
								}
							}
						}
					}
				}
			}
			if ((path != null) && (polygonName != null)) {
				polygons.put(polygonName, new PolygonData(polygonName, new Area(path), type, x
						+ getBounds().getX(), y + getBounds().getMaxY()));
			}
		}
	}

	/** ポリゴンの属性情報をファイルから読み込みます。
	 * @param polygons ポリゴン
	 * @param in 入力ストリーム
	 * @param classificationCode 図式分類コードの例
	 * @throws Exception 例外
	 */
	private void loadPolygonAttribute(Map<String, PolygonData> polygons, BufferedReader in,
			int classificationCode) throws Exception {
		String line;
		while ((line = in.readLine()) != null) {
			StringTokenizer tokenizer = new StringTokenizer(line, ",");
			if (tokenizer.countTokens() > 0) {
				String firstToken = tokenizer.nextToken(); // 「FH」or 図式分類コード
				if (!firstToken.equals("FH")) {
					// 属性レコードのとき
					int type = parsePolygonType(firstToken);
					String polygonName = tokenizer.nextToken(); // 個別番号
					if (tokenizer.hasMoreTokens()) {
						tokenizer.nextToken(); // 属性フィールドの数
					}
					StringBuffer stringBuffer = new StringBuffer();
					if (tokenizer.hasMoreTokens()) {
						if ((classificationCode == PolygonData.CLASSIFICATION_SI_TYO)
								|| (classificationCode == PolygonData.CLASSIFICATION_TYOME)) {
							tokenizer.nextToken(); // 属性フィールド 1
						} else if (classificationCode == PolygonData.CLASSIFICATION_BUILDING) {
							// 属性フィールド 1
							polygons.get(polygonName).setTatemonoCode(
									Integer.parseInt(tokenizer.nextToken()));
						} else {
							stringBuffer.append(tokenizer.nextToken());
							// 属性フィールド 1
						}
					}
					if (tokenizer.hasMoreTokens()) {
						stringBuffer.append(tokenizer.nextToken());
						// 属性フィールド 2
					}
					String attribute = stringBuffer.toString();
					if (classificationCode == PolygonData.CLASSIFICATION_TYOME) {
						attribute = Pattern.compile("二十丁目$").matcher(attribute).replaceFirst("20");
						attribute = Pattern.compile("十九丁目$").matcher(attribute).replaceFirst("19");
						attribute = Pattern.compile("十八丁目$").matcher(attribute).replaceFirst("18");
						attribute = Pattern.compile("十七丁目$").matcher(attribute).replaceFirst("17");
						attribute = Pattern.compile("十六丁目$").matcher(attribute).replaceFirst("16");
						attribute = Pattern.compile("十五丁目$").matcher(attribute).replaceFirst("15");
						attribute = Pattern.compile("十四丁目$").matcher(attribute).replaceFirst("14");
						attribute = Pattern.compile("十三丁目$").matcher(attribute).replaceFirst("13");
						attribute = Pattern.compile("十二丁目$").matcher(attribute).replaceFirst("12");
						attribute = Pattern.compile("十一丁目$").matcher(attribute).replaceFirst("11");
						attribute = Pattern.compile("十丁目$").matcher(attribute).replaceFirst("10");
						attribute = Pattern.compile("九丁目$").matcher(attribute).replaceFirst("９");
						attribute = Pattern.compile("八丁目$").matcher(attribute).replaceFirst("８");
						attribute = Pattern.compile("七丁目$").matcher(attribute).replaceFirst("７");
						attribute = Pattern.compile("六丁目$").matcher(attribute).replaceFirst("６");
						attribute = Pattern.compile("五丁目$").matcher(attribute).replaceFirst("５");
						attribute = Pattern.compile("四丁目$").matcher(attribute).replaceFirst("４");
						attribute = Pattern.compile("三丁目$").matcher(attribute).replaceFirst("３");
						attribute = Pattern.compile("二丁目$").matcher(attribute).replaceFirst("２");
						attribute = Pattern.compile("一丁目$").matcher(attribute).replaceFirst("１");
					} else if (classificationCode == PolygonData.CLASSIFICATION_BUILDING) {
						attribute = Pattern.compile("郵便局$").matcher(attribute).replaceFirst("局");
						attribute = Pattern.compile("警察署$").matcher(attribute).replaceFirst("署");
						attribute = Pattern.compile("消防署$").matcher(attribute).replaceFirst("署");
						attribute = Pattern.compile("コミュニティーセンター$").matcher(attribute)
								.replaceFirst("コミセン");
						attribute = Pattern.compile("小学校$").matcher(attribute).replaceFirst("小");
						attribute = Pattern.compile("中学校$").matcher(attribute).replaceFirst("中");
						attribute = Pattern.compile("高校$").matcher(attribute).replaceFirst("高");
						attribute = Pattern.compile("高等学校$").matcher(attribute).replaceFirst("高");
						attribute = Pattern.compile("工業高等専門学校$").matcher(attribute).replaceFirst(
								"工専");
						attribute = Pattern.compile("短期大学$").matcher(attribute).replaceFirst("短大");
						attribute = Pattern.compile("大学校$").matcher(attribute).replaceFirst("大");
						attribute = Pattern.compile("大学$").matcher(attribute).replaceFirst("大");
						attribute = Pattern.compile("センター$").matcher(attribute).replaceFirst("センタ");
						attribute = Pattern.compile("センタ−$").matcher(attribute).replaceFirst("センタ");
						attribute = Pattern.compile("公共職業安定所$").matcher(attribute).replaceFirst(
								"職安");
					} else if ((classificationCode == PolygonData.CLASSIFICATION_RAILROAD)
							|| (classificationCode == PolygonData.CLASSIFICATION_PARK)
							|| (classificationCode == PolygonData.CLASSIFICATION_SCHOOL)
							|| (classificationCode == PolygonData.CLASSIFICATION_TEMPLE)
							|| (classificationCode == PolygonData.CLASSIFICATION_GRAVEYARD)
							|| (classificationCode == PolygonData.CLASSIFICATION_OTHER)) {
						attribute = Pattern.compile("小学校$").matcher(attribute).replaceFirst("小");
						attribute = Pattern.compile("中学校").matcher(attribute).replaceAll("中");
						attribute = Pattern.compile("高校").matcher(attribute).replaceAll("高");
						attribute = Pattern.compile("高等学校").matcher(attribute).replaceAll("高");
						attribute = Pattern.compile("工業高等専門学校$").matcher(attribute).replaceFirst(
								"工専");
						attribute = Pattern.compile("短期大学$").matcher(attribute).replaceFirst("短大");
						attribute = Pattern.compile("大学校$").matcher(attribute).replaceFirst("大");
						attribute = Pattern.compile("大学$").matcher(attribute).replaceFirst("大");
					}
					polygons.get(polygonName).setAttribute(attribute);
				}
			}
		}
	}

	/** 地図がある領域をファイルから読み込みます。
	 * @throws Exception 例外
	 */
	private void loadRectangle() throws Exception {
		final BufferedReader in = new BufferedReader(new InputStreamReader(new FileInputStream(
				this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase() + Const.SEPARATOR
						+ this.mapName.toLowerCase() + ".txt")));
		String line;
		while ((line = in.readLine()) != null) {
			StringTokenizer tokenizer = new StringTokenizer(line, ",");
			if (tokenizer.countTokens() == 10) {
				tokenizer.nextToken();
				double y2 = -Double.parseDouble(tokenizer.nextToken());
				double x1 = Double.parseDouble(tokenizer.nextToken());
				double y1 = -Double.parseDouble(tokenizer.nextToken());
				double x2 = Double.parseDouble(tokenizer.nextToken());
				setRectangle(new Rectangle2D.Double(x1, y1, x2 - x1, y2 - y1));
				break;
			}
		}
		in.close();
	}

	/**
	 * 道の弧データを読み込みます。
	 * @throws Exception 例外
	 */
	void loadRoadArc() throws Exception {
		String fileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "road" + Const.SEPARATOR + "roadntwk.arc";
		String attributeFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "road" + Const.SEPARATOR + "road.atr";
		this.roadArc = new HashMap<String, ArcData>();
			if (new File(fileName).canRead()) {
				loadArc(this.roadArc, new BufferedReader(new InputStreamReader(new FileInputStream(
						fileName))));
				if (new File(attributeFileName).canRead()) {
					loadArcAttribute(this.roadArc, new BufferedReader(new InputStreamReader(
							new FileInputStream(attributeFileName), "SJIS")));
				}
			}
	}

	/** 
	 * 市町界のポリゴンファイルを読み込みます。
	 * @throws Exception 例外
	 */
	void loadSi_tyo() throws Exception {
		String polygonFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "gyousei" + Const.SEPARATOR + "si_tyo.pgn";
		this.si_tyo = new HashMap<String, PolygonData>();
			if (new File(polygonFileName).canRead()) {
				loadPolygon(this.si_tyo, this.gyousei, new BufferedReader(new InputStreamReader(
						new FileInputStream(polygonFileName))));
				/*
				 String attributeFileName = baseDir + Const.SEPARATOR + mapName.toUpperCase() + Const.SEPARATOR + "gyousei" + Const.SEPARATOR + "si_tyo.atr";
				 if (new File(attributeFileName).canRead()) {
				 loadPolygonAttribute(
				 si_tyo,
				 new BufferedReader(new InputStreamReader(new FileInputStream(attributeFileName), "SJIS")),
				 Polygon.CLASSIFICATION_SI_TYO);
				 }
				 */
			}
	}

	/**
	 * 建物のポリゴンファイルを読み込みます。
	 * @throws Exception 例外
	 */
	void loadTatemono() throws Exception {
		String polygonFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "tatemono" + Const.SEPARATOR + "tatemono.pgn";
		String attributeFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "tatemono" + Const.SEPARATOR + "tatemono.atr";
		this.tatemono = new HashMap<String, PolygonData>();
			if (new File(polygonFileName).canRead()) {
				loadPolygon(this.tatemono, this.tatemonoArc, new BufferedReader(
						new InputStreamReader(new FileInputStream(polygonFileName))));
				if (new File(attributeFileName).canRead()) {
					loadPolygonAttribute(this.tatemono, new BufferedReader(new InputStreamReader(
							new FileInputStream(attributeFileName), "SJIS")),
							PolygonData.CLASSIFICATION_BUILDING);
				}
			}
	}

	/** 建物の弧ファイルを読み込みます。
	 * @throws Exception 例外
	 */
	void loadTatemonoArc() throws Exception {
		String fileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "tatemono" + Const.SEPARATOR + "tatemono.arc";
		this.tatemonoArc = new HashMap<String, ArcData>();
			if (new File(fileName).canRead()) {
				loadArc(this.tatemonoArc, new BufferedReader(new InputStreamReader(
						new FileInputStream(fileName))));
			}
	}

	/** 丁目界のポリゴンファイルを読み込みます。
	 * @throws Exception 例外
	 */
	void loadTyome() throws Exception {
		this.tyome = new HashMap<String, PolygonData>();
			loadPolygon(this.tyome, this.gyousei, new BufferedReader(new InputStreamReader(
					new FileInputStream(this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
							+ Const.SEPARATOR + "gyousei" + Const.SEPARATOR + "tyome.pgn"))));
			loadPolygonAttribute(this.tyome,
					new BufferedReader(new InputStreamReader(new FileInputStream(this.baseDir
							+ Const.SEPARATOR + this.mapName.toUpperCase() + Const.SEPARATOR
							+ "gyousei" + Const.SEPARATOR + "tyome.atr"), "SJIS")),
					PolygonData.CLASSIFICATION_TYOME);
	}

	/**
	 * 場地のポリゴンデータを読み込みます。
	 * @throws Exception 例外
	 */
	void loadZyouti() throws Exception {
		String polygonFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "others" + Const.SEPARATOR + "zyouti.pgn";
		String attributeFileName = this.baseDir + Const.SEPARATOR + this.mapName.toUpperCase()
				+ Const.SEPARATOR + "others" + Const.SEPARATOR + "zyouti.atr";
		this.zyouti = new HashMap<String, PolygonData>();
			if (new File(polygonFileName).canRead()) {
				loadPolygon(this.zyouti, this.others, new BufferedReader(new InputStreamReader(
						new FileInputStream(polygonFileName))));
				if (new File(attributeFileName).canRead()) {
					loadPolygonAttribute(this.zyouti, new BufferedReader(new InputStreamReader(
							new FileInputStream(attributeFileName), "SJIS")),
							PolygonData.CLASSIFICATION_PARK);
				}
			}
	}

	/**
	 * ポリゴンの図式分類コードを定数に変換します。
	 * @param code 図式分類コード
	 * @return 定数
	 */
	private int parsePolygonType(String code) {
		if (code.equals("A1105")) {
			return PolygonData.CLASSIFICATION_SI_TYO;
		} else if (code.equals("A1106")) {
			return PolygonData.CLASSIFICATION_TYOME;
		} else if (code.equals("A6241")) {
			return PolygonData.CLASSIFICATION_RAILROAD;
		} else if (code.equals("A6242")) {
			return PolygonData.CLASSIFICATION_PARK;
		} else if (code.equals("A6243")) {
			return PolygonData.CLASSIFICATION_SCHOOL;
		} else if (code.equals("A6244")) {
			return PolygonData.CLASSIFICATION_TEMPLE;
		} else if (code.equals("A6215")) {
			return PolygonData.CLASSIFICATION_GRAVEYARD;
		} else if (code.equals("A6200")) {
			return PolygonData.CLASSIFICATION_OTHER;
		} else if (code.equals("A5101")) {
			return PolygonData.CLASSIFICATION_RIVER;
		} else if (code.equals("A5105")) {
			return PolygonData.CLASSIFICATION_LAKE;
		} else if (code.equals("A3500")) {
			return PolygonData.CLASSIFICATION_BUILDING;
		} else {
			return PolygonData.CLASSIFICATION_UNKNOWN;
		}
	}

	/**
	 * @param string
	 */
	private void setBaseDir(String string) {
		this.baseDir = string;
	}

	/**
	 * @param string
	 */
	private void setMapName(String string) {
		this.mapName = string;
	}

	/**
	 * @param rectangle2D
	 */
	private void setRectangle(Rectangle2D rectangle2D) {
		this.rectangle = rectangle2D;
	}

	/**
	 * この地図の文字列表現を取得します。
	 * @return 文字列表現
	 */
	public String toString() {
		return "[MapData name=" + this.mapName + "]";
	}
}
