

import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;

/**
 * 国土数値情報の鉄道データを読み込むユーティリティクラスです。
 * @author Kumano Tatsuo
 * 2006/03/17
 */
public class KSJ {
	/**
	 * 国土数値情報の鉄道データが記録されたファイル名
	 */
	private final static String FILENAME = "N02-07L-2K.txt";

	/**
	 * 文字コードセット
	 */
	private final static String CHARSET = "SJIS";

	/**
	 * 駅の一覧を読み込みます。
	 * Celeron 2GHzで4.3秒くらいかかります。
	 * @return 駅
	 */
	public static Map<Point2D, String> loadStations() {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		final Map<String, Point2D> map = new LinkedHashMap<String, Point2D>();
		final Scanner scanner = new Scanner(KSJ.class.getResourceAsStream(KSJ.FILENAME),
			KSJ.CHARSET);
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			if (line.startsWith("N")) {
				final String hasDP = line.substring(31, 33).trim();
				if (hasDP.equals("1")) {
					final float x = Integer.parseInt(line.substring(15, 23).trim()) / 36000f;
					final float y = Integer.parseInt(line.substring(23, 31).trim()) / 36000f;
					final String dp = line.substring(33, 43).trim();
					map.put(dp, new Point2D.Float(x, y));
				}
			} else if (line.startsWith("DP")) {
				final String dp = line.substring(3, 13);
				final String name = line.substring(16).trim();
				if (map.containsKey(dp)) {
					ret.put(map.get(dp), name);
				}
			}
		}
		scanner.close();
		return ret;
	}
	/**
	 * 線路の一覧を読み込みます。
	 * @return 線路
	 */
	public static Collection<Line2D> loadRailways() {
		final Collection<Line2D> ret = new ArrayList<Line2D>();
		final Map<String, Point2D> nodes = new LinkedHashMap<String, Point2D>();
		final Scanner scanner = new Scanner(KSJ.class.getResourceAsStream(KSJ.FILENAME),
			KSJ.CHARSET);
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			if (line.startsWith("N")) {
				final String id = line.substring(3, 9).trim() + "_" + line.substring(9, 15).trim();
				final float x = Integer.parseInt(line.substring(15, 23).trim()) / 36000f;
				final float y = Integer.parseInt(line.substring(23, 31).trim()) / 36000f;
				nodes.put(id, new Point2D.Float(x, y));
			} else if (line.startsWith("L")) {
				final String mesh = line.substring(3, 9).trim();
				final String node1 = mesh + "_" + line.substring(9, 15).trim();
				final String node2 = line.substring(15, 21).trim() + "_"
					+ line.substring(21, 27).trim();
				final String id = mesh + "_" + line.substring(27, 33).trim();
				if (nodes.containsKey(node1) && nodes.containsKey(node2)) {
					final Point2D point1 = nodes.get(node1);
					final Point2D point2 = nodes.get(node2);
					ret.add(new Line2D.Float((float) point1.getX(), (float) point1.getY(),
						(float) point2.getX(), (float) point2.getY()));
				}
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * テスト用のメソッドです。
	 * @param args
	 */
	public static void main(String[] args) {
		System.out.println(KSJ.loadRailways());
	}
}
