import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.util.Collection;
import java.util.Map;
import javax.swing.JPanel;

/**
 * 地図を描画するパネルです。
 * @author Kumano Tatsuo
 * 2006/03/17
 */
public class MapPanel extends JPanel implements KeyListener {
	/**
	 * 1つのセルを表すクラスです。
	 * @author Kumano Tatsuo
	 * 2006/03/17
	 */
	private class Cell {
		/**
		 * 駅かどうか
		 */
		boolean isStation;

		/**
		 * 駅名
		 */
		String stationName;

		/**
		 * 右に線路が延びているかどうか
		 */
		boolean isRight;

		/**
		 * 左に線路が延びているかどうか
		 */
		boolean isLeft;

		/**
		 * 下に線路が延びているかどうか
		 */
		boolean isDown;

		/**
		 * 上に線路が延びているかどうか
		 */
		boolean isUp;
	}

	/**
	 * 駅の一覧
	 */
	private final Map<Point2D, String> stations;

	/**
	 * 線路の一覧
	 */
	private final Collection<Line2D> railways;

	/**
	 * 表示倍率
	 */
	private float zoom;

	/**
	 * 中央のx座標
	 */
	private float centerX;

	/**
	 * 中央のy座標
	 */
	private float centerY;

	/**
	 * コンストラクタです。
	 * @param ekis 駅
	 * @param railways 線路
	 */
	public MapPanel(final Map<Point2D, String> ekis, final Collection<Line2D> railways) {
		this.stations = ekis;
		this.railways = railways;
		this.zoom = 10000;
		this.centerX = 135;
		this.centerY = 35;
		this.addKeyListener(this);
	}

	public void keyPressed(KeyEvent e) {
		switch (e.getKeyCode()) {
		case KeyEvent.VK_RIGHT:
			this.centerX += Const.GRID_WIDTH / this.zoom;
			super.repaint();
			break;
		case KeyEvent.VK_LEFT:
			this.centerX -= Const.GRID_WIDTH / this.zoom;
			super.repaint();
			break;
		case KeyEvent.VK_DOWN:
			this.centerY -= Const.GRID_WIDTH / this.zoom;
			super.repaint();
			break;
		case KeyEvent.VK_UP:
			this.centerY += Const.GRID_WIDTH / this.zoom;
			super.repaint();
			break;
		case KeyEvent.VK_SEMICOLON:
		case KeyEvent.VK_ADD:
		case KeyEvent.VK_PAGE_DOWN:
			this.zoom *= 1.1f;
			super.repaint();
			break;
		case KeyEvent.VK_MINUS:
		case KeyEvent.VK_SUBTRACT:
		case KeyEvent.VK_PAGE_UP:
			this.zoom /= 1.1f;
			super.repaint();
			break;
		}
	}

	public void keyReleased(KeyEvent e) {
	}

	public void keyTyped(KeyEvent e) {
	}

	@Override
	protected void paintComponent(Graphics graphics) {
		super.paintComponent(graphics);
		final Graphics2D g = (Graphics2D) graphics;
		// セルを作ってみる。
		final Cell[][] cells = new Cell[super.getHeight() / Const.SHORT_EDGE + 1][super.getWidth()
			/ Const.SHORT_EDGE + 1];
		for (int i = 0; i < cells.length; i++) {
			for (int j = 0; j < cells[i].length; j++) {
				cells[i][j] = new Cell();
			}
		}
		final int centerColumn = toColumn(super.getWidth() / 2, super.getHeight() / 2);
		final int centerRow = toRow(super.getWidth() / 2, super.getHeight() / 2);
		// 駅の情報を量子化してみる。
		for (final Map.Entry<Point2D, String> entry : this.stations.entrySet()) {
			final Point2D point = entry.getKey();
			final int x = (int) ((float) (point.getX() - this.centerX) * this.zoom)
				+ centerColumn * Const.GRID_WIDTH;
			final int y = (int) ((float) (this.centerY - point.getY()) * this.zoom)
				+ centerRow * Const.GRID_WIDTH;
			final int j = x / Const.GRID_WIDTH;
			final int i = y / Const.GRID_WIDTH;
			if (i >= 0 && j >= 0 && i < cells.length && j < cells[0].length) {
				cells[i][j].isStation = true;
				cells[i][j].stationName = entry.getValue();
			}
		}
		// 線路の情報を量子化してみる。
		for (final Line2D line : this.railways) {
			final Point2D point1 = line.getP1();
			final Point2D point2 = line.getP2();
			final float x1 = (float) (point1.getX() - this.centerX) * this.zoom + centerColumn * Const.GRID_WIDTH;
			final int j1 = ((int) x1) / Const.GRID_WIDTH;
			final float y1 = (float) (this.centerY - point1.getY()) * this.zoom + centerRow * Const.GRID_WIDTH;
			final int i1 = ((int) y1) / Const.GRID_WIDTH;
			final float x2 = (float) (point2.getX() - this.centerX) * this.zoom + centerColumn * Const.GRID_WIDTH;
			final int j2 = ((int) x2) / Const.GRID_WIDTH;
			final float y2 = (float) (this.centerY - point2.getY()) * this.zoom + centerRow * Const.GRID_WIDTH;
			final int i2 = ((int) y2) / Const.GRID_WIDTH;
			final Line2D.Float segment = new Line2D.Float(x1, y1, x2, y2);
			for (int i = Math.max(Math.min(i1, i2), 0); i <= Math.min(Math.max(i1, i2),
				cells.length - 1); i++) {
				for (int j = Math.max(Math.min(j1, j2), 0); j <= Math.min(Math.max(j1, j2),
					cells[0].length - 1); j++) {
					// 水平方向
					if (Line2D.linesIntersect(segment.getX1(), segment.getY1(), segment.getX2(),
						segment.getY2(), j * Const.GRID_WIDTH, i * Const.GRID_WIDTH, j
							* Const.GRID_WIDTH, i * Const.GRID_WIDTH + Const.GRID_WIDTH)) {
						if (i < cells.length) {
							if (j > 0) {
								cells[i][j - 1].isRight = true;
							}
							if (j < cells[0].length) {
								cells[i][j].isLeft = true;
							}
						}
					}
					// 垂直方向
					if (Line2D.linesIntersect(segment.getX1(), segment.getY1(), segment.getX2(),
						segment.getY2(), j * Const.GRID_WIDTH, i * Const.GRID_WIDTH, j
							* Const.GRID_WIDTH + Const.GRID_WIDTH, i * Const.GRID_WIDTH)) {
						if (j < cells[0].length) {
							if (i > 0) {
								cells[i - 1][j].isDown = true;
							}
							if (i < cells.length) {
								cells[i][j].isUp = true;
							}
						}
					}
				}
			}
		}
		// ドット絵を描画してみる。
		for (int i = 0; i < cells.length; i++) {
			for (int j = 0; j < cells[i].length; j++) {
				if (cells[i][j].isStation) {
					g
						.drawImage(Const.B, toX(i, j) - Const.PIVOT_X, toY(i, j) - Const.PIVOT_Y,
							this);
				} else {
					g
						.drawImage(Const.A, toX(i, j) - Const.PIVOT_X, toY(i, j) - Const.PIVOT_Y,
							this);
				}
				if (cells[i][j].isRight) {
					g.drawImage(Const.RIGHT, toX(i, j) - Const.PIVOT_X, toY(i, j) - Const.PIVOT_Y,
						this);
				}
				if (cells[i][j].isLeft) {
					g.drawImage(Const.LEFT, toX(i, j) - Const.PIVOT_X, toY(i, j) - Const.PIVOT_Y,
						this);
				}
				if (cells[i][j].isDown) {
					g.drawImage(Const.DOWN, toX(i, j) - Const.PIVOT_X, toY(i, j) - Const.PIVOT_Y,
						this);
				}
				if (cells[i][j].isUp) {
					g.drawImage(Const.UP, toX(i, j) - Const.PIVOT_X, toY(i, j) - Const.PIVOT_Y,
						this);
				}
			}
		}
		for (int i = 0; i < cells.length; i++) {
			for (int j = 0; j < cells[i].length; j++) {
				if (cells[i][j].isStation) {
					g.drawString(cells[i][j].stationName, toX(i + 1, j), toY(i + 1, j));
				}
			}
		}
//		g.setColor(Color.GRAY);
		// 線路を描画してみる。
//		for (final Line2D line : this.railways) {
//			final Point2D point1 = line.getP1();
//			final Point2D point2 = line.getP2();
//			g.draw(new Line2D.Float((float) (point1.getX() - this.centerX) * this.zoom
//				+ super.getWidth() / 2, (float) (this.centerY - point1.getY()) * this.zoom
//				+ super.getHeight() / 2, (float) (point2.getX() - this.centerX) * this.zoom
//				+ super.getWidth() / 2, (float) (this.centerY - point2.getY()) * this.zoom
//				+ super.getHeight() / 2));
//		}
		// 駅名を描画してみる。
//		for (final Map.Entry<Point2D, String> entry : this.stations.entrySet()) {
//			final Point2D point = entry.getKey();
//			final String name = entry.getValue();
//			g.drawString(name, (float) (point.getX() - this.centerX) * this.zoom + super.getWidth()
//				/ 2, (float) (this.centerY - point.getY()) * this.zoom + super.getHeight() / 2);
//		}
	}

	/**
	 * セルが描画されるべきx座標を求めます。
	 * @param i 行
	 * @param j 列
	 * @return x座標
	 */
	private int toX(int i, int j) {
		return j * Const.LONG_EDGE - i * Const.MIDLE_EDGE;
	}

	/**
	 * セルが描画されるべきy座標を求めます。
	 * @param i 行
	 * @param j 列
	 * @return y座標
	 */
	private int toY(int i, int j) {
		return i * Const.MIDLE_EDGE + j * Const.SHORT_EDGE - super.getWidth() / 2;
	}

	/**
	 * 指定した座標が含まれるセルの列を求めます。
	 * toX、toYから逆算したものです。
	 * @param x x座標
	 * @param y y座標
	 * @return 列
	 */
	private int toColumn(int x, int y) {
		return (x + y + super.getWidth() / 2) / (Const.SHORT_EDGE + Const.LONG_EDGE);
	}

	/**
	 * 指定した座標が含まれるセルの行を求めます。
	 * toX、toYから逆算したものです。
	 * @param x x座標
	 * @param y y座標
	 * @return 行
	 */
	private int toRow(int x, int y) {
		return (y * Const.LONG_EDGE - x * Const.SHORT_EDGE + Const.LONG_EDGE * super.getWidth() / 2)
			/ (Const.MIDLE_EDGE * (Const.SHORT_EDGE + Const.LONG_EDGE));
	}
}
